package com.biz.crm.confadmin.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.confadmin.mapper.KmsDirectStoreAreaMapper;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.confadmin.mapper.KmsDirectSystemMapper;
import com.biz.crm.confadmin.model.KmsDirectStoreAreaEntity;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.confadmin.service.IKmsDirectStoreAreaService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.nebular.kms.confadmin.req.KmsDirectStoreAreaReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectStoreAreaRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * 商超门店区域管理接口实现
 *
 * @author Gavin
 * @date 2021-05-06 14:48:12
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsDirectStoreAreaServiceExpandImpl")
public class KmsDirectStoreAreaServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsDirectStoreAreaMapper, KmsDirectStoreAreaEntity> implements IKmsDirectStoreAreaService {

    @Resource
    private KmsDirectStoreAreaMapper kmsDirectStoreAreaMapper;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;


    @Resource
    private KmsDirectSystemMapper directSystemMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsDirectStoreAreaRespVo> findList(KmsDirectStoreAreaReqVo reqVo) {
        Page<KmsDirectStoreAreaRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsDirectStoreAreaRespVo> list = kmsDirectStoreAreaMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            Map<String, Map<String, String>> dataMap = this.getDictMaps();
            list.forEach(x -> {
                if (StringUtils.isNotBlank(x.getEnableStatus())) {
                    x.setEnableStatusName(Optional.ofNullable(dataMap.get(TpmGlobalDictConstants.ENABLE_STATUS))
                            .orElse(Maps.newHashMap()).get(x.getEnableStatus()));
                }
            });
        }
        return PageResult.<KmsDirectStoreAreaRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 转换数据  map
     *
     * @return
     */
    public Map<String, Map<String, String>> getDictMaps() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsDirectStoreAreaRespVo
     */
    @Override
    public KmsDirectStoreAreaRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsDirectStoreAreaReqVo reqVo = new KmsDirectStoreAreaReqVo();
        reqVo.setId(id);
        List<KmsDirectStoreAreaRespVo> respVos = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(respVos)) {
            throw new BusinessException("查询数据不存在");
        }
        return respVos.get(0);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsDirectStoreAreaReqVo reqVo) {
        this.checkAreaSaveOrUpdate(reqVo);
        KmsDirectStoreAreaEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectStoreAreaEntity.class);
        this.save(entity);
        KmsDirectStoreAreaRespVo newData = CrmBeanUtil.copy(entity, KmsDirectStoreAreaRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 新增或者修改区域检验
     *
     * @param reqVo
     */
    private void checkAreaSaveOrUpdate(KmsDirectStoreAreaReqVo reqVo) {
        this.checkDataNotEmpty(reqVo);
        AssertUtils.isNotNull(directSystemMapper.selectById(reqVo.getDirectId()), "商超体系不存在");
        this.checkDataExist(reqVo);
    }

    /**
     * 检验数据重复
     *
     * @param reqVo
     */
    private void checkDataExist(KmsDirectStoreAreaReqVo reqVo) {
        LambdaQueryWrapper<KmsDirectStoreAreaEntity> wrapper = Wrappers.<KmsDirectStoreAreaEntity>lambdaQuery();
        if (StringUtils.isNotBlank(reqVo.getId())) {
            wrapper.ne(KmsDirectStoreAreaEntity::getId, reqVo.getId());
        }
        wrapper.eq(KmsDirectStoreAreaEntity::getDsAreaCode, reqVo.getDsAreaCode());
        wrapper.eq(KmsDirectStoreAreaEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode());
        if (CollectionUtils.isNotEmpty(kmsDirectStoreAreaMapper.selectList(wrapper))) {
            throw new BusinessException("该区域已存在");
        }
    }

    /**
     * 检验区域新增数据填写
     *
     * @param reqVo
     */
    protected void checkDataNotEmpty(KmsDirectStoreAreaReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getDirectId(), "商超体系不能为空");
        AssertUtils.isNotEmpty(reqVo.getDsAreaCode(), "区域编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getDsAreaName(), "区域名称不能为空");
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsDirectStoreAreaReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        AssertUtils.isNotNull(kmsDirectStoreAreaMapper.selectById(reqVo.getId()), "操作数据不存在");
        this.checkAreaSaveOrUpdate(reqVo);
        KmsDirectStoreAreaEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectStoreAreaEntity.class);
        KmsDirectStoreAreaRespVo oldData = this.query(reqVo.getId());
        this.updateById(entity);
        KmsDirectStoreAreaRespVo newData = this.query(reqVo.getId());
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsDirectStoreAreaEntity> kmsDirectStoreAreaEntities = kmsDirectStoreAreaMapper.selectBatchIds(ids);
        Map<String, String> map = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(kmsDirectStoreAreaEntities)) {
            kmsDirectStoreAreaEntities.forEach(o -> {
                map.put(o.getId(), o.getDsAreaName());
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
            List<KmsDirectStoreEntity> storeEntities = kmsDirectStoreMapper
                    .selectList(Wrappers.<KmsDirectStoreEntity>lambdaQuery()
                            .eq(KmsDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                            .in(KmsDirectStoreEntity::getAreaId, map.keySet()));
            if (CollectionUtils.isNotEmpty(storeEntities)) {
                Set<String> names = Sets.newHashSet();
                storeEntities.forEach(x -> {
                    if (map.containsKey(x.getAreaId())) {
                        names.add(map.get(x.getAreaId()));
                    }
                });
                throw new BusinessException(names.toString() + "被商超门店使用，不可删除");
            }
        }
        this.updateBatchById(kmsDirectStoreAreaEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为启用
        List<KmsDirectStoreAreaEntity> kmsDirectStoreAreaEntities = kmsDirectStoreAreaMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectStoreAreaEntities)) {
            kmsDirectStoreAreaEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectStoreAreaEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为禁用
        List<KmsDirectStoreAreaEntity> kmsDirectStoreAreaEntities = kmsDirectStoreAreaMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectStoreAreaEntities)) {
            kmsDirectStoreAreaEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectStoreAreaEntities);
    }
}
