package com.biz.crm.confadmin.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.confadmin.mapper.KmsDirectStoreAreaMapper;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.confadmin.model.KmsDirectStoreAreaEntity;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.nebular.kms.confadmin.req.KmsDirectStoreReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectStoreRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author maoshen
 * @date 2021/5/6.
 */
@Component
@Slf4j
public class KmsDirectStoreHelper {

    @Resource
    private KmsDirectStoreAreaMapper directStoreAreaMapper;

    @Resource
    private KmsDirectStoreMapper directStoreMapper;

    /**
     * 转换数据  map
     *
     * @return
     */
    public Map<String, Map<String, String>> getDictMaps() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换查询列表数据
     *
     * @param list
     */
    public void convertStoreData(List<KmsDirectStoreRespVo> list) {
        Map<String, Map<String, String>> dataMap = this.getDictMaps();
        if (CollectionUtils.isNotEmpty(list)) {
            Set<String> areaIds = Sets.newHashSet();
            list.forEach(x -> {
                if (StringUtils.isNotBlank(x.getEnableStatus())) {
                    x.setEnableStatusName(Optional.ofNullable(dataMap.get(TpmGlobalDictConstants.ENABLE_STATUS))
                            .orElse(Maps.newHashMap()).get(x.getEnableStatus()));
                }
                if (StringUtils.isNotBlank(x.getAreaId())) {
                    areaIds.add(x.getAreaId());
                }
            });
            if (CollectionUtils.isEmpty(areaIds)) {
                return;
            }
            List<KmsDirectStoreAreaEntity> storeAreaEntityList = directStoreAreaMapper
                    .selectList(Wrappers.<KmsDirectStoreAreaEntity>lambdaQuery()
                            .in(KmsDirectStoreAreaEntity::getId, areaIds));
            if (CollectionUtils.isNotEmpty(storeAreaEntityList)) {
                Map<String, KmsDirectStoreAreaEntity> storeAreaEntityMap = storeAreaEntityList.stream()
                        .collect(Collectors.toMap(KmsDirectStoreAreaEntity::getId, Function.identity()));
                list.forEach(x -> {
                    if (storeAreaEntityMap.containsKey(x.getAreaId())) {
                        x.setAreaName(storeAreaEntityMap.get(x.getAreaId()).getDsAreaName());
                    }
                });
            }
        }
    }

    /**
     * 新增 编辑数据检验
     *
     * @param reqVo
     */
    public void saveStoreCheck(KmsDirectStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getStoreCode(), "门店编码[storeCode]不能为空");
        AssertUtils.isNotEmpty(reqVo.getStoreName(), "门店名称[storeName]不能为空");
        AssertUtils.isNotEmpty(reqVo.getDirectId(), "商超体系[directId]不能为空");
        if (StringUtils.isNotBlank(reqVo.getAreaId())) {
            // 检验所属大区是否在 维护范围内
            KmsDirectStoreAreaEntity directStoreAreaEntity = directStoreAreaMapper.selectById(reqVo.getAreaId());
            AssertUtils.isNotNull(directStoreAreaEntity, "该门店区域不存在");
            if (!StringUtils.equals(directStoreAreaEntity.getDirectId(), reqVo.getDirectId())) {
                throw new BusinessException("门店区域不在当前维护范围内");
            }
        }
        this.checkStoreExist(reqVo);
    }

    /**
     * 如果storeCode 为唯一索引，可以注释
     *
     * @param reqVo
     */
    private void checkStoreExist(KmsDirectStoreReqVo reqVo) {
        LambdaQueryWrapper<KmsDirectStoreEntity> wrapper = Wrappers.<KmsDirectStoreEntity>lambdaQuery();
        if (StringUtils.isNotBlank(reqVo.getId())) {
            wrapper.ne(KmsDirectStoreEntity::getId, reqVo.getId());
        }
        wrapper.eq(KmsDirectStoreEntity::getStoreCode, reqVo.getStoreCode());
        wrapper.eq(KmsDirectStoreEntity::getDirectId, reqVo.getDirectId());
        wrapper.eq(KmsDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode());
        if (StringUtils.isNotBlank(reqVo.getAreaId())) {
            wrapper.eq(KmsDirectStoreEntity::getAreaId, reqVo.getAreaId());
            if (CollectionUtils.isNotEmpty(directStoreMapper.selectList(wrapper))) {
                throw new BusinessException("该商超下门店已存在");
            }
            return;
        }
        if (CollectionUtils.isNotEmpty(directStoreMapper.selectList(wrapper))) {
            throw new BusinessException("该商超下门店已存在");
        }
    }

    /**
     * @param reqVo
     */
    public void checkUpdate(KmsDirectStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        AssertUtils.isNotNull(directStoreMapper.selectById(reqVo.getId()), "操作数据不存在");
    }
}
