package com.biz.crm.confadmin.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.confadmin.service.IKmsDirectStoreService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.nebular.kms.confadmin.req.KmsDirectStoreReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectStoreRespVo;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectStoreMapper;
import com.biz.crm.supermarket.model.KmsTenantryDirectStoreEntity;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 商超门店管理接口实现
 *
 * @author Gavin
 * @date 2021-05-06 14:03:40
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsDirectStoreServiceExpandImpl")
public class KmsDirectStoreServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsDirectStoreMapper, KmsDirectStoreEntity> implements IKmsDirectStoreService {

    @Autowired
    KmsDirectStoreHelper helper;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;

    @Resource
    private KmsTenantryDirectStoreMapper kmsTenantryDirectStoreMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsDirectStoreRespVo> findList(KmsDirectStoreReqVo reqVo) {
        Page<KmsDirectStoreRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsDirectStoreRespVo> list = kmsDirectStoreMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            helper.convertStoreData(list);
        }
        return PageResult.<KmsDirectStoreRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsDirectStoreRespVo
     */
    @Override
    public KmsDirectStoreRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsDirectStoreReqVo reqVo = new KmsDirectStoreReqVo();
        reqVo.setId(id);
        List<KmsDirectStoreRespVo> respVos = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(respVos)) {
            throw new BusinessException("查询数据不存在");
        }
        return respVos.get(0);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsDirectStoreReqVo reqVo) {
        helper.saveStoreCheck(reqVo);
        KmsDirectStoreEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectStoreEntity.class);
        this.save(entity);
        KmsDirectStoreRespVo newData = CrmBeanUtil.copy(entity, KmsDirectStoreRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsDirectStoreReqVo reqVo) {
        helper.checkUpdate(reqVo);
        helper.saveStoreCheck(reqVo);
        KmsDirectStoreEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectStoreEntity.class);
        KmsDirectStoreRespVo oldData = CrmBeanUtil.copy(entity, KmsDirectStoreRespVo.class);
        this.updateById(entity);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        KmsDirectStoreRespVo newData = this.query(reqVo.getId());
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsDirectStoreEntity> kmsDirectStoreEntities = kmsDirectStoreMapper.selectBatchIds(ids);
        Map<String, String> map = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(kmsDirectStoreEntities)) {
            kmsDirectStoreEntities.forEach(o -> {
                map.put(o.getId(), o.getStoreName());
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
            List<KmsTenantryDirectStoreEntity> entities = kmsTenantryDirectStoreMapper
                    .selectList(Wrappers.<KmsTenantryDirectStoreEntity>lambdaQuery()
                            .in(KmsTenantryDirectStoreEntity::getStoreId, map.keySet())
                            .eq(KmsTenantryDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
            if (CollectionUtils.isNotEmpty(entities)) {
                Set<String> names = Sets.newHashSet();
                entities.forEach(x -> {
                    if (map.containsKey(x.getStoreId())) {
                        names.add(map.get(x.getStoreId()));
                    }
                });
                throw new BusinessException(names.toString() + "的商超门店在直营体系下，不可删除");
            }
        }
        this.updateBatchById(kmsDirectStoreEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为启用
        List<KmsDirectStoreEntity> kmsDirectStoreEntities = kmsDirectStoreMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectStoreEntities)) {
            kmsDirectStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectStoreEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为禁用
        List<KmsDirectStoreEntity> kmsDirectStoreEntities = kmsDirectStoreMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectStoreEntities)) {
            kmsDirectStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectStoreEntities);
    }
}
