package com.biz.crm.finance.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.finance.helper.KmsAuditTemplateHelper;
import com.biz.crm.finance.mapper.KmsAuditTemplateDirectMapper;
import com.biz.crm.finance.mapper.KmsAuditTemplateMapper;
import com.biz.crm.finance.model.KmsAuditTemplateDirectEntity;
import com.biz.crm.finance.model.KmsAuditTemplateEntity;
import com.biz.crm.finance.service.IKmsAuditTemplateDirectService;
import com.biz.crm.finance.service.IKmsAuditTemplateService;
import com.biz.crm.nebular.kms.finance.req.KmsAuditTemplateDirectReqVo;
import com.biz.crm.nebular.kms.finance.req.KmsAuditTemplateReqVo;
import com.biz.crm.nebular.kms.finance.resp.KmsAuditTemplateDirectRespVo;
import com.biz.crm.nebular.kms.finance.resp.KmsAuditTemplateRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 稽核模板接口实现
 *
 * @author Gavin
 * @date 2021-05-25 17:56:09
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsAuditTemplateServiceExpandImpl")
public class KmsAuditTemplateServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsAuditTemplateMapper, KmsAuditTemplateEntity> implements IKmsAuditTemplateService {

    @Resource
    private KmsAuditTemplateMapper kmsAuditTemplateMapper;

    @Resource
    private KmsAuditTemplateHelper kmsAuditTemplateHelper;

    @Resource
    private IKmsAuditTemplateDirectService kmsAuditTemplateDirectService;

    @Resource
    private KmsAuditTemplateDirectMapper kmsAuditTemplateDirectMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsAuditTemplateRespVo> findList(KmsAuditTemplateReqVo reqVo) {
        Map<String, String> auditConditionMap = DictUtil.dictMap("kms_audit_condition");
        Map<String, String> orderTypeMap = DictUtil.dictMap("kms_order_type");

        Page<KmsAuditTemplateRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        List<KmsAuditTemplateRespVo> list = kmsAuditTemplateMapper.findList(page, reqVo);
        list.forEach(o -> {
            o.setConditionList(Arrays.asList(o.getConditionArray().split(",")));
            o.setOrderTypeList(Arrays.asList(o.getOrderTypeArray().split(",")));
            //中文描述转换
            o.setConditionArray(o.getConditionList().stream().map(c -> auditConditionMap.get(c))
                    .collect(Collectors.joining(",")));
            o.setOrderTypeArray(o.getOrderTypeList().stream().map(c -> orderTypeMap.get(c))
                    .collect(Collectors.joining(",")));
        });
        return PageResult.<KmsAuditTemplateRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsAuditTemplateRespVo
     */
    @Override
    public KmsAuditTemplateRespVo query(KmsAuditTemplateReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "主键ID不能为空");
        KmsAuditTemplateEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, "未查询到数据信息");
        KmsAuditTemplateRespVo respVo = CrmBeanUtil.copy(entity, KmsAuditTemplateRespVo.class);
        respVo.setConditionList(Arrays.asList(respVo.getConditionArray().split(",")));
        respVo.setOrderTypeList(Arrays.asList(respVo.getOrderTypeArray().split(",")));
        //查询商超数据
        List<String> auditTemplateIds = Lists.newArrayList();
        auditTemplateIds.add(respVo.getId());
        KmsAuditTemplateDirectReqVo directReqVo = new KmsAuditTemplateDirectReqVo();
        directReqVo.setAuditTemplateIds(auditTemplateIds);
        directReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        List<KmsAuditTemplateDirectRespVo> directRespVos = kmsAuditTemplateDirectMapper.findList(directReqVo);
        respVo.setDirectVos(directRespVos);
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsAuditTemplateReqVo reqVo) {
        kmsAuditTemplateHelper.check(reqVo);
        KmsAuditTemplateEntity entity = CrmBeanUtil.copy(reqVo, KmsAuditTemplateEntity.class);
        this.save(entity);
        reqVo.setId(entity.getId());
        List<KmsAuditTemplateDirectEntity> directEntities = kmsAuditTemplateHelper.buildAuditDirect(reqVo);
        kmsAuditTemplateDirectService.saveBatch(directEntities);
        KmsAuditTemplateRespVo newDate = CrmBeanUtil.copy(entity, KmsAuditTemplateRespVo.class);
        newDate.setDirectVos(CrmBeanUtil.copyList(directEntities, KmsAuditTemplateDirectRespVo.class));
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newDate.getId(), newDate.getId(), newDate);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsAuditTemplateReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "请选择数据");
        kmsAuditTemplateHelper.check(reqVo);
        KmsAuditTemplateEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, "未查询到数据信息");
        entity.setTemplateName(reqVo.getTemplateName());
        entity.setConditionArray(reqVo.getConditionArray());
        entity.setOrderTypeArray(reqVo.getOrderTypeArray());
        KmsAuditTemplateRespVo oldData = this.query(reqVo);
        this.updateById(entity);
        //删除原商超信息
        kmsAuditTemplateDirectService.lambdaUpdate().eq(KmsAuditTemplateDirectEntity::getAuditTemplateId, entity.getId()).remove();
        List<KmsAuditTemplateDirectEntity> directEntities = kmsAuditTemplateHelper.buildAuditDirect(reqVo);
        kmsAuditTemplateDirectService.saveBatch(directEntities);
        KmsAuditTemplateRespVo newData = this.query(reqVo);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), newData, oldData);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(KmsAuditTemplateReqVo reqVo) {
        List<KmsAuditTemplateEntity> kmsAuditTemplateEntities = kmsAuditTemplateMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsAuditTemplateEntities)) {
            kmsAuditTemplateEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsAuditTemplateEntities);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsAuditTemplateReqVo reqVo) {
        //设置状态为启用
        List<KmsAuditTemplateEntity> kmsAuditTemplateEntities = kmsAuditTemplateMapper.selectBatchIds(reqVo.getIds());
        List<KmsAuditTemplateRespVo> oldDataList = CrmBeanUtil.copyList(kmsAuditTemplateEntities, KmsAuditTemplateRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsAuditTemplateEntities)) {
            kmsAuditTemplateEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsAuditTemplateEntities);
        List<KmsAuditTemplateRespVo> newDataList = CrmBeanUtil.copyList(kmsAuditTemplateEntities, KmsAuditTemplateRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(i).getId(), newDataList.get(i).getId(), newDataList.get(i), oldDataList.get(i));
        }
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsAuditTemplateReqVo reqVo) {
        //设置状态为禁用
        List<KmsAuditTemplateEntity> kmsAuditTemplateEntities = kmsAuditTemplateMapper.selectBatchIds(reqVo.getIds());
        List<KmsAuditTemplateRespVo> oldDataList = CrmBeanUtil.copyList(kmsAuditTemplateEntities, KmsAuditTemplateRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsAuditTemplateEntities)) {
            kmsAuditTemplateEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsAuditTemplateEntities);
        List<KmsAuditTemplateRespVo> newDataList = CrmBeanUtil.copyList(kmsAuditTemplateEntities, KmsAuditTemplateRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(i).getId(), newDataList.get(i).getId(), newDataList.get(i), oldDataList.get(i));
        }
    }

    /**
     * 查询所有有效稽核模版
     *
     * @param templateReqVo
     * @return
     */
    @Override
    public List<KmsAuditTemplateRespVo> findAll(KmsAuditTemplateReqVo templateReqVo) {
        List<KmsAuditTemplateRespVo> all = kmsAuditTemplateMapper.findAll(templateReqVo);
        if (org.springframework.util.CollectionUtils.isEmpty(all)) {
            throw new BusinessException("未查询到模版信息!");
        }
        List<String> auditTemplateIds = all.stream().map(KmsAuditTemplateRespVo::getId).collect(Collectors.toList());
        //查询模版关联商超
        KmsAuditTemplateDirectReqVo directReqVo = new KmsAuditTemplateDirectReqVo();
        directReqVo.setAuditTemplateIds(auditTemplateIds);
        directReqVo.setBsDirectSystemId(templateReqVo.getBsDirectSystemId());
        directReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        directReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<KmsAuditTemplateDirectRespVo> directRespVos = kmsAuditTemplateDirectMapper.findList(directReqVo);
        //根据模版分组
        Map<String, List<KmsAuditTemplateDirectRespVo>> directRespVosMap = directRespVos.stream()
                .collect(Collectors.groupingBy(KmsAuditTemplateDirectRespVo::getAuditTemplateId));
        all.forEach(o -> {
            List<KmsAuditTemplateDirectRespVo> respVos = directRespVosMap.get(o.getId());
            if (org.springframework.util.CollectionUtils.isEmpty(respVos)) {
                if (!StringUtils.isEmpty(templateReqVo.getBsDirectSystemId())) {
                    throw new BusinessException("稽核模版未维护该商超");
                }
            }
            o.setDirectVos(respVos);
        });
        return all;
    }
}
