package com.biz.crm.returnform.helper;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.eunm.kms.KmsEnum;
import com.biz.crm.grabrule.mapper.KmsDataAuthMapper;
import com.biz.crm.nebular.kms.rawdata.resp.KmsReturnFormDetailDwRespVo;
import com.biz.crm.nebular.mdm.CrmBaseVo;
import com.biz.crm.rawdata.model.KmsReturnFormDetailDwEntity;
import com.biz.crm.rawdata.model.KmsReturnGoodsDetailDwEntity;
import com.biz.crm.rawdata.service.IKmsReturnFormDetailDwService;
import com.biz.crm.rawdata.service.IKmsReturnGoodsDetailDwService;
import com.biz.crm.returnform.mapper.KmsReturnFormMapper;
import com.biz.crm.returnform.mapper.KmsReturnGoodsMapper;
import com.biz.crm.returnform.model.KmsReturnFormEntity;
import com.biz.crm.returnform.model.KmsReturnGoodsEntity;
import com.biz.crm.trans.service.IKmsTransDataService;
import com.biz.crm.trans.vo.TransDataGoodsVo;
import com.biz.crm.trans.vo.TransDataStoreVo;
import com.biz.crm.trans.vo.TransDataVo;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.KmsUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.returnform
 * @ClassName KmsReturnFormHelper
 * @Author Gavin
 * @Date 2021/5/21 下午3:22
 * @Description 退货单助手
 */
@Slf4j
@Component
public class KmsReturnFormHelper {
    @Resource
    private KmsReturnFormMapper kmsReturnFormMapper;

    @Resource
    private KmsReturnGoodsMapper kmsReturnGoodsMapper;

    @Resource
    private IKmsReturnFormDetailDwService kmsReturnFormDetailDwService;

    @Resource
    private IKmsReturnGoodsDetailDwService kmsReturnGoodsDetailDwService;

    @Resource
    private IKmsTransDataService kmsTransDataService;

    @Resource
    private KmsDataAuthMapper kmsDataAuthMapper;

    /**
     * 处理订货单版本信息
     *
     * @param orderNumbers
     * @return
     */
    public KmsReturnFormEntity transFormVersion(KmsReturnFormDetailDwRespVo dwRespVo) {
        String orderNumber = dwRespVo.getOrderNumber();
        List<KmsReturnFormEntity> kmsReturnFormEntities = Lists.newArrayList();
        //根据订单号查询原始单据
        List<KmsReturnFormDetailDwEntity> returnDwEntities = kmsReturnFormDetailDwService.lambdaQuery()
                .eq(KmsReturnFormDetailDwEntity::getOrderNumber, orderNumber).list();
        //版本处理
        //根据版本号排序取最大
        KmsReturnFormDetailDwEntity dwEntity = returnDwEntities.stream()
                .sorted(Comparator.comparing(KmsReturnFormDetailDwEntity::getVersionNumber).reversed())
                .findFirst().orElse(null);
        //查询是否存在已转换数据
        LambdaQueryWrapper<KmsReturnFormEntity> build = new LambdaQueryWrapper<KmsReturnFormEntity>()
                .eq(KmsReturnFormEntity::getOrderNumber, orderNumber);
        KmsReturnFormEntity returnFormEntity = kmsReturnFormMapper.selectOne(build);
        if (ObjectUtils.isEmpty(returnFormEntity)) {
            returnFormEntity = CrmBeanUtil.copy(dwEntity, KmsReturnFormEntity.class);
            returnFormEntity.setBsDirectSystemId(dwRespVo.getBsDirectSystemId());
            returnFormEntity.setCreateCode(dwRespVo.getCreateCode());
            returnFormEntity.setCreateName(dwRespVo.getCreateName());
            returnFormEntity.setCreateOrgCode(dwRespVo.getCreateOrgCode());
            returnFormEntity.setCreateOrgName(dwRespVo.getCreateOrgName());
            returnFormEntity.setCreatePosCode(dwRespVo.getCreatePosCode());
            returnFormEntity.setCreatePosName(dwRespVo.getCreatePosName());
        } else {
            BeanUtils.copyProperties(dwEntity, returnFormEntity, KmsUtils.ignoreProperties());
            returnFormEntity.setRemarks(dwEntity.getRemarks());
        }
        //更改转换标示为已转换
        returnDwEntities.forEach(dw -> {
            dw.setTransStatus(YesNoEnum.yesNoEnum.ONE.getValue());
        });
        kmsReturnFormDetailDwService.updateBatchById(returnDwEntities);
        return returnFormEntity;
    }

    /**
     * 处理订货单商品版本数据
     *
     * @param orderFormEntities
     * @param orderNumbers
     */
    public List<KmsReturnGoodsEntity> transGoodsVersion(KmsReturnFormEntity returnFormEntity) {
        String orderNumber = returnFormEntity.getOrderNumber();
        Integer versionNumber = returnFormEntity.getVersionNumber();
        //删除原订单商品明细
        LambdaUpdateWrapper<KmsReturnGoodsEntity> wrapper = new LambdaUpdateWrapper<KmsReturnGoodsEntity>()
                .eq(KmsReturnGoodsEntity::getOrderNumber, orderNumber);
        List<KmsReturnGoodsEntity> deleteOrderGoodsEntities = kmsReturnGoodsMapper.selectList(wrapper);
        Map<String, KmsReturnGoodsEntity> goodsMap = deleteOrderGoodsEntities.stream()
                .collect(Collectors.toMap(o -> o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode(), Function.identity()));
        //订单号查询原始订货单商品详情信息
        List<KmsReturnGoodsDetailDwEntity> detailDwEntities = kmsReturnGoodsDetailDwService.lambdaQuery()
                .eq(KmsReturnGoodsDetailDwEntity::getOrderNumber, orderNumber)
                .eq(KmsReturnGoodsDetailDwEntity::getVersionNumber, versionNumber)
                .list();
        Map<String, KmsReturnGoodsDetailDwEntity> dwEntityMap = detailDwEntities.stream()
                .collect(Collectors.toMap(o -> o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode(), Function.identity(), (v1, v2) -> v1));

        //订单号+版本号查询原始订货单商品详情信息
        Map<String, KmsReturnGoodsEntity> onlyMap = Maps.newHashMap();
        detailDwEntities.forEach(o -> {
            String key = o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode();
            KmsReturnGoodsEntity goodsEntity = CrmBeanUtil.copy(o, KmsReturnGoodsEntity.class);
            goodsEntity.setReturnAmount(StringUtils.isEmpty(goodsEntity.getReturnAmount()) ? "0" : goodsEntity.getReturnAmount());
            goodsEntity.setReturnAmountNot(StringUtils.isEmpty(goodsEntity.getReturnAmountNot()) ? "0" : goodsEntity.getReturnAmountNot());
            goodsEntity.setCurUnitOrderQuantity(StringUtils.isEmpty(goodsEntity.getCurUnitOrderQuantity()) ? "0" : goodsEntity.getCurUnitOrderQuantity());
            goodsEntity.setRemarks(o.getRemarks());
            //判断是否重复
            if (!ObjectUtils.isEmpty(onlyMap.get(key))) {
                KmsReturnGoodsEntity en = onlyMap.get(key);
                en.setReturnAmount(new BigDecimal(en.getReturnAmount())
                        .add(new BigDecimal(goodsEntity.getReturnAmount())).toString());
                en.setReturnAmountNot(new BigDecimal(en.getReturnAmountNot())
                        .add(new BigDecimal(goodsEntity.getReturnAmountNot())).toString());
                en.setCurUnitOrderQuantity(new BigDecimal(en.getCurUnitOrderQuantity())
                        .add(new BigDecimal(goodsEntity.getCurUnitOrderQuantity())).toString());
                onlyMap.put(key, en);
            } else {
                KmsReturnGoodsEntity entity = ObjectUtils.isEmpty(goodsMap.get(key))
                        ? new KmsReturnGoodsEntity() : goodsMap.get(key);
                BeanUtils.copyProperties(goodsEntity, entity, KmsUtils.ignoreProperties());
                onlyMap.put(key, entity);
            }
        });
        //删除数据
        List<String> goodsIds = Lists.newArrayList();
        goodsMap.forEach((k, v) -> {
            if (ObjectUtils.isEmpty(dwEntityMap.get(k))) {
                goodsIds.add(v.getId());
            }
        });
        if (!CollectionUtils.isEmpty(goodsIds)) {
            kmsReturnGoodsMapper.deleteBatchIds(goodsIds);
        }
        List<KmsReturnGoodsEntity> onlyList = Lists.newArrayList();
        onlyMap.forEach((k, v) -> {
            onlyList.add(v);
        });
        return onlyList;
    }

    /**
     * 数据转换
     *
     * @param returnFormEntity
     * @param goodsEntities
     */
    public void transData(KmsReturnFormEntity returnFormEntity, List<KmsReturnGoodsEntity> goodsEntities) {
        //封装转换数据信息
        TransDataVo transDataVo = new TransDataVo();
        transDataVo.setOrderType(KmsAdminEnum.OrderType.RETURN.getValue());
        returnFormEntity.setOrderStatusMsg("");
        TransDataStoreVo dataStoreVo = CrmBeanUtil.copy(returnFormEntity, TransDataStoreVo.class);
        dataStoreVo.setSoldToPartyCode(null);
        dataStoreVo.setSoldToPartyName(null);
        dataStoreVo.setKaOrderNumber(returnFormEntity.getKaOrderNumber());
        dataStoreVo.setOrderStatusMsg("");
        dataStoreVo.setKaName(returnFormEntity.getKaName());
        transDataVo.setStoreVos(dataStoreVo);

        //单据商品Map
        Map<String, List<KmsReturnGoodsEntity>> goodsEntityMap = Maps.newHashMap();
        goodsEntities.forEach(o -> {
            List<KmsReturnGoodsEntity> entities = goodsEntityMap.get(o.getOrderNumber());
            if (CollectionUtils.isEmpty(entities)) {
                entities = Lists.newArrayList();
            }
            entities.add(o);
            goodsEntityMap.put(o.getOrderNumber(), entities);
            TransDataGoodsVo dataGoodsVo = CrmBeanUtil.copy(o, TransDataGoodsVo.class);
            dataGoodsVo.setKaOrderNumber(o.getKaOrderNumber());
            dataGoodsVo.setOrderStatusMsg("");
            dataGoodsVo.setKaName(o.getKaName());
            dataGoodsVo.setOrderDate(returnFormEntity.getOrderDate());
            transDataVo.getGoodsVos().add(dataGoodsVo);
        });
        //数据转换
        kmsTransDataService.transData(transDataVo);
        //结果数据处理
        resultProcessing(returnFormEntity, goodsEntities, transDataVo);
    }

    /**
     * 数据处理
     *
     * @param returnFormEntity
     * @param goodsEntities
     * @param transDataVo
     */
    private void resultProcessing(KmsReturnFormEntity returnFormEntity, List<KmsReturnGoodsEntity> goodsEntities, TransDataVo transDataVo) {
        String orderNumber = returnFormEntity.getOrderNumber();
        //转换结果信息
        TransDataStoreVo transDataStoreVo = transDataVo.getStoreVos();
        //门店处理
        BeanUtils.copyProperties(transDataStoreVo, returnFormEntity, Arrays.stream(CrmBaseVo.class.getDeclaredFields())
                .map(Field::getName)
                .collect(Collectors.toList()).toArray(new String[]{}));
        //商品处理
        Map<String, TransDataGoodsVo> dataGoodsVoMap = transDataVo.getGoodsVos().stream()
                .collect(Collectors.toMap(TransDataGoodsVo::getKaGoodsCode, Function.identity()));
        goodsEntities.forEach(goods -> {
            //商品行信息处理
            String kaGoodsCode = goods.getKaGoodsCode();
            TransDataGoodsVo transDataGoodsVo = dataGoodsVoMap.get(kaGoodsCode);
            BeanUtils.copyProperties(transDataGoodsVo, goods, Arrays.stream(CrmBaseVo.class.getDeclaredFields())
                    .map(Field::getName)
                    .collect(Collectors.toList()).toArray(new String[]{}));
            if (!StringUtils.isEmpty(goods.getOrderStatusMsg())) {
                goods.setOrderStatus(KmsEnum.OrderStatus.S100.getValue());
            } else {
                goods.setOrderStatus(KmsEnum.OrderStatus.S400.getValue());
                goods.setOrderStatusMsg("转换成功");
            }
        });
        //确认转换状态
        String orderStatusMsg = returnFormEntity.getOrderStatusMsg();
        List<KmsReturnGoodsEntity> errorList = goodsEntities.stream()
                .filter(goods -> KmsEnum.OrderStatus.S100.getValue().equals(goods.getOrderStatus())).collect(Collectors.toList());
        if (!StringUtils.isEmpty(orderStatusMsg) || !CollectionUtils.isEmpty(errorList)) {
            returnFormEntity.setOrderStatus(KmsEnum.OrderStatus.S100.getValue());
            if (!CollectionUtils.isEmpty(errorList) && !StringUtils.isEmpty(orderStatusMsg)) {
                returnFormEntity.setOrderStatusMsg(orderStatusMsg + "|商品行信息存在转换失败数据");
            } else if (!CollectionUtils.isEmpty(errorList) && StringUtils.isEmpty(orderStatusMsg)) {
                returnFormEntity.setOrderStatusMsg("商品行信息存在转换失败数据");
            }
        } else {
            returnFormEntity.setOrderStatus(KmsEnum.OrderStatus.S400.getValue());
        }
        if (StringUtils.isEmpty(returnFormEntity.getOrderStatusMsg())) {
            returnFormEntity.setOrderStatusMsg("转换成功");
        }
    }
}