package com.biz.crm.sap.helper;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.KmsSapConfig;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.kms.KmsEnum;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderFormRespVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderGoodsRespVo;
import com.biz.crm.nebular.kms.sap.req.KmsSapOrderFormReqVo;
import com.biz.crm.orderform.helper.KmsOrderFormHelper;
import com.biz.crm.orderform.model.KmsOrderFormEntity;
import com.biz.crm.orderform.model.KmsOrderGoodsEntity;
import com.biz.crm.sap.model.KmsSapInvoiceEntity;
import com.biz.crm.util.*;
import com.biz.crm.webservice.order.req.KmsOrderFormPushSapReqVo;
import com.biz.crm.webservice.order.resp.KmsOrderFormPushSapRespVo;
import com.biz.crm.webservice.order.resp.KmsSapRespMessageItem;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.sql.*;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.orderform.helper
 * @ClassName KmsOrderFormPushSapHelper
 * @Author Gavin
 * @Date 2021/5/28 下午4:46
 * @Description 单据推送Sap
 */
@Slf4j
@Component
public class KmsSapHelper {

    @Resource
    private KmsSapConfig kmsSapConfig;

    @Resource
    private KmsOrderFormHelper kmsOrderFormHelper;


    /**
     * 推送Sap
     *
     * @param orderFormEntity
     * @param goodsEntities
     */
    public void push(KmsOrderFormEntity orderFormEntity, List<KmsOrderGoodsEntity> goodsEntities, Integer timeOfDay) {
        try {
            //数据转换
            KmsOrderFormRespVo orderFormRespVo = CrmBeanUtil.copy(orderFormEntity, KmsOrderFormRespVo.class);
            List<KmsOrderGoodsRespVo> orderGoodsRespVos = CrmBeanUtil.copyList(goodsEntities, KmsOrderGoodsRespVo.class);
            List<KmsOrderGoodsRespVo> goodsSortedRespVos = orderGoodsRespVos.stream()
                    .sorted(Comparator.comparing(KmsOrderGoodsRespVo::getItemIndex)).collect(Collectors.toList());
            orderFormRespVo.setGoodsVos(goodsSortedRespVos);
            orderFormRespVo.setTimeOfDay(timeOfDay);
            KmsOrderFormPushSapReqVo compensationReqVo = KmsOrderFormPushSapReqVo.builder()
                    .kmsOrderFormRespVo(orderFormRespVo)
                    .build();
            Map<String, KmsOrderGoodsEntity> goodsEntityMap = goodsEntities.stream().filter(o -> !StringUtils.isEmpty(o.getGoodsCode()))
                    .collect(Collectors.toMap(o -> o.getGoodsCode(), Function.identity(), (v1, v2) -> {
                        throw new BusinessException("存在相同商品数据信息");
                    }));
            if (CollectionUtils.isEmpty(goodsEntityMap)) {
                throw new BusinessException("没有转换成功的商品信息");
            }
            //处理行项目号
            AtomicInteger i = new AtomicInteger(1);
            compensationReqVo.getBody().getKmsZsdSalesorderDataRfcVo().getItmeVo().getItmeDataVos().forEach(o -> {
                String itemNo = String.valueOf(i.get() * 10);
                o.setPosex(itemNo);
                i.getAndIncrement();
            });
            String body = XmlBeanUtil.beanToXml(compensationReqVo, StandardCharsets.UTF_8.name());
            Result<String> result = HttpClientUtils.doPostWebService(kmsSapConfig.getSendSapAddr(),
                    kmsSapConfig.getSendSapUserName(),
                    kmsSapConfig.getSendSapPassWord(),
                    body);
            if (ComConstant.SC_OK_200.equals(result.getCode())) {
                KmsOrderFormPushSapRespVo respVo = XmlBeanUtil.xmlToBean(result.getResult(), KmsOrderFormPushSapRespVo.class);
                String etVbeln = respVo.getBody().getKmsZsdSalesorderDataRfcRespVo().getEtVbeln();
                if (!StringUtils.isEmpty(etVbeln)) {
                    List<KmsSapRespMessageItem> item = respVo.getBody().getKmsZsdSalesorderDataRfcRespVo().getEtBapireturn1().getItem();
                    if (CollectionUtils.isEmpty(item)) {
                        //状态更改
                        orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S500.getValue());
                        orderFormEntity.setOrderStatusMsg(KmsEnum.OrderStatus.S500.getDescription());
                        orderFormEntity.setSapNumber(etVbeln);
                        orderFormEntity.setIsTrans(YesNoEnum.yesNoEnum.no.getValue());
                        compensationReqVo.getBody().getKmsZsdSalesorderDataRfcVo().getItmeVo().getItmeDataVos().forEach(o -> {
                            String key = o.getMatnr();
                            KmsOrderGoodsEntity goodsEntity = goodsEntityMap.get(key);
                            if (!ObjectUtils.isEmpty(goodsEntity)) {
                                goodsEntity.setSapItemNo(o.getPosex());
                            }
                        });
                    } else {
                        orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S500.getValue());
                        StringBuilder sb = new StringBuilder("");
                        item.forEach(o -> {
                            sb.append(o.getMessageMsgV4() + "|");
                        });
                        orderFormEntity.setOrderStatusMsg(sb.toString());
                    }
                } else {
                    List<KmsSapRespMessageItem> item = respVo.getBody().getKmsZsdSalesorderDataRfcRespVo().getEtBapireturn1().getItem();
                    StringBuilder sb = new StringBuilder("");
                    item.forEach(o -> {
                        sb.append(o.getMessageMsgV4() + "|");
                    });
                    orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S501.getValue());
                    orderFormEntity.setOrderStatusMsg(sb.toString());
                }
            } else {
                //状态更改
                orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S501.getValue());
                orderFormEntity.setOrderStatusMsg(result.getMessage());
            }
        } catch (BusinessException e) {
            log.info("单据推送SAP异常：{}", e.getMessage());
            orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S501.getValue());
            orderFormEntity.setOrderStatusMsg(e.getMessage());
        } catch (Exception e) {
            log.info("单据推送SAP异常：{}", KmsUtils.errorString(e.getStackTrace(), null));
            orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S501.getValue());
            orderFormEntity.setOrderStatusMsg("订单推送异常！");
        }
    }

    /**
     * 获取总条数
     *
     * @param reqVo
     * @return
     */
    public Integer getHanaInvoiceCount(KmsSapOrderFormReqVo reqVo) {
        Integer count = 0;
        Connection con = JDBCUtils.getConnection();
        Statement sta = null;
        ResultSet resultSet = null;
        try {
            //通过连接创建数据库执行对象
            sta = con.createStatement();
            String whereStr = buildWhereStr(reqVo);
            //生产-ECCPRD,测试-SAPERP
            String sql = "SELECT COUNT(*) FROM ECCPRD.ZFSSC_VBRK_VBRP_KMS t WHERE " + whereStr;
            log.info("SAP抓取SQL：{}", sql);
            resultSet = sta.executeQuery(sql);
            while (resultSet.next()) {
                count = resultSet.getInt(1);
            }
        } catch (SQLException e) {
            log.info("拉取SAP数据失败:{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } finally {
            JDBCUtils.closeResource(con, sta, resultSet);
        }
        return count;
    }

    protected String buildWhereStr(KmsSapOrderFormReqVo reqVo) {
        //发票类型
        String whereStr = "t.FKART IN (" + reqVo.getInvoiceTypeList().stream().collect(Collectors.joining(",")) + ")";
        //抓取公司
        if (!CollectionUtils.isEmpty(reqVo.getCompanyCodeList())) {
            whereStr += " AND t.BUKRS IN (" + reqVo.getCompanyCodeList().stream().collect(Collectors.joining(",")) + ")";
        } else if (!StringUtils.isEmpty(reqVo.getCompanyCode())) {
            whereStr += " AND t.BUKRS IN ('" + reqVo.getCompanyCode() + "')";
        }
        //reqVo.setStoreCode("300029213");
        //抓取门店
        if (!CollectionUtils.isEmpty(reqVo.getStoreCodeList())) {
            whereStr += " AND t.KUNNR IN (" + reqVo.getStoreCodeList().stream().collect(Collectors.joining(",")) + ")";
        }
        //抓取售达方
        if (!CollectionUtils.isEmpty(reqVo.getSoldToPartyCodeList())) {
            whereStr += " AND t.KUNAG IN (" + reqVo.getSoldToPartyCodeList().stream().collect(Collectors.joining(",")) + ")";
        }
        //抓取时间
        if (StringUtils.isEmpty(reqVo.getInvoiceCreateTime())) {
            //发票时间
            whereStr += " AND t.FKDAT = '" + reqVo.getStartTime().replaceAll("-", "") + "' AND t.FKIMG = t.FKIMGZP";
        } else {
            //创建时间
            whereStr += " AND t.ERZET LIKE '" + reqVo.getInvoiceCreateTime().replaceAll("-", "") + "%' AND t.FKIMG = t.FKIMGZP";
        }
        return whereStr;
    }

    /**
     * 抓取SAP发票数据
     *
     * @param reqVo
     * @return
     */
    public List<KmsSapInvoiceEntity> pullHanaInvoice(KmsSapOrderFormReqVo reqVo) {
        Connection con = JDBCUtils.getConnection();
        Statement sta = null;
        ResultSet resultSet = null;
        List<KmsSapInvoiceEntity> invoiceEntities = Lists.newArrayList();
        try {
            int offset = reqVo.getPageSize() * reqVo.getPageNum();
            //通过连接创建数据库执行对象
            sta = con.createStatement();
            String whereStr = buildWhereStr(reqVo);
            //生产-ECCPRD,测试-SAPERP
            String sql = "SELECT " + KmsConstant.SAP_COLUMNS + " FROM ECCPRD.ZFSSC_VBRK_VBRP_KMS t WHERE " + whereStr;
            //" ORDER BY t.ERZET ASC LIMIT " + reqVo.getPageSize() + " OFFSET " + offset;
            log.info("抓取SAP数据SQL：{}", sql);
            resultSet = sta.executeQuery(sql);
            ResultSetMetaData meta = resultSet.getMetaData();
            int cols = meta.getColumnCount();
            while (resultSet.next()) {
                Class<?> clazz = KmsSapInvoiceEntity.class;
                Object obj = KmsSapInvoiceEntity.class.newInstance();
                for (int i = 1; i <= cols; i++) {
                    Field field = null;
                    field = clazz.getDeclaredField(meta.getColumnName(i).toLowerCase());
                    field.setAccessible(true);
                    field.set(obj, resultSet.getObject(i));
                }
                KmsSapInvoiceEntity entity = (KmsSapInvoiceEntity) obj;
                entity.setId(entity.getVbeln() + entity.getPosnr());
                invoiceEntities.add(entity);
            }
        } catch (IllegalAccessException e) {
            log.info("拉取SAP数据失败:{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } catch (InstantiationException e) {
            log.info("拉取SAP数据失败:{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } catch (SQLException e) {
            log.info("拉取SAP数据失败:{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } catch (NoSuchFieldException e) {
            log.info("拉取SAP数据失败:{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } finally {
            JDBCUtils.closeResource(con, sta, resultSet);
        }
        return invoiceEntities;
    }
}