package com.biz.crm.supermarket.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.mdm.cusorg.MdmCusOrgFeign;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectCustomerOrgReqVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectCustomerOrgRespVo;
import com.biz.crm.nebular.mdm.cusorg.CusOrgVo;
import com.biz.crm.nebular.mdm.cusorg.MdmCustomerOrgSelectRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.supermarket.service.IKmsTenantryDirectCustomerOrgService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.cxf.interceptor.Fault;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.xml.soap.SOAPException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 客户组织与抓单商超（直营体系）关系表接口实现
 *
 * @author Gavin
 * @date 2021-04-13 17:05:17
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsTenantryDirectCustomerOrgServiceExpandImpl")
public class KmsTenantryDirectCustomerOrgServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsTenantryDirectCustomerOrgMapper, KmsTenantryDirectCustomerOrgEntity> implements IKmsTenantryDirectCustomerOrgService {

    @Resource
    private KmsTenantryDirectCustomerOrgMapper kmsTenantryDirectCustomerOrgMapper;

    @Resource
    private KmsTenantryDirectHelper helper;

    @Resource
    private MdmCusOrgFeign mdmCusOrgFeign;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsTenantryDirectCustomerOrgRespVo> findList(KmsTenantryDirectCustomerOrgReqVo reqVo) {
        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getCompanyOrgCode())){
            List<MdmOrgRespVo> childrenOrgListIncludeSelf = OrgUtil.getChildrenOrgListIncludeSelf(reqVo.getCompanyOrgCode());
            if (org.springframework.util.CollectionUtils.isEmpty(childrenOrgListIncludeSelf)) {
                throw new Fault(new SOAPException("公司代码错误，企业组织查询不到相关信息"));
            }
            List<String> orgCodes = childrenOrgListIncludeSelf.stream()
                    .map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            reqVo.setOrgCodes(orgCodes);
        }
        if (reqVo.getPageSize() == null) {
            reqVo.setPageSize(100);
        }
        Page<KmsTenantryDirectCustomerOrgRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsTenantryDirectCustomerOrgRespVo> list = kmsTenantryDirectCustomerOrgMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            helper.convertData(list);
        }
        return PageResult.<KmsTenantryDirectCustomerOrgRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsTenantryDirectCustomerOrgRespVo
     */
    @Override
    public KmsTenantryDirectCustomerOrgRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsTenantryDirectCustomerOrgReqVo reqVo = new KmsTenantryDirectCustomerOrgReqVo();
        reqVo.setId(id);
        List<KmsTenantryDirectCustomerOrgRespVo> list = this.findList(reqVo).getData();
        AssertUtils.isNotEmpty(list, "查询数据不存在");
        return list.get(0);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsTenantryDirectCustomerOrgReqVo reqVo) {
        List<KmsTenantryDirectCustomerOrgEntity> list = helper.checkSaveData(reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            KmsTenantryDirectCustomerOrgEntity updateEntity = list.get(0);
            KmsTenantryDirectCustomerOrgRespVo oldData = CrmBeanUtil.copy(list.get(0), KmsTenantryDirectCustomerOrgRespVo.class);
            updateEntity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            updateEntity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
            this.updateById(updateEntity);
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            KmsTenantryDirectCustomerOrgRespVo newData = this.query(updateEntity.getId());
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getCustomerOrgCode(), oldData, newData);
            return;
        }
        KmsTenantryDirectCustomerOrgEntity entity = CrmBeanUtil.copy(reqVo, KmsTenantryDirectCustomerOrgEntity.class);
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        this.save(entity);
        KmsTenantryDirectCustomerOrgRespVo newData = CrmBeanUtil.copy(entity, KmsTenantryDirectCustomerOrgRespVo.class);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getBsDirectSystemCode(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsTenantryDirectCustomerOrgReqVo reqVo) {
        this.checkUpdate(reqVo.getId());
        KmsTenantryDirectCustomerOrgEntity entity = this.getById(reqVo.getId());
        KmsTenantryDirectCustomerOrgRespVo oldData = CrmBeanUtil.copy(entity, KmsTenantryDirectCustomerOrgRespVo.class);
        BeanUtils.copyProperties(reqVo, entity);
        this.updateById(entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        KmsTenantryDirectCustomerOrgRespVo newData = this.query(entity.getId());
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getCustomerOrgCode(), oldData, newData);
    }

    /**
     * 修改检验
     *
     * @param id
     */
    private void checkUpdate(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsTenantryDirectCustomerOrgEntity entity = kmsTenantryDirectCustomerOrgMapper
                .selectOne(Wrappers.<KmsTenantryDirectCustomerOrgEntity>lambdaQuery()
                        .eq(KmsTenantryDirectCustomerOrgEntity::getId, id));
        AssertUtils.isNotNull(entity, "操作数据不存在");
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectCustomerOrgEntity> kmsTenantryDirectCustomerOrgEntities =
                kmsTenantryDirectCustomerOrgMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectCustomerOrgEntities)) {
            kmsTenantryDirectCustomerOrgEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
            helper.checkDelete(ids);
        }
        this.updateBatchById(kmsTenantryDirectCustomerOrgEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectCustomerOrgEntity> kmsTenantryDirectCustomerOrgEntities =
                kmsTenantryDirectCustomerOrgMapper.selectBatchIds(ids);
        List<KmsTenantryDirectCustomerOrgRespVo> oldDataList = CrmBeanUtil.copyList(kmsTenantryDirectCustomerOrgEntities, KmsTenantryDirectCustomerOrgRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectCustomerOrgEntities)) {
            kmsTenantryDirectCustomerOrgEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectCustomerOrgEntities);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<KmsTenantryDirectCustomerOrgRespVo> newDataList = CrmBeanUtil.copyList(kmsTenantryDirectCustomerOrgEntities, KmsTenantryDirectCustomerOrgRespVo.class);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newDataList.get(i).getId(),
                    newDataList.get(i).getBsDirectSystemCode(), newDataList.get(i), oldDataList.get(i));
        }
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectCustomerOrgEntity> kmsTenantryDirectCustomerOrgEntities =
                kmsTenantryDirectCustomerOrgMapper.selectBatchIds(ids);
        List<KmsTenantryDirectCustomerOrgRespVo> oldDataList = CrmBeanUtil.copyList(kmsTenantryDirectCustomerOrgEntities, KmsTenantryDirectCustomerOrgRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectCustomerOrgEntities)) {
            kmsTenantryDirectCustomerOrgEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectCustomerOrgEntities);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<KmsTenantryDirectCustomerOrgRespVo> newDataList = CrmBeanUtil.copyList(kmsTenantryDirectCustomerOrgEntities, KmsTenantryDirectCustomerOrgRespVo.class);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newDataList.get(i).getId(),
                    newDataList.get(i).getBsDirectSystemCode(), newDataList.get(i), oldDataList.get(i));
        }
    }

    /**
     * 新增直营体系 获取客户组织下拉框
     *
     * @return
     */
    @Override
    public List<MdmCustomerOrgSelectRespVo> findCusOrgList(CusOrgVo cusOrgVo) {
        Result<List<MdmCustomerOrgSelectRespVo>> cusOrgListResult;
        try {
            cusOrgListResult = mdmCusOrgFeign.customerOrgSelect(cusOrgVo);
        } catch (Exception e) {
            log.error("{}", e);
            throw new BusinessException(" 新增直营体系 获取客户组织列表,调用mdm接口失败");
        }
        List<MdmCustomerOrgSelectRespVo> respVoList = ApiResultUtil.objResult(cusOrgListResult, true);
        return respVoList;
    }
}
