package com.biz.crm.finance.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.KmsConfig;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.kms.KmsEnum;
import com.biz.crm.finance.service.IKmsAuditFormService;
import com.biz.crm.finance.service.IKmsAuditTemplateDirectService;
import com.biz.crm.finance.service.IKmsAuditTemplateService;
import com.biz.crm.nebular.kms.acceptanceform.req.KmsAcceptanceFormReqVo;
import com.biz.crm.nebular.kms.acceptanceform.resp.KmsAuditAcceptanceFormRespVo;
import com.biz.crm.nebular.kms.finance.req.KmsAuditFormReqVo;
import com.biz.crm.nebular.kms.finance.req.KmsAuditTemplateReqVo;
import com.biz.crm.nebular.kms.finance.resp.KmsAuditFormExportRespVo;
import com.biz.crm.nebular.kms.finance.resp.KmsAuditFormRespVo;
import com.biz.crm.nebular.kms.finance.resp.KmsAuditTemplateRespVo;
import com.biz.crm.nebular.kms.sap.req.KmsSapOrderFormReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.KmsOrgUtil;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 稽核数据
 *
 * @author Gavin
 * @date 2021-05-27 11:09:31
 */
@Slf4j
@RestController
@RequestMapping("/kmsauditform")
@Api(tags = "稽核数据")
public class KmsAuditFormController {
    @Autowired
    private IKmsAuditFormService kmsAuditFormService;

    @Resource
    private IKmsAuditTemplateService kmsAuditTemplateService;

    @Resource
    private IKmsAuditTemplateDirectService kmsAuditTemplateDirectService;

    @Resource
    private RedissonUtil redissonUtil;

    @Resource
    private KmsConfig kmsConfig;

    /**
     * 列表
     */
    @ApiOperation(value = "稽核匹配列表")
    @PostMapping("/matchList")
    @CrmLog
    public Result<PageResult<KmsAuditFormRespVo>> matchList(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.WAIT.getValue());
        PageResult<KmsAuditFormRespVo> result = kmsAuditFormService.findList(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 未匹配验收单查询稽核匹配列表
     */
    @ApiOperation(value = "未匹配验收单查询稽核匹配列表")
    @PostMapping("/selectMatchList")
    @CrmLog
    public Result<PageResult<KmsAuditFormRespVo>> selectMatchList(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.WAIT.getValue());
        PageResult<KmsAuditFormRespVo> result = kmsAuditFormService.selectMatchList(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 列表
     */
    @ApiOperation(value = "稽核核定列表")
    @PostMapping("/approvedList")
    @CrmLog
    public Result<PageResult<KmsAuditFormRespVo>> approvedList(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.CONFIRMED.getValue());
        PageResult<KmsAuditFormRespVo> result = kmsAuditFormService.findList(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 汇总
     */
    @ApiOperation(value = "稽核匹配汇总报表")
    @PostMapping("/matchSummaryReport")
    @CrmLog
    public Result<PageResult<KmsAuditFormRespVo>> matchSummaryReport(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        ConcurrentHashMap<String, Object> map = ThreadLocalUtil.get();
        map.put(GlobalParam.FUNCTION_CODE, "audit_summary_list");
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.WAIT.getValue());
        PageResult<KmsAuditFormRespVo> result = kmsAuditFormService.matchSummaryReport(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 汇总
     */
    @ApiOperation(value = "稽核匹配汇总")
    @PostMapping("/matchSummaryTotalCount")
    @CrmLog
    public Result<KmsAuditFormRespVo> matchSummaryTotalCount(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.WAIT.getValue());
        ConcurrentHashMap<String, Object> map = ThreadLocalUtil.get();
        map.put(GlobalParam.FUNCTION_CODE, "audit_summary_list");
        KmsAuditFormRespVo result = kmsAuditFormService.matchSummaryTotalCount(kmsAuditFormReqVo);
        return Result.ok(result);
    }


    /**
     * 稽核匹配导出
     */
    @ApiOperation(value = "稽核匹配导出")
    @PostMapping("/auditExport")
    @CrmLog
    public Result<PageResult<KmsAuditFormExportRespVo>> auditExport(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormReqVo.setAuditStatus(KmsEnum.AuditStatus.WAIT.getValue());
        PageResult<KmsAuditFormExportRespVo> result = kmsAuditFormService.auditExport(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 未匹配验收单列表
     */
    @ApiOperation(value = "未匹配验收单列表")
    @PostMapping("/notmatchedAcceptanceFormList")
    @CrmLog
    public Result<PageResult<KmsAuditAcceptanceFormRespVo>> notmatchedAcceptanceFormList(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        PageResult<KmsAuditAcceptanceFormRespVo> result = kmsAuditFormService.notmatchedAcceptanceFormList(kmsAcceptanceFormReqVo);
        return Result.ok(result);
    }

    /**
     * 查询/手动取消匹配详情
     */
    @ApiOperation(value = "查询/手动取消匹配详情")
    @PostMapping("/query")
    @CrmLog
    public Result<KmsAuditFormRespVo> query(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        KmsAuditFormRespVo respVo = kmsAuditFormService.query(kmsAuditFormReqVo);
        return Result.ok(respVo);
    }

    /**
     * 手动匹配查询已匹配验收单数据列表
     */
    @ApiOperation(value = "手动匹配查询已匹配验收单数据列表")
    @PostMapping("/matchedAcc")
    @CrmLog
    public Result<PageResult<KmsAuditAcceptanceFormRespVo>> matchedAcc(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        PageResult<KmsAuditAcceptanceFormRespVo> result = kmsAuditFormService.matchedAcc(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 拉取SAP数据
     */
    @ApiOperation(value = "拉取SAP数据（用于汇总稽核基础数据）")
    @PostMapping("/pullSap")
    @CrmLog
    public Result pullSap(@RequestBody KmsSapOrderFormReqVo kmsSapOrderFormReqVo) {
        //查询所有有效稽核模版
        KmsAuditTemplateReqVo templateReqVo = new KmsAuditTemplateReqVo();
        templateReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        templateReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        templateReqVo.setBsDirectSystemId(kmsSapOrderFormReqVo.getBsDirectSystemId());
        templateReqVo.setId(kmsSapOrderFormReqVo.getAuditTemplateId());
        List<KmsAuditTemplateRespVo> auditTemplateRespVos = kmsAuditTemplateService.findAll(templateReqVo);
        AssertUtils.isNotEmpty(auditTemplateRespVos, "未查询到模版信息");
        auditTemplateRespVos.forEach(o -> {
            MdmOrgRespVo companyOrg = KmsOrgUtil.getCompanyOrg(o.getCreateOrgCode());
            AssertUtils.isNotNull(companyOrg, "模版[" + o.getTemplateName() + "]没有所属公司信息!");
            o.setSapOrgCode(companyOrg.getSapOrgCode());
            o.setOrgCode(companyOrg.getOrgCode());
        });
        List<String> sapOrgCodes = auditTemplateRespVos.stream()
                .map(KmsAuditTemplateRespVo::getOrgCode).distinct().collect(Collectors.toList());
        if (sapOrgCodes.size() > 1) {
            throw new BusinessException("匹配模版所属组织机构不同");
        }
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.PULL_SAP_LOCK_KEY + "_" + sapOrgCodes.get(0));
            if (redissonUtil.tryLock(rLock, 0, 1800)) {
                kmsSapOrderFormReqVo.setAuditTemplateVo(auditTemplateRespVos);
                kmsAuditFormService.pullSap(kmsSapOrderFormReqVo);
                //汇总
                if (!CollectionUtils.isEmpty(kmsSapOrderFormReqVo.getInvoiceIds())) {
                    kmsAuditFormService.pullSapSummary(kmsSapOrderFormReqVo);
                }
            } else {
                throw new BusinessException("有人正在抓取SAP数据，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("操作失败:" + e.getMessage());
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok();
    }

    /**
     * 确认
     */
    @ApiOperation(value = "确认")
    @PostMapping("/confirm")
    @CrmLog
    public Result confirm(@RequestBody List<String> ids) {
        kmsAuditFormService.confirm(ids);
        return Result.ok();
    }

    /**
     * 取消确认
     */
    @ApiOperation(value = "取消确认")
    @PostMapping("/cancelConfirm")
    @CrmLog
    public Result cancelConfirm(@RequestBody List<String> ids) {
        kmsAuditFormService.cancelConfirm(ids);
        return Result.ok();
    }

    /**
     * 删除稽核数据
     */
    @ApiOperation(value = "删除稽核数据")
    @PostMapping("/deleteBatch")
    @CrmLog
    public Result deleteBatch(@RequestBody List<String> ids) {
        ids.forEach(i -> {
            KmsAuditFormReqVo kmsAuditFormReqVo = new KmsAuditFormReqVo();
            kmsAuditFormReqVo.setId(i);
            kmsAuditFormService.deleteBatch(kmsAuditFormReqVo);
        });
        return Result.ok();
    }

    /**
     * 添加备注
     */
    @ApiOperation(value = "添加备注")
    @PostMapping("/addRemark")
    @CrmLog
    public Result addRemark(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormService.addRemark(kmsAuditFormReqVo);
        return Result.ok();
    }

    /**
     * 手动取消匹配提交
     */
    @ApiOperation(value = "手动取消匹配提交")
    @PostMapping("/manualCancelmatch")
    @CrmLog
    public Result manualCancelmatch(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormService.manualCancelmatch(kmsAuditFormReqVo);
        return Result.ok();
    }

    /**
     * 手动匹配详情
     */
    @ApiOperation(value = "手动匹配详情")
    @PostMapping("/manualmatchQuery")
    @CrmLog
    public Result<KmsAuditFormRespVo> manualmatchQuery(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        KmsAuditFormRespVo respVo = kmsAuditFormService.manualmatchQuery(kmsAuditFormReqVo);
        return Result.ok(respVo);
    }

    /**
     * 手动匹配查询待匹配验收单数据列表
     */
    @ApiOperation(value = "手动匹配查询待匹配验收单数据列表")
    @PostMapping("/waitMatchedAcc")
    @CrmLog
    public Result<PageResult<KmsAuditAcceptanceFormRespVo>> waitMatchedAcc(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        PageResult<KmsAuditAcceptanceFormRespVo> result = kmsAuditFormService.waitMatchedAcc(kmsAuditFormReqVo);
        return Result.ok(result);
    }

    /**
     * 手动匹配提交
     */
    @ApiOperation(value = "手动匹配提交")
    @PostMapping("/manualmatch")
    @CrmLog
    public Result manualmatch(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormService.manualmatch(kmsAuditFormReqVo);
        return Result.ok();
    }

    /**
     * 手动批量匹配
     */
    @ApiOperation(value = "手动批量匹配")
    @PostMapping("/manualBstchMatch")
    @CrmLog
    public Result manualBstchMatch(@RequestBody List<String> ids) {
        AssertUtils.isNotEmpty(ids, "请选择数据");
        List<KmsAuditFormRespVo> auditFormRespVos = kmsAuditFormService.findListByIds(ids);
        KmsAuditFormRespVo auditFormRespVo = auditFormRespVos.get(0);
        String orgCode = auditFormRespVo.getOrgCode();
        //模版ID
        String auditTemplateId = auditFormRespVo.getAuditTemplateId();
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_BSTCH_MATCH_LOCK_KEY + "_" + orgCode);
            if (redissonUtil.tryLock(rLock, 0, 1800)) {
                KmsAuditTemplateReqVo templateReqVo = new KmsAuditTemplateReqVo();
                templateReqVo.setId(auditTemplateId);
                List<KmsAuditTemplateRespVo> auditTemplateRespVos = kmsAuditTemplateService.findAll(templateReqVo);
                KmsAuditTemplateRespVo auditTemplateRespVo = auditTemplateRespVos.get(0);
                auditFormRespVos.stream()
                        //.filter(o -> !KmsEnum.AuditResult.NO_DIFFERENCE.getValue().equals(o.getAuditResult()))
                        .forEach(o -> {
                            kmsAuditFormService.auditMatch(o, auditTemplateRespVo);
                            //处理延迟收货
                            o.setRemarks("延迟收货");
                            kmsAuditFormService.delayedReceipt(o, auditTemplateRespVo);
                        });
            } else {
                throw new BusinessException("有人正在处理数据，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("操作失败:" + e.getMessage());
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok();
    }

    /**
     * 未匹配验收单匹配稽核单提交
     */
    @ApiOperation(value = "未匹配验收单匹配稽核单提交")
    @PostMapping("/submitUnmatchedAcce")
    @CrmLog
    public Result submitUnmatchedAcce(@RequestBody KmsAuditFormReqVo kmsAuditFormReqVo) {
        kmsAuditFormService.submitUnmatchedAcce(kmsAuditFormReqVo);
        return Result.ok();
    }
}
