package com.biz.crm.returnform.service.impl;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.xml.soap.SOAPException;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.rawdata.mapper.KmsReturnFormDetailDwMapper;
import com.biz.crm.rawdata.model.KmsReturnFormDetailDwEntity;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.cxf.interceptor.Fault;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.kms.rawdata.resp.KmsReturnFormDetailDwRespVo;
import com.biz.crm.nebular.kms.returnform.req.KmsReturnFormReqVo;
import com.biz.crm.nebular.kms.returnform.resp.KmsReturnFormExportRespVo;
import com.biz.crm.nebular.kms.returnform.resp.KmsReturnFormRespVo;
import com.biz.crm.nebular.kms.returnform.resp.KmsReturnGoodsRespVo;
import com.biz.crm.returnform.helper.KmsReturnFormHelper;
import com.biz.crm.returnform.mapper.KmsReturnFormMapper;
import com.biz.crm.returnform.mapper.KmsReturnGoodsMapper;
import com.biz.crm.returnform.model.KmsReturnFormEntity;
import com.biz.crm.returnform.model.KmsReturnGoodsEntity;
import com.biz.crm.returnform.service.IKmsReturnFormService;
import com.biz.crm.returnform.service.IKmsReturnGoodsService;
import com.biz.crm.returnform.service.IKmsReturnStatusRecordService;

import lombok.extern.slf4j.Slf4j;

/**
 * 退货单信息表接口实现
 *
 * @author Gavin
 * @date 2021-05-12 18:23:07
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsReturnFormServiceExpandImpl")
public class KmsReturnFormServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsReturnFormMapper, KmsReturnFormEntity> implements IKmsReturnFormService {

    @Resource
    private KmsReturnFormMapper kmsReturnFormMapper;

    @Resource
    private KmsReturnGoodsMapper kmsReturnGoodsMapper;

    @Resource
    private KmsReturnFormHelper kmsReturnFormHelper;

    @Resource
    private IKmsReturnGoodsService kmsReturnGoodsService;

    @Resource
    private IKmsReturnStatusRecordService kmsReturnStatusRecordService;

    @Resource
    private KmsReturnFormDetailDwMapper returnFormDetailDwMapper;
    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsReturnFormRespVo> findList(KmsReturnFormReqVo reqVo) {
        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getCompanyOrgCode())){
            List<MdmOrgRespVo> childrenOrgListIncludeSelf = OrgUtil.getChildrenOrgListIncludeSelf(reqVo.getCompanyOrgCode());
            if (org.springframework.util.CollectionUtils.isEmpty(childrenOrgListIncludeSelf)) {
                throw new Fault(new SOAPException("公司代码错误，企业组织查询不到相关信息"));
            }
            List<String> orgCodes = childrenOrgListIncludeSelf.stream()
                    .map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            reqVo.setOrgCodes(orgCodes);
        }
        Page<KmsReturnFormRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsReturnFormRespVo> list = kmsReturnFormMapper.findList(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsReturnFormRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            list.forEach(o -> {
                //计算总金额
                if (StringUtils.isEmpty(o.getFinalReturnAmount())){
                    LambdaQueryWrapper<KmsReturnGoodsEntity> wrapper=new LambdaQueryWrapper<>();
                    wrapper.eq(KmsReturnGoodsEntity::getVersionNumber,o.getVersionNumber());
                    wrapper.eq(KmsReturnGoodsEntity::getOrderNumber,o.getOrderNumber());
                    List<KmsReturnGoodsEntity> kmsReturnGoodsEntities = kmsReturnGoodsMapper.selectList(wrapper);
                    if (CollectionUtil.listNotEmptyNotSizeZero(kmsReturnGoodsEntities)){
                        List<KmsReturnGoodsRespVo> kmsReturnGoodsRespVos = CrmBeanUtil.copyList(kmsReturnGoodsEntities, KmsReturnGoodsRespVo.class);
                        BigDecimal returnAmount=new BigDecimal(0);
                        for (KmsReturnGoodsRespVo goodsRespVo:kmsReturnGoodsRespVos){
                            if (StringUtils.isNotEmpty(goodsRespVo.getReturnAmount())){
                                returnAmount=returnAmount.add(new BigDecimal(goodsRespVo.getReturnAmount().replaceAll("-","")));
                            }
                        }
                        o.setFinalReturnAmount(returnAmount.toString());
                    }
                }
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
            });
        }
        return PageResult.<KmsReturnFormRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsReturnFormRespVo
     */
    @Override
    public KmsReturnFormRespVo query(KmsReturnFormReqVo reqVo) {
        String id = reqVo.getId();
        AssertUtils.isNotEmpty(id, "单据ID不能为空");
        KmsReturnFormEntity returnFormEntity = kmsReturnFormMapper.selectById(id);
        AssertUtils.isNotNull(returnFormEntity, "未查询到数据信息");
        KmsReturnFormRespVo formRespVo = CrmBeanUtil.copy(returnFormEntity, KmsReturnFormRespVo.class);
        String orderNumber = formRespVo.getOrderNumber();
        //查询商品行信息
        LambdaQueryWrapper<KmsReturnGoodsEntity> build = new LambdaQueryWrapper<KmsReturnGoodsEntity>()
                .eq(KmsReturnGoodsEntity::getOrderNumber, orderNumber);
        List<KmsReturnGoodsEntity> goodsEntities = kmsReturnGoodsMapper.selectList(build);
        goodsEntities.forEach(goodsEntity->{
            if (StringUtils.isNotEmpty(goodsEntity.getReturnAmount())){
                goodsEntity.setReturnAmount(goodsEntity.getReturnAmount().replaceAll("-",""));
            }
        });
        formRespVo.setReturnGoodsVos(CrmBeanUtil.copyList(goodsEntities, KmsReturnGoodsRespVo.class));
        return formRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsReturnFormReqVo reqVo) {
        KmsReturnFormEntity entity = CrmBeanUtil.copy(reqVo, KmsReturnFormEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsReturnFormReqVo reqVo) {
        KmsReturnFormEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        if (CollectionUtil.listEmpty(ids)){
            throw new BusinessException("请选择一条数据操作!");
        }
        List<KmsReturnFormEntity> kmsReturnFormEntities = kmsReturnFormMapper.selectBatchIds(ids);
        if (CollectionUtil.listEmpty(kmsReturnFormEntities)){
            throw new BusinessException("未找到删除数据!");
        }
        LambdaUpdateWrapper<KmsReturnFormEntity> wrapper=new LambdaUpdateWrapper<>();
        wrapper.in(KmsReturnFormEntity::getId,ids);
        kmsReturnFormMapper.delete(wrapper);
        List<String> orderList=Lists.newArrayList();
        List<KmsReturnFormRespVo> kmsReturnFormRespVos = CrmBeanUtil.copyList(kmsReturnFormEntities, KmsReturnFormRespVo.class);
        for (KmsReturnFormRespVo vo:kmsReturnFormRespVos){
            orderList.add(vo.getOrderNumber());
        }
        //删除dw表原始单据
        LambdaUpdateWrapper<KmsReturnFormDetailDwEntity> dwWrapper=new LambdaUpdateWrapper<>();
        dwWrapper.in(KmsReturnFormDetailDwEntity::getOrderNumber,orderList);
        returnFormDetailDwMapper.delete(dwWrapper);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsReturnFormReqVo reqVo) {
        //设置状态为启用
        List<KmsReturnFormEntity> kmsReturnFormEntities = kmsReturnFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsReturnFormEntities)) {
            kmsReturnFormEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsReturnFormEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsReturnFormReqVo reqVo) {
        //设置状态为禁用
        List<KmsReturnFormEntity> kmsReturnFormEntities = kmsReturnFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsReturnFormEntities)) {
            kmsReturnFormEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsReturnFormEntities);
    }

    /**
     * 手动转换退货单
     *
     * @param kmsReturnFormReqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void manualTrans(KmsReturnFormReqVo kmsReturnFormReqVo) {
        List<String> ids = kmsReturnFormReqVo.getIds();
        AssertUtils.isNotEmpty(ids, "请选择待转换数据");
        if (ids.size() > 100) {
            throw new BusinessException("最多只能处理100条数据！");
        }
        //查询订单状态，待确认、转换失败
        List<KmsReturnFormEntity> returnFormEntities = this.lambdaQuery()
                .in(KmsReturnFormEntity::getId, ids).list();
        List<KmsReturnFormDetailDwRespVo> dwRespVos = CrmBeanUtil.copyList(returnFormEntities, KmsReturnFormDetailDwRespVo.class);
        dwRespVos.forEach(dwRespVo -> {
            //匹配处理验收单版本信息
            KmsReturnFormEntity returnFormEntity = kmsReturnFormHelper.transFormVersion(dwRespVo);
            //匹配处理验收单商品详情新版本数据
            List<KmsReturnGoodsEntity> goodsEntities = kmsReturnFormHelper.transGoodsVersion(returnFormEntity);
            //数据转换
            kmsReturnFormHelper.transData(returnFormEntity, goodsEntities);
            //保存验收单头信息
            this.saveOrUpdate(returnFormEntity);
            //保存验收单商品行信息
            kmsReturnGoodsService.saveOrUpdateBatch(goodsEntities);
            //保存订单状态记录表
            kmsReturnStatusRecordService.insert(returnFormEntity);
        });
    }

    /**
     * 自动转换退货单
     *
     * @param orderNumber
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoTransData(KmsReturnFormDetailDwRespVo dwRespVo) {
        //匹配处理验收单版本信息
        KmsReturnFormEntity returnFormEntity = kmsReturnFormHelper.transFormVersion(dwRespVo);
        //匹配处理验收单商品详情新版本数据
        List<KmsReturnGoodsEntity> goodsEntities = kmsReturnFormHelper.transGoodsVersion(returnFormEntity);
        //数据转换
        kmsReturnFormHelper.transData(returnFormEntity, goodsEntities);
        //保存验收单头信息
        this.saveOrUpdate(returnFormEntity);
        //保存验收单商品行信息
        kmsReturnGoodsService.saveOrUpdateBatch(goodsEntities);
        //保存订单状态记录表
        kmsReturnStatusRecordService.insert(returnFormEntity);
    }

    /**
     * 导出 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsReturnFormExportRespVo> listForExport(KmsReturnFormReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getOrderDateStart(),"订单开始日期不能为空");
        AssertUtils.isNotEmpty(reqVo.getOrderDateEnd(),"订单结束日期不能为空");
        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getCompanyOrgCode())){
            List<MdmOrgRespVo> childrenOrgListIncludeSelf = OrgUtil.getChildrenOrgListIncludeSelf(reqVo.getCompanyOrgCode());
            if (org.springframework.util.CollectionUtils.isEmpty(childrenOrgListIncludeSelf)) {
                throw new Fault(new SOAPException("公司代码错误，企业组织查询不到相关信息"));
            }
            List<String> orgCodes = childrenOrgListIncludeSelf.stream()
                    .map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            reqVo.setOrgCodes(orgCodes);
        }
        Page<KmsReturnFormExportRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsReturnFormExportRespVo> list = kmsReturnFormMapper.findListForExport(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsReturnFormExportRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            list.forEach(o -> {
                //计算总金额
                if (StringUtils.isEmpty(o.getFinalReturnAmount())){
                    LambdaQueryWrapper<KmsReturnGoodsEntity> wrapper=new LambdaQueryWrapper<>();
                    wrapper.eq(KmsReturnGoodsEntity::getVersionNumber,o.getVersionNumber());
                    wrapper.eq(KmsReturnGoodsEntity::getOrderNumber,o.getOrderNumber());
                    List<KmsReturnGoodsEntity> kmsReturnGoodsEntities = kmsReturnGoodsMapper.selectList(wrapper);
                    if (CollectionUtil.listNotEmptyNotSizeZero(kmsReturnGoodsEntities)){
                        List<KmsReturnGoodsRespVo> kmsReturnGoodsRespVos = CrmBeanUtil.copyList(kmsReturnGoodsEntities, KmsReturnGoodsRespVo.class);
                        BigDecimal returnAmount=new BigDecimal(0);
                        for (KmsReturnGoodsRespVo goodsRespVo:kmsReturnGoodsRespVos){
                            if (StringUtils.isNotEmpty(goodsRespVo.getReturnAmount())){
                                returnAmount=returnAmount.add(new BigDecimal(goodsRespVo.getReturnAmount().replaceAll("-","")));
                            }
                        }
                        o.setFinalReturnAmount(returnAmount.toString());
                    }
                }
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
            });
        }
        return PageResult.<KmsReturnFormExportRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
}
