package com.biz.crm.acceptanceform.controller;

import com.biz.crm.acceptanceform.service.IKmsAcceptanceFormService;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.nebular.kms.acceptanceform.req.KmsAcceptanceFormReqVo;
import com.biz.crm.nebular.kms.acceptanceform.resp.KmsAcceptanceFormExportRespVo;
import com.biz.crm.nebular.kms.acceptanceform.resp.KmsAcceptanceFormRespVo;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.List;

/**
 * 转换后验收单
 *
 * @author Gavin
 * @date 2021-05-12 18:03:22
 */
@Slf4j
@RestController
@RequestMapping("/kmsacceptanceform")
@Api(tags = "商超单据-验收单")
public class KmsAcceptanceFormController {
    @Autowired
    private IKmsAcceptanceFormService kmsAcceptanceFormService;

    @Resource
    private RedissonUtil redissonUtil;

    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/list")
    @CrmLog
    public Result<PageResult<KmsAcceptanceFormRespVo>> list(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        PageResult<KmsAcceptanceFormRespVo> result = kmsAcceptanceFormService.findList(kmsAcceptanceFormReqVo);
        return Result.ok(result);
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @PostMapping("/query")
    @CrmLog
    public Result<KmsAcceptanceFormRespVo> query(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        KmsAcceptanceFormRespVo respVo = kmsAcceptanceFormService.query(kmsAcceptanceFormReqVo);
        return Result.ok(respVo);
    }

    /**
     * 新增
     */
    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        kmsAcceptanceFormService.save(kmsAcceptanceFormReqVo);
        return Result.ok();
    }

    /**
     * 更新
     */
    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        kmsAcceptanceFormService.update(kmsAcceptanceFormReqVo);
        return Result.ok("修改成功");
    }

    /**
     * 删除
     */
    @ApiOperation(value = "删除")
    @PostMapping("/delete")
    @CrmLog
    public Result delete(@RequestBody List<String> ids) {
        KmsAcceptanceFormReqVo reqVo = new KmsAcceptanceFormReqVo().setIds(ids);
        kmsAcceptanceFormService.deleteBatch(reqVo);
        return Result.ok("删除成功");
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        kmsAcceptanceFormService.enableBatch(kmsAcceptanceFormReqVo);
        return Result.ok("启用成功");
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        kmsAcceptanceFormService.disableBatch(kmsAcceptanceFormReqVo);
        return Result.ok("禁用成功");
    }

    /**
     * 手动转换验收单
     */
    @ApiOperation(value = "手动转换验收单")
    @PostMapping("/manualTrans")
    @CrmLog
    public Result manualTrans(@RequestBody List<String> ids) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_TRANS_FORM);
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo = new KmsAcceptanceFormReqVo();
                kmsAcceptanceFormReqVo.setIds(ids);
                kmsAcceptanceFormService.manualTrans(kmsAcceptanceFormReqVo);
            } else {
                throw new BusinessException("有人正在进行单据转换，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw e;
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }


    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/listForExport")
    @CrmLog
    public Result<PageResult<KmsAcceptanceFormExportRespVo>> listForExport(@RequestBody KmsAcceptanceFormReqVo kmsAcceptanceFormReqVo) {
        PageResult<KmsAcceptanceFormExportRespVo> result = kmsAcceptanceFormService.listForExport(kmsAcceptanceFormReqVo);
        return Result.ok(result);
    }
}
