package com.biz.crm.confadmin.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.KmsGlobalDictConstants;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.mapper.KmsDirectOrderTypeMapper;
import com.biz.crm.confadmin.mapper.KmsDirectThreadMapper;
import com.biz.crm.confadmin.model.KmsDirectOrderTypeEntity;
import com.biz.crm.confadmin.model.KmsDirectThreadEntity;
import com.biz.crm.confadmin.service.IKmsDirectThreadService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.nebular.kms.confadmin.req.KmsDirectThreadReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectThreadRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 商超单据抓取方式接口实现
 *
 * @author Gavin
 * @date 2021-05-06 15:12:00
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsDirectThreadServiceExpandImpl")
public class KmsDirectThreadServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsDirectThreadMapper, KmsDirectThreadEntity> implements IKmsDirectThreadService {

    @Resource
    private KmsDirectThreadMapper kmsDirectThreadMapper;

    @Resource
    private KmsDirectOrderTypeMapper kmsDirectOrderTypeMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 获取字典 map
     */
    public Map<String, Map<String, String>> getDictMaps() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(KmsGlobalDictConstants.ENABLE_STATUS);
        dictCodes.add(KmsGlobalDictConstants.KMS_FILE_CLASSIFY);
        dictCodes.add(KmsGlobalDictConstants.KMS_ORDER_TYPE);
        dictCodes.add(KmsGlobalDictConstants.KMS_ORDER_DOWNLOAD_WAY);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsDirectThreadRespVo> findList(KmsDirectThreadReqVo reqVo) {
        Page<KmsDirectThreadRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsDirectThreadRespVo> list = kmsDirectThreadMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            Map<String, Map<String, String>> kmsDataMap = this.getDictMaps();
            list.forEach(x -> {
                if (ObjectUtils.isEmpty(x.getFileClassify())) {
                    x.setFileClassifyDesc(Optional.ofNullable(kmsDataMap.get(KmsGlobalDictConstants.KMS_FILE_CLASSIFY))
                            .orElse(Maps.newHashMap()).get(x.getFileClassify().toString()));
                }
                if (StringUtils.isNotBlank(x.getOrderType())) {
                    x.setOrderTypeName(Optional.ofNullable(kmsDataMap.get(KmsGlobalDictConstants.KMS_ORDER_TYPE))
                            .orElse(Maps.newHashMap()).get(x.getOrderType()));
                }
                if (ObjectUtils.isEmpty(x.getFileDownloadWay())) {
                    x.setFileDownloadWayDesc(Optional.ofNullable(kmsDataMap.get(KmsGlobalDictConstants.KMS_ORDER_DOWNLOAD_WAY))
                            .orElse(Maps.newHashMap()).get(x.getFileDownloadWay().toString()));
                }
                if (ObjectUtils.isEmpty(x.getOrderDownloadWay())) {
                    x.setOrderDownloadWayDesc(Optional.ofNullable(kmsDataMap.get(KmsGlobalDictConstants.KMS_ORDER_DOWNLOAD_WAY))
                            .orElse(Maps.newHashMap()).get(x.getOrderDownloadWay().toString()));
                }
                if (StringUtils.isNotBlank(x.getEnableStatus())) {
                    x.setEnableStatusName(Optional.ofNullable(kmsDataMap.get(KmsGlobalDictConstants.ENABLE_STATUS))
                            .orElse(Maps.newHashMap()).get(x.getEnableStatus()));
                }
            });
        }
        return PageResult.<KmsDirectThreadRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsDirectThreadRespVo
     */
    @Override
    public KmsDirectThreadRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsDirectThreadReqVo reqVo = new KmsDirectThreadReqVo();
        reqVo.setId(id);
        List<KmsDirectThreadRespVo> respVos = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(respVos)) {
            throw new BusinessException("查询数据不存在");
        }
        return respVos.get(0);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsDirectThreadReqVo reqVo) {
        this.saveOrUpdateCheck(reqVo);
        KmsDirectThreadEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectThreadEntity.class);
        this.save(entity);
        KmsDirectThreadRespVo newData = CrmBeanUtil.copy(entity, KmsDirectThreadRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 新增或编辑检验
     *
     * @param reqVo
     */
    protected void saveOrUpdateCheck(KmsDirectThreadReqVo reqVo) {
        this.checkDataNotEmpty(reqVo);
        List<KmsDirectOrderTypeEntity> orderTypeEntities = kmsDirectOrderTypeMapper
                .selectList(Wrappers.<KmsDirectOrderTypeEntity>lambdaQuery()
                        .eq(KmsDirectOrderTypeEntity::getDirectId, reqVo.getDirectId()));
        if (CollectionUtils.isEmpty(orderTypeEntities)) {
            throw new BusinessException("所选商超系统没有支持的单据类型");
        }
        Set<String> orderTypeSet = orderTypeEntities.stream().map(KmsDirectOrderTypeEntity::getOrderType).collect(Collectors.toSet());
        if (!orderTypeSet.contains(reqVo.getOrderType())) {
            throw new BusinessException("该商超系统不支持该单据类型");
        }
        this.checkThreadDataExist(reqVo);
    }

    /**
     * 检验数据重复
     *
     * @param reqVo
     */
    private void checkThreadDataExist(KmsDirectThreadReqVo reqVo) {
        LambdaQueryWrapper<KmsDirectThreadEntity> wrapper = Wrappers.<KmsDirectThreadEntity>lambdaQuery();
        if (StringUtils.isNotBlank(reqVo.getId())) {
            wrapper.ne(KmsDirectThreadEntity::getId, reqVo.getId());
        }
        wrapper.eq(KmsDirectThreadEntity::getDirectId, reqVo.getDirectId());
        wrapper.eq(KmsDirectThreadEntity::getOrderType, reqVo.getOrderType());
        wrapper.eq(KmsDirectThreadEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode());
        List<KmsDirectThreadEntity> entities = kmsDirectThreadMapper.selectList(wrapper);
        if (CollectionUtils.isNotEmpty(entities)) {
            throw new BusinessException("该商超系统的该单据抓取方式已存在");
        }
    }

    /**
     * 检验非空数据必填
     *
     * @param reqVo
     */
    private void checkDataNotEmpty(KmsDirectThreadReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getDirectId(), "商超系统不能不空");
        AssertUtils.isNotEmpty(reqVo.getOrderType(), "单据类型不能为空");
        AssertUtils.isNotNull(reqVo.getOrderDownloadWay(), "单据下载方式不能为空");
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsDirectThreadReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        AssertUtils.isNotNull(kmsDirectThreadMapper.selectById(reqVo.getId()), "操作数据不存在");
        this.saveOrUpdateCheck(reqVo);
        KmsDirectThreadEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectThreadEntity.class);
        KmsDirectThreadRespVo oldData = this.query(reqVo.getId());
        this.updateById(entity);
        KmsDirectThreadRespVo newData = this.query(reqVo.getId());
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsDirectThreadEntity> kmsDirectThreadEntities = kmsDirectThreadMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectThreadEntities)) {
            kmsDirectThreadEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsDirectThreadEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为启用
        List<KmsDirectThreadEntity> kmsDirectThreadEntities = kmsDirectThreadMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectThreadEntities)) {
            kmsDirectThreadEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectThreadEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为禁用
        List<KmsDirectThreadEntity> kmsDirectThreadEntities = kmsDirectThreadMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsDirectThreadEntities)) {
            kmsDirectThreadEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectThreadEntities);
    }
}
