package com.biz.crm.kaproduct.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.kaproduct.mapper.KmsProductUnitMapper;
import com.biz.crm.kaproduct.mapper.KmsTenantryDirectProductMapper;
import com.biz.crm.kaproduct.model.KmsProductUnitEntity;
import com.biz.crm.kaproduct.model.KmsTenantryDirectProductEntity;
import com.biz.crm.kaproduct.service.IKmsProductUnitService;
import com.biz.crm.kaproduct.service.IKmsTenantryDirectProductService;
import com.biz.crm.nebular.kms.kaproduct.req.KmsProductUnitReqVo;
import com.biz.crm.nebular.kms.kaproduct.req.KmsTenantryDirectProductReqVo;
import com.biz.crm.nebular.kms.kaproduct.resp.KmsProductForExportRespVo;
import com.biz.crm.nebular.kms.kaproduct.resp.KmsProductUnitRespVo;
import com.biz.crm.nebular.kms.kaproduct.resp.KmsTenantryDirectProductRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 企业产品与商超产品关系接口实现
 *
 * @author Gavin
 * @date 2021-04-15 14:28:52
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsTenantryDirectProductServiceExpandImpl")
public class KmsTenantryDirectProductServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsTenantryDirectProductMapper, KmsTenantryDirectProductEntity> implements IKmsTenantryDirectProductService {

    @Resource
    private KmsTenantryDirectProductMapper kmsTenantryDirectProductMapper;

    @Resource
    private IKmsProductUnitService kmsProductUnitService;

    @Resource
    private KmsProductUnitMapper kmsProductUnitMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsTenantryDirectProductRespVo> findList(KmsTenantryDirectProductReqVo reqVo) {
        reqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        //reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Page<KmsTenantryDirectProductRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsTenantryDirectProductRespVo> list = kmsTenantryDirectProductMapper.findList(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsTenantryDirectProductRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            list.forEach(o -> {
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
            });
        }
        return PageResult.<KmsTenantryDirectProductRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsTenantryDirectProductRespVo
     */
    @Override
    public KmsTenantryDirectProductRespVo query(KmsTenantryDirectProductReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "数据ID不能为空");
        KmsTenantryDirectProductRespVo respVo = kmsTenantryDirectProductMapper.findById(reqVo.getId());
        AssertUtils.isNotNull(respVo, "未查询到数据信息");
        //查询单位关系
        List<KmsProductUnitEntity> productUnitEntities = kmsProductUnitService.lambdaQuery()
                .eq(KmsProductUnitEntity::getTenantryDirectProductId, respVo.getId())
                .orderByAsc(KmsProductUnitEntity::getId).list();
        respVo.setUnitReqVos(CrmBeanUtil.copyList(productUnitEntities, KmsProductUnitRespVo.class));
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsTenantryDirectProductReqVo reqVo) {
        this.check(reqVo);
        this.checkDataExist(reqVo);
        KmsTenantryDirectProductEntity entity = CrmBeanUtil.copy(reqVo, KmsTenantryDirectProductEntity.class);
        this.save(entity);
        //保存单位转换关系
        List<KmsProductUnitEntity> unitEntities = CrmBeanUtil.copyList(reqVo.getUnitReqVos(), KmsProductUnitEntity.class);
        unitEntities.forEach(o -> {
            o.setTenantryDirectProductId(entity.getId());
        });
        kmsProductUnitService.saveBatch(unitEntities);
        KmsTenantryDirectProductRespVo newData = CrmBeanUtil.copy(entity, KmsTenantryDirectProductRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        newData.setUnitReqVos(CrmBeanUtil.copyList(unitEntities, KmsProductUnitRespVo.class));
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 新增修改数据校验
     *
     * @param reqVo
     */
    private void checkDataExist(KmsTenantryDirectProductReqVo reqVo) {
        reqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        List<KmsTenantryDirectProductEntity> entities = kmsTenantryDirectProductMapper.selectListForSave(reqVo);
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        if (reqVo.getTimeOfWeek() == null) {
            throw new BusinessException("该商品已经存在");
        }
        if (reqVo.getTimeOfWeek() != null) {
            Set<Integer> integerSet = entities.stream().map(KmsTenantryDirectProductEntity::getTimeOfWeek)
                    .collect(Collectors.toSet());
            if (integerSet.contains(reqVo.getTimeOfWeek())) {
                throw new BusinessException("该星期数的商品已存在");
            }
        }
    }

    /**
     * 参数检查
     *
     * @param reqVo
     */
    protected void check(KmsTenantryDirectProductReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemId(), "直营体系Id不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemName(), "直营体系名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getProductCode(), "企业产品编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getProductName(), "企业产品名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getKaProductCode(), "商超产品编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getIsDefault(), "是否默认商品不能为空");
        AssertUtils.isNotEmpty(reqVo.getSellPartyCode(), "售达方编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getSellPartyName(), "售达方名称不能为空");
        String isDefault = reqVo.getIsDefault();
        if (StringUtils.isNotEmpty(isDefault)) {
            if (!YesNoEnum.yesNoEnum.YES.getValue().equals(isDefault) && !YesNoEnum.yesNoEnum.NO.getValue().equals(isDefault)) {
                throw new BusinessException("是否默认商品数据错误(Y/N)");
            }
        }
        String isDirect = reqVo.getIsDirect();
        if (StringUtils.isNotEmpty(isDirect)) {
            if (!YesNoEnum.yesNoEnum.YES.getValue().equals(isDirect) && !YesNoEnum.yesNoEnum.NO.getValue().equals(isDirect)) {
                throw new BusinessException("是否直营商品数据错误(Y/N)");
            }
        }
        List<KmsProductUnitReqVo> unitReqVos = reqVo.getUnitReqVos();
        AssertUtils.isNotEmpty(unitReqVos, "单位转换关系不能为空");
        unitReqVos.forEach(o -> {
            AssertUtils.isNotEmpty(o.getUnitCode(), "企业产品单位编码不能为空");
            AssertUtils.isNotEmpty(o.getUnitName(), "企业产品单位名称不能为空");
            AssertUtils.isNotEmpty(o.getKaUnitCode(), "商超产品单位名称不能为空");
            AssertUtils.isNotEmpty(o.getOrderType(), "单据类型不能为空");
            String orderTypeStr = KmsAdminEnum.OrderType.GETMAP.get(o.getOrderType());
            AssertUtils.isNotEmpty(orderTypeStr, "单据类型错误");
            AssertUtils.isNotNull(o.getRatio(), "单位转换关系商超单位数量[ratio]不能为空");
            AssertUtils.isNotNull(o.getNumerator(), "单位转换关系企业单位数量[numerator]不能为空");
        });
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsTenantryDirectProductReqVo reqVo) {
        this.check(reqVo);
        AssertUtils.isNotEmpty(reqVo.getId(), "更新数据ID不能为空");
        KmsTenantryDirectProductEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, "未查询到待更新数据信息");
        this.checkDataExist(reqVo);
        BeanUtils.copyProperties(reqVo, entity);
        KmsTenantryDirectProductRespVo oldData = this.query(reqVo);
        this.updateById(entity);
        //删除原单位转换关系
        kmsProductUnitService.lambdaUpdate().eq(KmsProductUnitEntity::getTenantryDirectProductId, entity.getId()).remove();
        //保存新的转换关系
        List<KmsProductUnitEntity> unitEntities = CrmBeanUtil.copyList(reqVo.getUnitReqVos(), KmsProductUnitEntity.class);
        unitEntities.forEach(o -> {
            o.setTenantryDirectProductId(entity.getId());
        });
        kmsProductUnitService.saveBatch(unitEntities);
        KmsTenantryDirectProductRespVo newData = this.query(reqVo);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(KmsTenantryDirectProductReqVo reqVo) {
        List<KmsTenantryDirectProductEntity> kmsTenantryDirectProductEntities = kmsTenantryDirectProductMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectProductEntities)) {
            kmsTenantryDirectProductEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectProductEntities);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsTenantryDirectProductReqVo reqVo) {
        //设置状态为启用
        List<KmsTenantryDirectProductEntity> kmsTenantryDirectProductEntities = kmsTenantryDirectProductMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectProductEntities)) {
            kmsTenantryDirectProductEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectProductEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsTenantryDirectProductReqVo reqVo) {
        //设置状态为禁用
        List<KmsTenantryDirectProductEntity> kmsTenantryDirectProductEntities = kmsTenantryDirectProductMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectProductEntities)) {
            kmsTenantryDirectProductEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectProductEntities);
    }


    /**
     * 列表导出
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsProductForExportRespVo> listForExport(KmsTenantryDirectProductReqVo reqVo) {
        reqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        Page<KmsProductForExportRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsProductForExportRespVo> list = kmsTenantryDirectProductMapper.findListForExport(page, reqVo);
        return PageResult.<KmsProductForExportRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
}
