package com.biz.crm.orderform.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.kms.acceptanceform.resp.KmsAcceptanceFormRespVo;
import com.biz.crm.nebular.kms.orderform.req.KmsOrderFormReqVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderFormExportRespVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderFormRespVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderGoodsRespVo;
import com.biz.crm.nebular.kms.rawdata.resp.KmsOrderFormDetailDwRespVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectCustomerOrgRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.orderform.helper.KmsOrderFormHelper;
import com.biz.crm.orderform.mapper.KmsOrderFormMapper;
import com.biz.crm.orderform.model.KmsOrderFormEntity;
import com.biz.crm.orderform.model.KmsOrderGoodsEntity;
import com.biz.crm.orderform.service.IKmsOrderFormService;
import com.biz.crm.orderform.service.IKmsOrderGoodsService;
import com.biz.crm.orderform.service.IKmsOrderStatusRecordService;
import com.biz.crm.sap.helper.KmsSapHelper;
import com.biz.crm.supermarket.service.IKmsTenantryDirectCustomerOrgService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.cxf.interceptor.Fault;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import javax.xml.soap.SOAPException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 商超订货单表接口实现
 *
 * @author Gavin
 * @date 2021-05-12 18:17:09
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsOrderFormServiceExpandImpl")
public class KmsOrderFormServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsOrderFormMapper, KmsOrderFormEntity> implements IKmsOrderFormService {

    @Resource
    private KmsOrderFormMapper kmsOrderFormMapper;

    @Resource
    private IKmsOrderGoodsService kmsOrderGoodsService;

    @Resource
    private KmsOrderFormHelper kmsOrderFormHelper;

    @Resource
    private IKmsOrderStatusRecordService kmsOrderStatusRecordService;

    @Resource
    private KmsSapHelper kmsSapHelper;

    @Resource
    private IKmsTenantryDirectCustomerOrgService kmsTenantryDirectCustomerOrgService;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsOrderFormRespVo> findList(KmsOrderFormReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getCompanyOrgCode())){
            List<MdmOrgRespVo> childrenOrgListIncludeSelf = OrgUtil.getChildrenOrgListIncludeSelf(reqVo.getCompanyOrgCode());
            if (org.springframework.util.CollectionUtils.isEmpty(childrenOrgListIncludeSelf)) {
                throw new Fault(new SOAPException("公司代码错误，企业组织查询不到相关信息"));
            }
            List<String> orgCodes = childrenOrgListIncludeSelf.stream()
                    .map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            reqVo.setOrgCodes(orgCodes);
        }
        Page<KmsOrderFormRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsOrderFormRespVo> list = kmsOrderFormMapper.findList(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsOrderFormRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            //区域名称
            List<String> ids = list.stream().map(KmsOrderFormRespVo::getStoreRelatedId).distinct().collect(Collectors.toList());
            List<KmsOrderFormRespVo> nameList = kmsOrderFormMapper.findAreaName(ids);
            Map<String, List<KmsOrderFormRespVo>> areaNameMap = nameList.stream()
                    .collect(Collectors.groupingBy(KmsOrderFormRespVo::getId));
            list.forEach(o -> {
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
                List<KmsOrderFormRespVo> vos = areaNameMap.get(o.getStoreRelatedId());
                if (CollectionUtil.listNotEmptyNotSizeZero(vos)){
                    o.setAreaName(vos.get(0).getAreaName());
                }
            });
        }
        return PageResult.<KmsOrderFormRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsOrderFormRespVo
     */
    @Override
    public KmsOrderFormRespVo query(KmsOrderFormReqVo reqVo) {
        String id = reqVo.getId();
        AssertUtils.isNotEmpty(id, "单据ID不能为空");
        KmsOrderFormEntity orderFormEntity = kmsOrderFormMapper.selectById(id);
        AssertUtils.isNotNull(orderFormEntity, "未查询到数据信息");
        KmsOrderFormRespVo formRespVo = CrmBeanUtil.copy(orderFormEntity, KmsOrderFormRespVo.class);
        String orderNumber = formRespVo.getOrderNumber();
        //查询商品行信息
        List<KmsOrderGoodsEntity> goodsEntities = kmsOrderGoodsService.lambdaQuery()
                .eq(KmsOrderGoodsEntity::getOrderNumber, orderNumber).list();
        List<KmsOrderGoodsEntity> goodsSortedEntities = goodsEntities.stream()
                .sorted(Comparator.comparing(KmsOrderGoodsEntity::getItemIndex)).collect(Collectors.toList());
        formRespVo.setGoodsVos(CrmBeanUtil.copyList(goodsSortedEntities, KmsOrderGoodsRespVo.class));
        return formRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsOrderFormReqVo reqVo) {
        KmsOrderFormEntity entity = CrmBeanUtil.copy(reqVo, KmsOrderFormEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsOrderFormReqVo reqVo) {
        KmsOrderFormEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        if (CollectionUtil.listEmpty(ids)){
            throw new BusinessException("请选择一条数据操作!");
        }
        List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsOrderFormEntities)) {
            kmsOrderFormEntities.forEach(o -> {
                //单据编码集合
                if (!StringUtils.isEmpty(o.getSapNumber())) {
                    throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                }
            });
        }
        LambdaUpdateWrapper<KmsOrderFormEntity> updateWrapper=new LambdaUpdateWrapper<>();
        updateWrapper.in(KmsOrderFormEntity::getId,ids);
        kmsOrderFormMapper.delete(updateWrapper);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsOrderFormReqVo reqVo) {
        //设置状态为启用
        List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsOrderFormEntities)) {
            kmsOrderFormEntities.forEach(o -> {
                //单据编码集合
                if (!StringUtils.isEmpty(o.getSapNumber())) {
                    throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                }
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsOrderFormEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsOrderFormReqVo reqVo) {
        //设置状态为禁用
        List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsOrderFormEntities)) {
            kmsOrderFormEntities.forEach(o -> {
                //单据编码集合
                if (!StringUtils.isEmpty(o.getSapNumber())) {
                    throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                }
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsOrderFormEntities);
    }

    /**
     * 加急订单
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void expedited(KmsOrderFormReqVo reqVo) {
        List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsOrderFormEntities)) {
            kmsOrderFormEntities.forEach(o -> {
                //单据编码集合
                if (!StringUtils.isEmpty(o.getSapNumber())) {
                    throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                }
                o.setIsExpedited("是");
            });
        }
        this.updateBatchById(kmsOrderFormEntities);
    }

    /**
     * 取消加急
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancelExpedited(KmsOrderFormReqVo reqVo) {
        List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsOrderFormEntities)) {
            kmsOrderFormEntities.forEach(o -> {
                //单据编码集合
                if (!StringUtils.isEmpty(o.getSapNumber())) {
                    throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                }
                o.setIsExpedited(null);
            });
        }
        this.updateBatchById(kmsOrderFormEntities);
    }

    /**
     * 手动转换单据
     *
     * @param kmsOrderFormReqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void manualTrans(KmsOrderFormReqVo kmsOrderFormReqVo) {
        List<String> ids = kmsOrderFormReqVo.getIds();
        AssertUtils.isNotEmpty(ids, "请选择待转换数据");
        if (ids.size() > 100) {
            throw new BusinessException("最多只能处理100条数据！");
        }
        //查询订单状态，待确认、转换失败
        List<KmsOrderFormEntity> orderFormEntities = this.lambdaQuery()
                .in(KmsOrderFormEntity::getId, ids).list();
        orderFormEntities.forEach(o -> {
            if (YesNoEnum.yesNoEnum.no.getValue().equals(o.getIsTrans())) {
                throw new BusinessException("[" + o.getOrderNumber() + "]不能再次转换!");
            }
        });
        List<KmsOrderFormDetailDwRespVo> dwRespVos = CrmBeanUtil.copyList(orderFormEntities, KmsOrderFormDetailDwRespVo.class);
        dwRespVos.forEach(o -> {
            //手动匹配处理订货单版本信息
            KmsOrderFormEntity orderFormEntity = kmsOrderFormHelper.transFormVersion(o);
            //手动匹配处理订货单商品详情新版本数据
            List<KmsOrderGoodsEntity> goodsEntities = kmsOrderFormHelper.transGoodsVersion(orderFormEntity);
            //数据转换
            kmsOrderFormHelper.transData(orderFormEntity, goodsEntities);
            //保存订货单头信息
            this.saveOrUpdate(orderFormEntity);
            //保证订货单商品行信息
            kmsOrderGoodsService.saveOrUpdateBatch(goodsEntities);
            //保存订单状态记录表
            kmsOrderStatusRecordService.insert(orderFormEntity);
        });
    }

    /**
     * 自动转换单据
     *
     * @param dwRespVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoTransData(KmsOrderFormDetailDwRespVo dwRespVo) {
        //手动匹配处理订货单版本信息
        KmsOrderFormEntity orderFormEntity = kmsOrderFormHelper.transFormVersion(dwRespVo);
        if (!ObjectUtils.isEmpty(orderFormEntity) && ObjectUtils.isEmpty(orderFormEntity.getSapNumber())) {
            //手动匹配处理订货单商品详情新版本数据
            List<KmsOrderGoodsEntity> goodsEntities = kmsOrderFormHelper.transGoodsVersion(orderFormEntity);
            //数据转换
            kmsOrderFormHelper.transData(orderFormEntity, goodsEntities);
            //保存订货单头信息
            this.saveOrUpdate(orderFormEntity);
            //保证订货单商品行信息
            kmsOrderGoodsService.saveOrUpdateBatch(goodsEntities);
            //保存订单状态记录表
            kmsOrderStatusRecordService.insert(orderFormEntity);
        }
    }

    /**
     * 单据推送查询单据信息
     *
     * @param params
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<KmsOrderFormRespVo> pushList(KmsOrderFormReqVo params) {
        AssertUtils.isNotEmpty(params.getIds(), "请选择需要推送的单据信息");
        if (params.getIds().size() > 5) {
            throw new BusinessException("最多只能处理5条数据！");
        }
        List<KmsOrderFormEntity> orderFormEntities = this.listByIds(params.getIds());
        orderFormEntities.forEach(o -> {
            if (!StringUtils.isEmpty(o.getSapNumber())) {
                throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
            }
        });
        //查询行信息
        List<KmsOrderGoodsEntity> goodsEntitiesAll = Lists.newArrayList();
        //数据转换
        orderFormEntities.forEach(o -> {
            //查询行信息
            List<KmsOrderGoodsEntity> goodsEntities = kmsOrderGoodsService.lambdaQuery()
                    .in(KmsOrderGoodsEntity::getOrderNumber, o.getOrderNumber()).list();
            //数据转换
            kmsOrderFormHelper.transData(o, goodsEntities);
            //保存订货单头信息
            this.saveOrUpdate(o);
            //保证订货单商品行信息
            kmsOrderGoodsService.saveOrUpdateBatch(goodsEntities);
            //保存订单状态记录表
            kmsOrderStatusRecordService.insert(o);
            goodsEntitiesAll.addAll(goodsEntities);
        });
        //返回数据处理
        List<KmsOrderFormRespVo> respVos = CrmBeanUtil.copyList(orderFormEntities, KmsOrderFormRespVo.class);
        Map<String, List<KmsOrderGoodsEntity>> goodsMap = goodsEntitiesAll.stream()
                .collect(Collectors.groupingBy(KmsOrderGoodsEntity::getOrderNumber));
        respVos.forEach(o -> {
            o.setGoodsVos(CrmBeanUtil.copyList(goodsMap.get(o.getOrderNumber()), KmsOrderGoodsRespVo.class));
        });
        return respVos;
    }

    /**
     * 单据推送
     *
     * @param params
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void push(KmsOrderFormReqVo params) {
        KmsOrderFormEntity orderFormEntity = this.getById(params.getId());
        //查询直营体系获取时间阈值
        KmsTenantryDirectCustomerOrgRespVo customerOrgRespVo = kmsTenantryDirectCustomerOrgService.query(orderFormEntity.getBsDirectSystemId());
        //查询商品行信息
        List<KmsOrderGoodsEntity> goodsEntities = kmsOrderGoodsService.lambdaQuery()
                .eq(KmsOrderGoodsEntity::getOrderNumber, orderFormEntity.getOrderNumber()).list();
        kmsSapHelper.push(orderFormEntity, goodsEntities, customerOrgRespVo.getTimeOfDay());
        this.saveOrUpdate(orderFormEntity);
        kmsOrderGoodsService.saveOrUpdateBatch(goodsEntities);
        kmsOrderStatusRecordService.insert(orderFormEntity);
    }


    @Override
    public PageResult<KmsOrderFormExportRespVo> listFprExport(KmsOrderFormReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getOrderDateStart(),"订单开始日期不能为空");
        AssertUtils.isNotEmpty(reqVo.getOrderDateEnd(),"订单结束日期不能为空");
        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getCompanyOrgCode())){
            List<MdmOrgRespVo> childrenOrgListIncludeSelf = OrgUtil.getChildrenOrgListIncludeSelf(reqVo.getCompanyOrgCode());
            if (org.springframework.util.CollectionUtils.isEmpty(childrenOrgListIncludeSelf)) {
                throw new Fault(new SOAPException("公司代码错误，企业组织查询不到相关信息"));
            }
            List<String> orgCodes = childrenOrgListIncludeSelf.stream()
                    .map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            reqVo.setOrgCodes(orgCodes);
        }
        Page<KmsOrderFormExportRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsOrderFormExportRespVo> list = kmsOrderFormMapper.findListForExport(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsOrderFormExportRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            list.forEach(o -> {
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
            });
        }
        return PageResult.<KmsOrderFormExportRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
}