package com.biz.crm.log.template.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.log.template.entity.LogFieldEntity;
import com.biz.crm.log.template.entity.LogTemplateEntity;
import com.biz.crm.log.template.repositories.LogTemplateRepositories;
import com.biz.crm.log.template.service.LogTemplateService;
import com.biz.crm.log.utils.CrmLogContants;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.swagger.annotations.ApiModelProperty;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchTemplate;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;

/**
 * 日志规则逻辑处理实现类
 * @Author: chenrong
 * @Date: 2020/12/9 15:13
 */
@ConditionalOnMissingBean(name = "logTemplateServiceImpl")
@Service(value = "logTemplateService")
public class LogTemplateServiceImpl implements LogTemplateService {

  private Logger LOGGER = LoggerFactory.getLogger(LogTemplateServiceImpl.class);
  private static final String LOG_TEMPLATE_REDIS_PATH = "dms:log";

  @Resource
  private RedissonClient redissonClient;
  @Resource
  private ElasticsearchTemplate elasticsearchTemplate;
  @Resource
  private RedissonUtil redissonUtil;
  @Resource
  private LogTemplateRepositories logTemplateRepositories;

  @Override
  public Map<String, LogFieldEntity> findFieldsByType(String type) {
    Map<String, LogFieldEntity> fieldVos = Maps.newTreeMap();
    try {
      Class cls = Class.forName(type);
      List<Field> fields = this.getFields(cls);
      for (Field field : fields) {
        LogFieldEntity filedVo = this.buildFieldVo(field);
        fieldVos.put(filedVo.getFieldName(), filedVo);
      }
    } catch (ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      throw new BusinessException("获取字段列表失败，请确认传入类型是否正确");
    }
    return fieldVos;
  }

  @Override
  public Map<String, LogFieldEntity> findFieldsByTypeAndCode(String type, String templateCode, String tree) {
    if (StringUtils.isEmpty(type) || StringUtils.isEmpty(templateCode)) {
      return Maps.newHashMap();
    }
    Map<String, LogFieldEntity> fieldVos = this.findFieldsByType(type);
    LogTemplateEntity logTemplateVo = this.logTemplateRepositories.findByCode(templateCode);
    if (logTemplateVo == null || logTemplateVo.getFieldMap() == null) {
      return Maps.newHashMap();
    }
    // 根据模板上的已后字段，设置是否比较
    Map<String, LogFieldEntity> oldFields = logTemplateVo.getFieldMap();
    Map<String, LogFieldEntity> map = Maps.newTreeMap();
    fieldVos.forEach((k, v) -> {
//      v.setCompare(this.hasField(v, oldFields, tree));
      map.put(v.getFieldName(), v);
    });
    return map;
  }

  /**
   * 判断集合中是否包含某个字段
   * @param vo
   * @param fields
   * @param tree
   * @return
   */
  private boolean hasField(LogFieldEntity vo, Map<String, LogFieldEntity> fields, String tree) {
    if (vo == null || !CollectionUtil.mapNotEmpty(fields)) {
      return false;
    }
    if (StringUtils.isEmpty(tree)) {
      if (fields.get(vo.getFieldName()) != null) {
        return true;
      }
      return false;
    }
    String[] trees = tree.split("\\.+");
    if (trees.length < 1) {
      return false;
    }
    String nTree = "";
    for (int i = 1; i < trees.length; i++) {
      nTree = nTree.concat(trees[i]);
      if (i < trees.length - 1) {
        nTree = nTree.concat(".");
      }
    }
    return this.hasField(vo, fields.get(trees[0]) == null ? null : fields.get(trees[0]).getChildren(), nTree);
  }

  /**
   * 1、校验参数非空，模板不存在
   * 2、处理字段属性
   * 3、保存到es
   * 4、保存到redis
   * @param vo
   */
  @Override
  public void create(LogTemplateEntity vo) {
//    //1、
//    ValidateUtils.validate(vo, "新增时，参数不能为空");
//    vo.setCode(vo.getType());
//    LogTemplateEntity logTemplateEntity = this.logTemplateRepositories.findByCode(vo.getCode());
//    ValidateUtils.isTrue(logTemplateEntity == null, "模板【%s】已存在，不能新增", vo.getCode());
//    vo.setFieldMap(this.transferFieldMap(vo.getFieldList()));
//    vo.setFieldList(null);
//    this.validateOnlyKey(vo.getFieldMap());
//    //2、
//    vo.setId(CodeUtil.getCode());
//    vo.setEnableState(true);
//    vo.setCreateTime(new Date());
//    UserRedis user = UserUtils.getUser();
//    if (user != null) {
//      vo.setCreateAccount(user.getUsername());
//    }
//    Map<String, LogFieldEntity> fields = vo.getFieldMap();
//    ValidateUtils.notEmpty(fields, "保存日志模板时，字段列表必须传入");
//    this.removeUnCompareField(fields);
//    //3、
//    EsUtil.indexExsit(elasticsearchTemplate, LogTemplateEntity.class, redissonUtil);
//    this.logTemplateRepositories.save(vo);
//    //4、
//    RMap<String, String> template = redissonClient.getMap(LOG_TEMPLATE_REDIS_PATH);
//    template.put(vo.getCode(), JSONObject.toJSONString(vo));
  }

  /**
   * 校验集合字段的唯一主键是否存在
   * @param fieldMap
   */
  private void validateOnlyKey(Map<String, LogFieldEntity> fieldMap) {
    if (!CollectionUtil.mapNotEmpty(fieldMap)) {
      return;
    }
    fieldMap.forEach((k, v) -> {
      if (v.getHostType() == 2) {
        ValidateUtils.validate(v.getOnlyKey(), "名称为【%s】类型为【%s】的字段因为是集合，所以onlyKey（主键）名称不能为空", v.getFieldName(), v.getType());
      }
      this.validateOnlyKey(v.getChildren());
    });
  }

  /**
   * 去掉无用字段
   * @param fields
   */
  private void removeUnCompareField(Map<String, LogFieldEntity> fields) {
//    if (!CollectionUtil.mapNotEmpty(fields)) {
//      return;
//    }
//    Set<String> codes = Sets.newHashSet();
//    fields.forEach((k, v) -> {
//      if (!v.isCompare()) {
//        codes.add(k);
//      } else {
//        this.removeUnCompareField(v.getChildren());
//      }
//    });
//    codes.forEach(c -> fields.remove(c));
  }

  /**
   * 1、校验参数非空和原始记录是否存在
   * 2、保存到es
   * 3、保存到redis
   * @param vo
   */
  @Override
  public void update(LogTemplateEntity vo) {
//    //1、
//    ValidateUtils.validate(vo, "编辑时，参数不能为空");
//    ValidateUtils.validate(vo.getCode(), "编辑时，模板编码不能为空");
//    vo.setFieldMap(this.transferFieldMap(vo.getFieldList()));
//    vo.setFieldList(null);
//    ValidateUtils.notEmpty(vo.getFieldMap(), "编辑时，字段列表不能为空");
//    LogTemplateEntity logTemplateEntity = this.logTemplateRepositories.findById(vo.getId()).orElse(null);
//    ValidateUtils.validate(logTemplateEntity, "没有获取到原始记录，请确认");
//    this.validateOnlyKey(vo.getFieldMap());
//    this.removeUnCompareField(vo.getFieldMap());
//    //2、
//    vo.setModifyTime(new Date());
//    UserRedis user = UserUtils.getUser();
//    if (user != null) {
//      vo.setModifyAccount(user.getUsername());
//    }
//    this.logTemplateRepositories.save(vo);
//    //3、
//    RMap<String, String> template = redissonClient.getMap(LOG_TEMPLATE_REDIS_PATH);
//    template.put(vo.getCode(), JSONObject.toJSONString(vo));
  }

  @Override
  public LogTemplateEntity read(String templateCode) {
    ValidateUtils.validate(templateCode, "读取日志模板时，模板编码不能为空");
    RMap<String, String> template = redissonClient.getMap(LOG_TEMPLATE_REDIS_PATH);
    String json = template.get(templateCode);
    LogTemplateEntity logTemplateVo = JSONObject.parseObject(json, LogTemplateEntity.class);
    return logTemplateVo;
  }

  @Override
  public PageResult<LogTemplateEntity> findPageByConditions(LogTemplateEntity req) {
//    ValidateUtils.isTrue(req.getPageNum() > 0, "分页查询时页码必须大于0");
//    ValidateUtils.isTrue(req.getPageSize() > 0, "分页查询时每页显示条数必须大于0");
//    PageRequest page = PageRequest.of(req.getPageNum() - 1, req.getPageSize());
//    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
//    if (req.getStartTime() != null && req.getEndTime() != null) {
//      boolQueryBuilder.filter(QueryBuilders.rangeQuery("createTime").from(req.getStartTime()).to(req.getEndTime()));
//    }
//    if (StringUtils.isNotEmpty(req.getCode())) {
//      boolQueryBuilder.filter(QueryBuilders.matchQuery("code", req.getCode()));
//    }
//    SearchQuery searchQuery = new NativeSearchQueryBuilder()
//            .withIndices(CrmLogContants.ES_LOG_TEMPLATE_INDEX)//索引名
//            .withQuery(boolQueryBuilder)//查询条件
//            .withSort(SortBuilders.fieldSort("createTime").unmappedType("keyword").order(SortOrder.DESC))
//            .withPageable(page)//分页参数
//            .build();
//    Page<LogTemplateEntity> pageResult = elasticsearchTemplate.queryForPage(searchQuery, LogTemplateEntity.class);
//    if(CollectionUtil.listNotEmpty(pageResult.getContent())) {
//      pageResult.getContent().forEach(c -> c.setFieldList(this.transferFieldList(c.getFieldMap())));
//    }
//    PageResult<LogTemplateEntity> result = PageResult.<LogTemplateEntity>builder().data(pageResult.getContent())
//            .count(pageResult.getTotalElements()).build();
//    return result;
    return null;
  }

  @Override
  public void enable(String templateCode, boolean status) {
    ValidateUtils.validate(templateCode, "删除操作时，模板编码不能为空");
    LogTemplateEntity logTemplateEntity = this.logTemplateRepositories.findByCode(templateCode);
    ValidateUtils.validate(logTemplateEntity, "没有获取到原始记录，请确认编码是否正确传入");
//    logTemplateEntity.setEnableState(status);
    this.update(logTemplateEntity);
  }

  @Override
  public void deleteByCode(String templateCode) {
    ValidateUtils.validate(templateCode, "删除操作时，模板编码不能为空");
    LogTemplateEntity logTemplateEntity = this.logTemplateRepositories.findByCode(templateCode);
    ValidateUtils.validate(logTemplateEntity, "没有获取到原始记录，请确认编码是否正确传入");
    this.logTemplateRepositories.delete(logTemplateEntity);
    RMap<String, String> template = redissonClient.getMap(LOG_TEMPLATE_REDIS_PATH);
    template.remove(templateCode);
  }

  @Override
  public List<LogFieldEntity> findFieldListByType(String type) {
    Map<String, LogFieldEntity> map = this.findFieldsByType(type);
    return this.transferFieldList(map);
  }

  @Override
  public List<LogFieldEntity> findFieldListByTypeAndCode(String type, String templateCode, String tree) {
    Map<String, LogFieldEntity> map = this.findFieldsByTypeAndCode(type, templateCode, tree);
    return this.transferFieldList(map);
  }

  @Override
  public void delByIds(ArrayList<String> ids) {
    ValidateUtils.validate(ids, "删除操作时，模板编码不能为空");
    for(String id : ids) {
      LogTemplateEntity logTemplateEntity = this.logTemplateRepositories.findById(id).orElse(null);
      ValidateUtils.validate(logTemplateEntity, "没有获取到原始记录，请确认id【%s】是否正确", id);
      this.logTemplateRepositories.delete(logTemplateEntity);
    }
  }

  @Override
  public String readJson(String templateCode) {
    LogTemplateEntity logTemplateEntity = this.read(templateCode);
    if(logTemplateEntity == null) {
      return null;
    }
    return JSONObject.toJSONString(logTemplateEntity);
  }

  /**
   * 字段从map转化为list
   * @param list
   * @return
   */
  private Map<String, LogFieldEntity> transferFieldMap(List<LogFieldEntity> list) {
    if(CollectionUtil.listEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, LogFieldEntity> map = Maps.newTreeMap();
    this.buildChildMap(map, list);
    return map;
  }

  /**
   * 递归转化子级
   * @param map
   * @param list
   */
  private void buildChildMap(Map<String, LogFieldEntity> map, List<LogFieldEntity> list) {
//    if(CollectionUtil.listEmpty(list)) {
//      return;
//    }
//    list.forEach(li -> {
//      map.put(li.getFieldName(), li);
//      if(CollectionUtil.listNotEmpty(li.getFieldList())) {
//        Map<String, LogFieldEntity> child = Maps.newTreeMap();
//        li.setChildren(child);
//        this.buildChildMap(child, li.getFieldList());
//      }
//    });
  }

  /**
   * 字段从map转化为list
   * @param map
   * @return
   */
  private List<LogFieldEntity> transferFieldList(Map<String, LogFieldEntity> map) {
    if(!CollectionUtil.mapNotEmpty(map)) {
      return Lists.newArrayList();
    }
    List<LogFieldEntity> list = Lists.newArrayList();
    this.buildChildList(list, map);
    return list;
  }

  /**
   * 递归转化子字段
   * @param list
   * @param map
   */
  private void buildChildList(List<LogFieldEntity> list, Map<String, LogFieldEntity> map) {
    if(!CollectionUtil.mapNotEmpty(map)) {
      return;
    }
    map.forEach((k, v) -> {
      list.add(v);
      if(CollectionUtil.mapNotEmpty(v.getChildren())) {
        List<LogFieldEntity> fieldEntities = Lists.newArrayList();
//        v.setFieldList(fieldEntities);
        this.buildChildList(fieldEntities, v.getChildren());
      }
    });
  }

  /**
   * 构建日志模板字段对象
   * @param field
   * @return
   */
  private LogFieldEntity buildFieldVo(Field field) {
    ValidateUtils.validate(field, "获取字段异常");
    LogFieldEntity filedVo = new LogFieldEntity();
    filedVo.setFieldName(field.getName());
    filedVo.setType(field.getType().getName());
    ApiModelProperty apiModelProperty = field.getAnnotation(ApiModelProperty.class);
    if (apiModelProperty != null) {
      filedVo.setDesc(apiModelProperty.value());
    }
    if (this.isBasicType(field.getType())) {
      filedVo.setHostType(0);
      filedVo.setHostTypeName("基础类型");
    } else if (field.getType() == Collection.class
            || field.getType() == List.class
            || field.getType() == Set.class) {
      Type fType = field.getGenericType();
      ParameterizedType pType = (ParameterizedType) fType;
      if (pType != null && pType.getActualTypeArguments() != null && pType.getActualTypeArguments().length > 0) {
        filedVo.setType(pType.getActualTypeArguments()[0].getTypeName());
      }
      filedVo.setHostType(2);
      filedVo.setHostTypeName("集合类型");
    } else if (field.getType() == Map.class) {
      Type fType = field.getGenericType();
      ParameterizedType pType = (ParameterizedType) fType;
      if (pType != null && pType.getActualTypeArguments() != null && pType.getActualTypeArguments().length > 1) {
        filedVo.setType(pType.getActualTypeArguments()[1].getTypeName());
      }
      filedVo.setHostType(3);
      filedVo.setHostTypeName("MAP类型");
    } else if (field.getType() != Map.class) {
      filedVo.setHostType(1);
      filedVo.setHostTypeName("对象类型");
    }
    return filedVo;
  }

  /**
   * 判断是否基础类型
   * @param type
   * @return
   */
  private boolean isBasicType(Class<?> type) {
    return  type == int.class || type == Integer.class
            || type == double.class || type == Double.class
            || type == long.class || type == Long.class
            || type == float.class || type == Float.class
            || type == char.class || type == Character.class
            || type == byte.class || type == Byte.class
            || type == String.class || type == Date.class
            || type == java.sql.Date.class || type == BigDecimal.class
            || type == BigInteger.class || type == File.class
            || type == InputStream.class || type == OutputStream.class
            || type.isArray();
  }

  /**
   * 获取所有字段，包括所有父类
   * @param c
   * @return
   */
  private List<Field> getFields(Class c) {
    List<Field> fields = Lists.newArrayList();
    while (c != null) {
      fields.addAll(Arrays.asList(c.getDeclaredFields()));
      c = c.getSuperclass();
    }
    return fields;
  }
}
