package com.biz.crm.log.template.service.impl;

import com.alibaba.fastjson.JSON;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.log.template.entity.LogFieldEntity;
import com.biz.crm.log.template.entity.LogTemplateEntity;
import com.biz.crm.log.template.repositories.LogTemplateRepositories;
import com.biz.crm.log.template.service.LogTemplateService;
import com.biz.crm.log.template.util.LogTemplateUtil;
import com.biz.crm.log.utils.CrmLogContants;
import com.biz.crm.nebular.log.template.LogFieldVo;
import com.biz.crm.nebular.log.template.LogTemplateVo;
import com.biz.crm.util.EsUtil;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.redisson.api.RedissonClient;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchTemplate;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 日志规则逻辑处理实现类
 * @Author: chenrong
 * @Date: 2020/12/9 15:13
 */
@ConditionalOnMissingBean(name = "logTemplateServiceImpl")
@Service(value = "logTemplateService")
public class LogTemplateServiceImpl implements LogTemplateService {

  @Resource
  private RedissonClient redissonClient;
  @Resource
  private ElasticsearchTemplate elasticsearchTemplate;
  @Resource
  private RedissonUtil redissonUtil;
  @Resource
  private LogTemplateRepositories logTemplateRepositories;

  /**
   * 1、验证
   * 2、转换并保存
   * @param vo
   */
  @Override
  public void add(LogTemplateVo vo) {
    //1
    ValidateUtils.validate(vo.getBusinessName(),"业务模块名字不能为空!");
    ValidateUtils.validate(vo.getMenuCode(),"业务模块编码不能为空!");
    ValidateUtils.validate(vo.getTypeClassPath(),"模板类全路径不能为空!");
    ValidateUtils.validate(vo.getFieldList(),"字段列表不能为空!");

    //2
    LogTemplateEntity logTemplateEntity = LogTemplateUtil.voToEntityForAdd(vo);

    //3
    logTemplateRepositories.save(logTemplateEntity);
  }

  @Override
  public PageResult<LogTemplateVo> listWithPage(LogTemplateVo vo) {
    PageResult<LogTemplateVo> pageResult = new PageResult<LogTemplateVo>();
    pageResult.setCount(Long.parseLong("0"));
    pageResult.setData(new ArrayList<>());
    if(!EsUtil.indexExsit(elasticsearchTemplate,LogTemplateEntity.class,redissonUtil)){
      return pageResult;
    }

    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
    if(!StringUtils.isEmpty(vo.getMenuCode())){
      boolQueryBuilder.must(QueryBuilders.termQuery("menuCode.keyword",vo.getMenuCode()));
    }
    if(null != vo.getEnableFlag()){
      boolQueryBuilder.must(QueryBuilders.termQuery("enableFlag",vo.getEnableFlag()));
    }
    if(!StringUtils.isEmpty(vo.getBusinessName())){
      boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("businessName",vo.getBusinessName()));
    }
    String[] fileds = {"id","menuCode","businessName","typeClassPath","enableFlag"};
    SearchQuery searchQuery = new NativeSearchQueryBuilder()
            .withIndices(CrmLogContants.ES_LOG_TEMPLATE_INDEX)//索引名
            .withQuery(boolQueryBuilder)//查询条件
            .withPageable(PageRequest.of(vo.getPageNum().intValue()-1,vo.getPageSize()))//分页参数
            .withFields(fileds)
            .build();
    Page<LogTemplateVo> scroll = elasticsearchTemplate.queryForPage(searchQuery,LogTemplateVo.class);
    List<LogTemplateVo> list = scroll.getContent();
    if(CollectionUtils.isEmpty(list)){
      return pageResult;
    }
    pageResult.setData(list);
    pageResult.setCount(scroll.getTotalElements());
    return pageResult;
  }

  /**
   * 1、根据id查询模板所有信息
   * 2、处理字段信息
   * @param id
   * @return
   */
  @Override
  public LogTemplateVo findById(String id) {
    LogTemplateVo reVo = new LogTemplateVo();
    //1
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(id);
    if(null == optional || null == optional.get()){
      throw new BusinessException("您指定的模板不存在或已经被删除!");
    }
    LogTemplateEntity LogTemplateEntity = optional.get();
    BeanUtils.copyProperties(LogTemplateEntity,reVo);
    //2
    Map<String, LogFieldEntity> fieldMap = LogTemplateEntity.getFieldMap();
    reVo.setFieldList(new ArrayList<>());
    if(!CollectionUtils.isEmpty(fieldMap)){
      List<LogFieldEntity> fieldEntityList = new ArrayList<LogFieldEntity>(fieldMap.values());
      reVo.setFieldList(JSON.parseArray(
              JSON.toJSONString(fieldEntityList),
              LogFieldVo.class
      ));
    }
    return reVo;
  }

  /**
   * 1、验证
   * 2、转换并且保存
   * @param vo
   */
  @Override
  public void update(LogTemplateVo vo) {
    //1
    ValidateUtils.validate(vo.getId(),"请指定要编辑的模板!");
    ValidateUtils.validate(vo.getBusinessName(),"业务模块名字不能为空!");
    ValidateUtils.validate(vo.getMenuCode(),"业务模块编码不能为空!");
    ValidateUtils.validate(vo.getTypeClassPath(),"模板类全路径不能为空!");
    ValidateUtils.validate(vo.getFieldList(),"字段列表不能为空!");
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(vo.getId());
    if(null == optional || null == optional.get()){
      throw new BusinessException("指定的模板不存在货已经被删除！");
    }
    //2
    LogTemplateEntity logTemplateEntity = LogTemplateUtil.voToEntityForUpdate(vo,optional.get());

    //3
    logTemplateRepositories.deleteById(vo.getId());
    logTemplateRepositories.save(logTemplateEntity);
  }

  /**
   * 1、验证
   * 2、修改
   * @param id
   * @param yesNoCodeNumberEnum
   */
  @Override
  public void updateEnable(String id, YesNoEnum.YesNoCodeNumberEnum yesNoCodeNumberEnum) {
    //1
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(id);
    if(null == optional || null == optional.get()){
      throw new BusinessException("您指定的模板不存在或已经被删除!");
    }

    //2
    LogTemplateEntity LogTemplateEntity = optional.get();
    LogTemplateEntity.setEnableFlag(yesNoCodeNumberEnum.getCode());
    logTemplateRepositories.deleteById(LogTemplateEntity.getId());
    logTemplateRepositories.save(LogTemplateEntity);
  }

  @Override
  public void delByIds(ArrayList<String> ids) {
    logTemplateRepositories.deleteByIdIn(ids);
  }

  /**
   * 1、读取类
   * 2、读取字段列表
   * @param type
   * @return
   */
  @Override
  public List<LogFieldVo> findFieldsByType(String type) {
    //1
    Class clazz = null;
    try {
      clazz = Class.forName(type);
    } catch (ClassNotFoundException e) {
      throw new BusinessException("您指定的类不存在",e);
    }

    //2
    return LogTemplateUtil.packageFiledVo(LogTemplateUtil.getFields(clazz));
  }

  @Override
  public LogTemplateEntity findTemplateByCode(String code) {
    return logTemplateRepositories.findByMenuCode(code);
  }
}
