package com.biz.crm.crmlog.template.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.template.repositories.LogTemplateRepositories;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.log.LogEunm;
import com.biz.crm.crmlog.template.entity.LogFieldEntity;
import com.biz.crm.crmlog.template.entity.LogTemplateEntity;
import com.biz.crm.crmlog.template.service.LogTemplateService;
import com.biz.crm.crmlog.template.util.LogTemplateUtil;
import com.biz.crm.crmlog.utils.CrmLogContants;
import com.biz.crm.nebular.log.template.LogFieldVo;
import com.biz.crm.nebular.log.template.LogTemplateVo;
import com.biz.crm.util.EsUtil;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.redisson.api.RedissonClient;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchTemplate;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;

/**
 * 日志规则逻辑处理实现类
 * @Author: chenrong
 * @Date: 2020/12/9 15:13
 */
@ConditionalOnMissingBean(name = "logTemplateServiceImpl")
@Service(value = "logTemplateService")
public class LogTemplateServiceImpl implements LogTemplateService {

  @Resource
  private RedissonClient redissonClient;
  @Resource
  private ElasticsearchTemplate elasticsearchTemplate;
  @Resource
  private RedissonUtil redissonUtil;
  @Resource
  private LogTemplateRepositories logTemplateRepositories;

  /**
   * 1、验证
   * 2、转换并保存
   * @param vo
   */
  @Override
  public void add(LogTemplateVo vo) {
    //1
    ValidateUtils.validate(vo.getBusinessName(),"业务模块名字不能为空!");
    ValidateUtils.validate(vo.getMenuCode(),"业务模块编码不能为空!");
    ValidateUtils.validate(vo.getTypeClassPath(),"模板类全路径不能为空!");
    ValidateUtils.validate(vo.getFieldList(),"字段列表不能为空!");

    //2
    LogTemplateEntity logTemplateEntity = LogTemplateUtil.voToEntityForAdd(vo);

    //3
    logTemplateRepositories.save(logTemplateEntity);
  }

  @Override
  public PageResult<LogTemplateVo> listWithPage(LogTemplateVo vo) {
    PageResult<LogTemplateVo> pageResult = new PageResult<LogTemplateVo>();
    pageResult.setCount(Long.parseLong("0"));
    pageResult.setData(new ArrayList<>());
    if(!EsUtil.indexExsit(elasticsearchTemplate,LogTemplateEntity.class,redissonUtil)){
      return pageResult;
    }

    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
    if(!StringUtils.isEmpty(vo.getMenuCode())){
      boolQueryBuilder.must(QueryBuilders.termQuery("menuCode.keyword",vo.getMenuCode()));
    }
    if(null != vo.getEnableFlag()){
      boolQueryBuilder.must(QueryBuilders.termQuery("enableFlag",vo.getEnableFlag()));
    }
    if(!StringUtils.isEmpty(vo.getBusinessName())){
      boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("businessName",vo.getBusinessName()));
    }
    String[] fileds = {"id","menuCode","businessName","typeClassPath","enableFlag"};
    SearchQuery searchQuery = new NativeSearchQueryBuilder()
            .withIndices(CrmLogContants.ES_LOG_TEMPLATE_INDEX)//索引名
            .withQuery(boolQueryBuilder)//查询条件
            .withPageable(PageRequest.of(vo.getPageNum().intValue()-1,vo.getPageSize()))//分页参数
            .withFields(fileds)
            .build();
    Page<LogTemplateEntity> scroll = elasticsearchTemplate.queryForPage(searchQuery,LogTemplateEntity.class);
    List<LogTemplateEntity> list = scroll.getContent();
    if(CollectionUtils.isEmpty(list)){
      return pageResult;
    }
    List<LogTemplateVo> listVo = new ArrayList<>(list.size());
    for(LogTemplateEntity entity : list){
      LogTemplateVo logTemplateVo = new LogTemplateVo();
      BeanUtils.copyProperties(entity,logTemplateVo);
      listVo.add(logTemplateVo);
    }
    pageResult.setData(listVo);
    pageResult.setCount(scroll.getTotalElements());
    return pageResult;
  }

  /**
   * 1、根据id查询模板所有信息
   * 2、处理字段信息
   * @param id
   * @return
   */
  @Override
  public LogTemplateVo findById(String id) {
    LogTemplateVo reVo = new LogTemplateVo();
    //1
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(id);
    if(null == optional || null == optional.get()){
      throw new BusinessException("您指定的模板不存在或已经被删除!");
    }
    LogTemplateEntity logTemplateEntity = optional.get();
    BeanUtils.copyProperties(logTemplateEntity,reVo);
    //2
    List<LogFieldVo> fieldList = new ArrayList<>();
    reVo.setFieldList(fieldList);
    String fieldListStr = logTemplateEntity.getFieldListStr();//字符串字段模板
    if(StringUtils.isEmpty(fieldListStr)){
      return reVo;
    }
    List<LogFieldEntity> fileds = JSONArray.parseArray(fieldListStr,LogFieldEntity.class);
    for(LogFieldEntity logFieldEntity : fileds){
      LogFieldVo logFieldVo = new LogFieldVo();
      BeanUtils.copyProperties(logFieldEntity,logFieldVo);
      fieldList.add(logFieldVo);
      if(logFieldVo.getHostType().intValue() != LogEunm.HostTypeEunm.BASIC.getCode().intValue()
              && !StringUtils.isEmpty(logFieldEntity.getChildrenListStr())){
        logFieldVo.setChildrenList(JSONArray.parseArray(logFieldEntity.getChildrenListStr(),LogFieldVo.class));
      }
    }
    return reVo;
  }

  /**
   * 1、验证
   * 2、转换并且保存
   * @param vo
   */
  @Override
  public void update(LogTemplateVo vo) {
    //1
    ValidateUtils.validate(vo.getId(),"请指定要编辑的模板!");
    ValidateUtils.validate(vo.getBusinessName(),"业务模块名字不能为空!");
    ValidateUtils.validate(vo.getMenuCode(),"业务模块编码不能为空!");
    ValidateUtils.validate(vo.getTypeClassPath(),"模板类全路径不能为空!");
    ValidateUtils.validate(vo.getFieldList(),"字段列表不能为空!");
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(vo.getId());
    if(null == optional || null == optional.get()){
      throw new BusinessException("指定的模板不存在货已经被删除！");
    }
    //2
    LogTemplateEntity logTemplateEntity = LogTemplateUtil.voToEntityForUpdate(vo,optional.get());

    //3
    logTemplateRepositories.deleteById(vo.getId());
    logTemplateRepositories.save(logTemplateEntity);
  }

  /**
   * 1、验证
   * 2、修改
   * @param id
   * @param yesNoCodeNumberEnum
   */
  @Override
  public void updateEnable(String id, YesNoEnum.YesNoCodeNumberEnum yesNoCodeNumberEnum) {
    //1
    Optional<LogTemplateEntity> optional = logTemplateRepositories.findById(id);
    if(null == optional || null == optional.get()){
      throw new BusinessException("您指定的模板不存在或已经被删除!");
    }

    //2
    LogTemplateEntity LogTemplateEntity = optional.get();
    LogTemplateEntity.setEnableFlag(yesNoCodeNumberEnum.getCode());
    logTemplateRepositories.deleteById(LogTemplateEntity.getId());
    logTemplateRepositories.save(LogTemplateEntity);
  }

  @Override
  public void delByIds(ArrayList<String> ids) {
    logTemplateRepositories.deleteByIdIn(ids);
  }

  /**
   * 1、读取类
   * 2、读取字段列表
   * @param type
   * @return
   */
  @Override
  public List<LogFieldVo> findFieldsByType(String type) {
    //1
    Class clazz = null;
    try {
      clazz = Class.forName(type);
    } catch (ClassNotFoundException e) {
      throw new BusinessException("您指定的类不存在",e);
    }

    //2
    return LogTemplateUtil.packageFiledVo(LogTemplateUtil.getFields(clazz));
  }

  /**
   * 1、查询出实体类
   * 2、转换数据
   * @param code
   * @return
   */
  @Override
  public LogTemplateEntity findTemplateByCode(String code) {
    //1
    LogTemplateEntity logTemplateEntity = logTemplateRepositories.findByMenuCode(code);
    Map<String, LogFieldEntity> fieldMap = new HashMap<>();
    logTemplateEntity.setFieldMap(fieldMap);
    if(null == logTemplateEntity){
      return null;
    }

    //2
    if(StringUtils.isEmpty(logTemplateEntity.getFieldListStr())){
      return null;
    }
    List<LogFieldEntity> fieldList = JSONArray.parseArray(logTemplateEntity.getFieldListStr(),LogFieldEntity.class);
    if(CollectionUtils.isEmpty(fieldList)){
      return null;
    }
    for(LogFieldEntity logFieldEntity : fieldList){
      fieldMap.put(logFieldEntity.getFieldName(),logFieldEntity);
      //子字段map
      Map<String, LogFieldEntity> childrenMap = new HashMap<>();
      logFieldEntity.setChildren(childrenMap);
      if(StringUtils.isEmpty(logFieldEntity.getChildrenListStr())){
        continue;
      }
      List<LogFieldEntity> childrenList = JSONArray.parseArray(logFieldEntity.getChildrenListStr(),LogFieldEntity.class);
      if(CollectionUtils.isEmpty(fieldList)){
        continue;
      }
      for(LogFieldEntity chiledFiled : childrenList){
        childrenMap.put(chiledFiled.getFieldName(),chiledFiled);
      }
    }
    return logTemplateEntity;
  }
}
