package com.biz.crm.approval.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.approval.mapper.MdmApprovalConfigMapper;
import com.biz.crm.approval.model.MdmApprovalConfigEntity;
import com.biz.crm.approval.service.MdmApprovalConfigConditionService;
import com.biz.crm.approval.service.MdmApprovalConfigService;
import com.biz.crm.approval.service.MdmApprovalConfigSubmitterService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.MdmApprovalConfig;
import com.biz.crm.nebular.mdm.approval.req.MdmApprovalConfigConditionReqVo;
import com.biz.crm.nebular.mdm.approval.req.MdmApprovalConfigReqVo;
import com.biz.crm.nebular.mdm.approval.req.MdmApprovalConfigSubmitterReqVo;
import com.biz.crm.nebular.mdm.approval.resp.*;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 审批配置主表接口实现
 *
 * @author Tao.Chen
 * @date 2021-03-01 17:50:29
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmApprovalConfigServiceExpandImpl")
public class MdmApprovalConfigServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmApprovalConfigMapper, MdmApprovalConfigEntity> implements MdmApprovalConfigService {

    @Resource
    private MdmApprovalConfigMapper mdmApprovalConfigMapper;
    @Resource
    private MdmApprovalConfigConditionService mdmApprovalConfigConditionService;
    @Resource
    private MdmApprovalConfigSubmitterService mdmApprovalConfigSubmitterService;

    @Override
    public PageResult<MdmApprovalConfigRespVo> findApprovalConfigPageList(MdmApprovalConfigReqVo mdmApprovalConfigReqVo) {
        Page<MdmApprovalConfigRespVo> page = PageUtil.buildPage(mdmApprovalConfigReqVo.getPageNum(), mdmApprovalConfigReqVo.getPageSize());
        List<MdmApprovalConfigRespVo> list = mdmApprovalConfigMapper.findApprovalConfigPageList(page, mdmApprovalConfigReqVo);
        return PageResult.<MdmApprovalConfigRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmApprovalConfigRespVo> findApprovalConfigList(MdmApprovalConfigReqVo reqVo) {
        List<MdmApprovalConfigEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getMenuCode()), MdmApprovalConfigEntity::getMenuCode, reqVo.getMenuCode())
                .eq(StringUtils.isNotEmpty(reqVo.getBusinessCode()), MdmApprovalConfigEntity::getBusinessCode, reqVo.getBusinessCode())
                .eq(StringUtils.isNotEmpty(reqVo.getControlFlag()), MdmApprovalConfigEntity::getControlFlag, reqVo.getControlFlag())
                .eq(StringUtils.isNotEmpty(reqVo.getApprovalType()), MdmApprovalConfigEntity::getApprovalType, reqVo.getApprovalType())
                .eq(StringUtils.isNotEmpty(reqVo.getControlScope()), MdmApprovalConfigEntity::getControlScope, reqVo.getControlScope())
                .eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()), MdmApprovalConfigEntity::getEnableStatus, reqVo.getEnableStatus())
                .like(StringUtils.isNotEmpty(reqVo.getBusinessName()), MdmApprovalConfigEntity::getBusinessName, reqVo.getBusinessName())
                .orderByDesc(MdmApprovalConfigEntity::getCreateDate)
                .orderByDesc(MdmApprovalConfigEntity::getCreateDateSecond)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return CrmBeanUtil.copyList(list, MdmApprovalConfigRespVo.class);
        }
        return new ArrayList<>();
    }

    @Override
    public MdmApprovalConfigRespVo queryById(String id) {
        Assert.hasText(id, "缺失id");
        MdmApprovalConfigEntity entity = this.getById(id);
        if (entity != null) {
            MdmApprovalConfigRespVo respVo = CrmBeanUtil.copy(entity, MdmApprovalConfigRespVo.class);
            respVo.setConditionList(mdmApprovalConfigConditionService.findConditionList(respVo.getBusinessCode()));
            respVo.setSubmitterList(mdmApprovalConfigSubmitterService.findSubmitterList(respVo.getBusinessCode()));
            return respVo;
        }
        return null;
    }

    @Override
    public MdmApprovalConfigRedisVo queryForRedis(String businessCode) {
        Assert.hasText(businessCode, "缺失业务编码");
        MdmApprovalConfigEntity entity = this.lambdaQuery()
                .eq(MdmApprovalConfigEntity::getBusinessCode, businessCode)
                .one();
        if (entity != null) {
            MdmApprovalConfigRedisVo redisVo = CrmBeanUtil.copy(entity, MdmApprovalConfigRedisVo.class);
            List<MdmApprovalConfigConditionRespVo> conditionList = mdmApprovalConfigConditionService.findConditionList(businessCode);
            if (CollectionUtil.listNotEmptyNotSizeZero(conditionList)) {
                redisVo.setConditionList(CrmBeanUtil.copyList(conditionList, MdmApprovalConfigConditionRedisVo.class));
            }
            List<MdmApprovalConfigSubmitterRespVo> submitterList = mdmApprovalConfigSubmitterService.findSubmitterList(businessCode);
            if (CollectionUtil.listNotEmptyNotSizeZero(submitterList)) {
                redisVo.setSubmitterList(CrmBeanUtil.copyList(submitterList, MdmApprovalConfigSubmitterRedisVo.class));
            }
            return redisVo;
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmApprovalConfigReqVo reqVo) {
        reqVo.setId(null);
        this.verify(reqVo);
        MdmApprovalConfigEntity entity = CrmBeanUtil.copy(reqVo, MdmApprovalConfigEntity.class);
        this.save(entity);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getConditionList())) {
            mdmApprovalConfigConditionService.saveConditionList(reqVo.getBusinessCode(), reqVo.getConditionList());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSubmitterList())) {
            mdmApprovalConfigSubmitterService.saveSubmitterList(reqVo.getBusinessCode(), reqVo.getSubmitterList());
        }
        //清除缓存
        ApprovalConfigUtil.deleteCache(entity.getBusinessCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmApprovalConfigReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        this.verify(reqVo);
        MdmApprovalConfigEntity entity = this.getById(reqVo.getId());
        mdmApprovalConfigConditionService.deleteConditionList(entity.getBusinessCode());
        mdmApprovalConfigSubmitterService.deleteSubmitterList(entity.getBusinessCode());
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getConditionList())) {
            mdmApprovalConfigConditionService.saveConditionList(reqVo.getBusinessCode(), reqVo.getConditionList());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSubmitterList())) {
            mdmApprovalConfigSubmitterService.saveSubmitterList(reqVo.getBusinessCode(), reqVo.getSubmitterList());
        }
        //清除缓存
        ApprovalConfigUtil.deleteCache(entity.getBusinessCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notNull(ids, "缺失参数");
        List<MdmApprovalConfigEntity> list = this.lambdaQuery()
                .in(MdmApprovalConfigEntity::getId, ids)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                mdmApprovalConfigConditionService.deleteConditionList(item.getBusinessCode());
                mdmApprovalConfigSubmitterService.deleteSubmitterList(item.getBusinessCode());
            });
            this.removeByIds(list.stream().map(MdmApprovalConfigEntity::getId).collect(Collectors.toList()));
            //清除缓存
            ApprovalConfigUtil.deleteCache(list.stream().map(MdmApprovalConfigEntity::getBusinessCode).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void open(List<String> ids) {
        Assert.notNull(ids, "缺失参数");
        List<MdmApprovalConfigEntity> list = this.lambdaQuery()
                .in(MdmApprovalConfigEntity::getId, ids)
                .list()
                .stream().filter(x -> !YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getControlFlag())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setControlFlag(YesNoEnum.yesNoEnum.ONE.getValue());
            });
            this.updateBatchById(list);
            //清除缓存
            ApprovalConfigUtil.deleteCache(list.stream().map(MdmApprovalConfigEntity::getBusinessCode).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void close(List<String> ids) {
        Assert.notNull(ids, "缺失参数");
        List<MdmApprovalConfigEntity> list = this.lambdaQuery()
                .in(MdmApprovalConfigEntity::getId, ids)
                .list()
                .stream().filter(x -> !YesNoEnum.yesNoEnum.ZERO.getValue().equals(x.getControlFlag())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setControlFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
            });
            this.updateBatchById(list);
            //清除缓存
            ApprovalConfigUtil.deleteCache(list.stream().map(MdmApprovalConfigEntity::getBusinessCode).collect(Collectors.toList()));
        }
    }

    /**
     * 校验
     *
     * @param reqVo
     */
    private void verify(MdmApprovalConfigReqVo reqVo) {
        Assert.hasText(reqVo.getBusinessCode(), "缺失业务编码");
        Assert.hasText(reqVo.getBusinessName(), "缺失业务名称");
        Assert.hasText(reqVo.getControlFlag(), "缺失开启状态");
        Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(reqVo.getControlFlag()) || YesNoEnum.yesNoEnum.ZERO.getValue().equals(reqVo.getControlFlag()), "开启状态错误");
        List<MdmApprovalConfigEntity> list = this.lambdaQuery()
                .eq(MdmApprovalConfigEntity::getBusinessCode, reqVo.getBusinessCode())
                .select(MdmApprovalConfigEntity::getId)
                .list()
                .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(list), "业务编码已存在，请重新输入");
        if (StringUtils.isNotEmpty(reqVo.getId())) {
            Assert.hasText(reqVo.getApprovalType(), "缺失审批类型");
            Assert.hasText(reqVo.getControlScope(), "缺失控制范围");
            if (MdmApprovalConfig.ControlScope.PART_IN_CONDITION.getValue().equals(reqVo.getControlScope()) || MdmApprovalConfig.ControlScope.PART_OUT_CONDITION.getValue().equals(reqVo.getControlScope())) {
                Assert.hasText(reqVo.getGroupCheckWay(), "条件组判断逻辑");
                if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getConditionList())) {
                    for (MdmApprovalConfigConditionReqVo condition :
                            reqVo.getConditionList()) {
                        Assert.hasText(condition.getFieldCode(), "缺失字段编码");
                        Assert.hasText(condition.getFieldName(), "缺失字段名称");
                        Assert.hasText(condition.getFieldType(), "缺失字段类型");
                        Assert.hasText(condition.getFieldKeywords(), "缺失字段值");
                        Assert.hasText(condition.getCompareType(), "缺失比较类型");
                        Assert.hasText(condition.getControlFlag(), "缺失开启状态");
                        Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(condition.getControlFlag()) || YesNoEnum.yesNoEnum.ZERO.getValue().equals(condition.getControlFlag()), "开启状态错误");
                        if (MdmApprovalConfig.FieldType.DICTIONARY.getValue().equals(condition.getFieldType())) {
                            Assert.hasText(condition.getFieldDataSource(), "缺失数据字典");
                            Map<String, String> dictMap = DictUtil.dictMap(condition.getFieldDataSource());
                            for (String keyword :
                                    condition.getFieldKeywords().split(",")) {
                                Assert.isTrue(dictMap.containsKey(keyword), "数据字典【" + condition.getFieldDataSource() + "】中不存在值【" + keyword + "】");
                            }
                        }
                    }
                }
                if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSubmitterList())) {
                    for (MdmApprovalConfigSubmitterReqVo submitter :
                            reqVo.getSubmitterList()) {
                        Assert.hasText(submitter.getSubmitterType(), "缺失提交人匹配对象");
                        Assert.hasText(submitter.getControlFlag(), "缺失开启状态");
                        Assert.hasText(submitter.getFieldKeywords(), "缺失字段值");
                        Assert.hasText(submitter.getCompareType(), "缺失比较类型");
                        Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(submitter.getControlFlag()) || YesNoEnum.yesNoEnum.ZERO.getValue().equals(submitter.getControlFlag()), "开启状态错误");
                    }
                }
            }
        }
    }

}
