package com.biz.crm.bpmrole.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.bpmrole.mapper.MdmBpmRoleMapper;
import com.biz.crm.bpmrole.model.MdmBpmRoleEntity;
import com.biz.crm.bpmrole.service.MdmBpmRoleService;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.bpmrole.req.MdmBpmRoleReqVo;
import com.biz.crm.nebular.mdm.bpmrole.req.MdmBpmRoleSelectReqVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRedisRespVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRespVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleSelectRespVo;
import com.biz.crm.position.service.MdmPositionBpmRoleService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 流程角色表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-26 20:07:10
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmBpmRoleServiceExpandImpl")
public class MdmBpmRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmBpmRoleMapper, MdmBpmRoleEntity> implements MdmBpmRoleService {

    @Resource
    private MdmBpmRoleMapper mdmBpmRoleMapper;
    @Resource
    private MdmPositionBpmRoleService mdmPositionBpmRoleService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Resource
    private RocketMQProducer rocketMQProducer;

    @Override
    public PageResult<MdmBpmRoleRespVo> findList(MdmBpmRoleReqVo reqVo) {
        Page<MdmBpmRoleRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmBpmRoleRespVo> list = mdmBpmRoleMapper.findList(page, reqVo);
        return PageResult.<MdmBpmRoleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmBpmRoleRespVo detail(String id, String bpmRoleCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(bpmRoleCode)) {
            MdmBpmRoleEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmBpmRoleEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(bpmRoleCode), MdmBpmRoleEntity::getBpmRoleCode, bpmRoleCode)
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmBpmRoleRespVo.class);
            }
        }
        return null;
    }

    @Override
    public List<MdmBpmRoleRespVo> queryBatchByBpmRoleCodeList(List<String> bpmRoleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList)) {
            List<String> collect = bpmRoleCodeList.stream().filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());
            if (!collect.isEmpty()) {
                List<MdmBpmRoleEntity> list = this.lambdaQuery()
                        .in(MdmBpmRoleEntity::getBpmRoleCode, collect)
                        .list();
                if (list != null && list.size() > 0) {
                    return CrmBeanUtil.copyList(list, MdmBpmRoleRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmBpmRoleReqVo reqVo) {
        reqVo.setId(null);
        Assert.hasText(reqVo.getBpmRoleName(), "缺失流程角色名称");
        MdmBpmRoleEntity entity = CrmBeanUtil.copy(reqVo, MdmBpmRoleEntity.class);
        if (StringUtils.isEmpty(entity.getBpmRoleCode())) {
            entity.setBpmRoleCode(CodeUtil.generateCode(CodeRuleEnum.MDM_BPM_ROLE_CODE.getCode()));
        } else {
            List<MdmBpmRoleEntity> list = this.lambdaQuery()
                    .eq(MdmBpmRoleEntity::getBpmRoleCode, reqVo.getBpmRoleCode())
                    .select(MdmBpmRoleEntity::getBpmRoleCode)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "当前流程角色编码已存在，请重新输入。");
        }
        List<MdmBpmRoleEntity> nameList = this.lambdaQuery()
                .eq(MdmBpmRoleEntity::getBpmRoleName, reqVo.getBpmRoleName())
                .select(MdmBpmRoleEntity::getBpmRoleName)
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(nameList), "当前流程角色名称已存在，请重新输入。");
        this.save(entity);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(menuObject.toString(), entity.getId(), entity.getBpmRoleCode(), CrmBeanUtil.copy(entity, MdmBpmRoleReqVo.class));
        }
        //发送MQ消息
        this.sendBpmRoleAdd(Collections.singletonList(CrmBeanUtil.copy(entity, MdmBpmRoleRespVo.class)));
        BpmRoleUtil.deleteAllCache();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmBpmRoleReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        Assert.hasText(reqVo.getBpmRoleCode(), "缺失流程角色编码");
        Assert.hasText(reqVo.getBpmRoleName(), "缺失流程角色名称");
        MdmBpmRoleEntity entity = this.getById(reqVo.getId());
        MdmBpmRoleReqVo oldObject = CrmBeanUtil.copy(entity,MdmBpmRoleReqVo.class);
        Assert.notNull(entity, "无效的id");
        Assert.isTrue(entity.getBpmRoleCode().equals(reqVo.getBpmRoleCode()), "流程角色编码不能修改");
        List<MdmBpmRoleEntity> nameList = this.lambdaQuery()
                .eq(MdmBpmRoleEntity::getBpmRoleName, reqVo.getBpmRoleName())
                .ne(MdmBpmRoleEntity::getId,reqVo.getId())
                .select(MdmBpmRoleEntity::getBpmRoleName,MdmBpmRoleEntity::getId)
                .list()
                .stream().filter(x -> !reqVo.getId().equals(x.getId()))
                .collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(nameList), "名称已经存在，不能重复添加");
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            MdmBpmRoleReqVo newObject = CrmBeanUtil.copy(entity, MdmBpmRoleReqVo.class);
            crmLogSendUtil.sendForUpdate(menuObject.toString(), reqVo.getId(), entity.getBpmRoleCode(), oldObject, newObject);
        }
        //发送MQ消息
        this.sendBpmRoleUpdate(Collections.singletonList(CrmBeanUtil.copy(entity, MdmBpmRoleRespVo.class)));
        BpmRoleUtil.deleteByBpmRoleCode(Collections.singletonList(entity.getBpmRoleCode()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmBpmRoleEntity entity = this.getById(id);
            Assert.notNull(entity, "无效的id");

            //删除与职位的关联关系
            mdmPositionBpmRoleService.removeByBpmRole(entity.getBpmRoleCode());

        });
        List<MdmBpmRoleEntity> list = this.lambdaQuery()
                .in(MdmBpmRoleEntity::getId,ids)
                .list();
        List<String> code = list.stream().map(MdmBpmRoleEntity::getBpmRoleCode).collect(Collectors.toList());
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            list.forEach(item -> {
                crmLogSendUtil.sendForDel(menuObject.toString(), item.getId(), item.getBpmRoleCode(), item);
            });
        }
        this.removeByIds(ids);
        //发送MQ消息
        this.sendBpmRoleDelete(CrmBeanUtil.copyList(list, MdmBpmRoleRespVo.class));
        BpmRoleUtil.deleteByBpmRoleCode(code);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmBpmRoleEntity> list = this.lambdaQuery()
                .in(MdmBpmRoleEntity::getId, ids)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(x -> {
                x.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
            this.updateBatchById(list);
            //发送MQ消息
            this.sendBpmRoleUpdate(CrmBeanUtil.copyList(list, MdmBpmRoleRespVo.class));
            BpmRoleUtil.deleteByBpmRoleCode(list.stream().map(MdmBpmRoleEntity::getBpmRoleCode).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmBpmRoleEntity> list = this.lambdaQuery()
                .in(MdmBpmRoleEntity::getId, ids)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(x -> {
                x.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
            this.updateBatchById(list);
            //发送MQ消息
            this.sendBpmRoleUpdate(CrmBeanUtil.copyList(list, MdmBpmRoleRespVo.class));
            BpmRoleUtil.deleteByBpmRoleCode(list.stream().map(MdmBpmRoleEntity::getBpmRoleCode).collect(Collectors.toList()));
        }
    }

    @Override
    public List<MdmBpmRoleSelectRespVo> selectList(MdmBpmRoleSelectReqVo reqVo) {
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmBpmRoleSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmBpmRoleMapper.bpmRoleSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }

    @Override
    public Set<String> findBpmRoleCodeSetByPositionCodeList(List<String> positionCodeList) {
        Assert.notEmpty(positionCodeList, "职位编码集合不能为空");
        return mdmPositionBpmRoleService.findBpmRoleCodeSetByPositionCodeList(positionCodeList);
    }

    @Override
    public List<MdmBpmRoleRedisRespVo> findRedisListByBpmRoleCodeList(List<String> bpmRoleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList)){
            List<MdmBpmRoleEntity> list = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList),MdmBpmRoleEntity::getBpmRoleCode,bpmRoleCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmBpmRoleRedisRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmBpmRoleRedisRespVo> findRedisList() {
        List<MdmBpmRoleEntity> list = this.lambdaQuery().list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)){
            return CrmBeanUtil.copyList(list,MdmBpmRoleRedisRespVo.class);
        }
        return new ArrayList<>();
    }

    protected void sendBpmRoleAdd(List<MdmBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位新增发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.BPM_ROLE_ADD_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位新增发送MQ消息成功：{}", list);
    }

    protected void sendBpmRoleUpdate(List<MdmBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位编辑发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.BPM_ROLE_UPDATE_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位编辑发送MQ消息成功：{}", list);
    }

    protected void sendBpmRoleDelete(List<MdmBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位删除发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.BPM_ROLE_DELETE_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位删除发送MQ消息成功：{}", list);
    }
}
