package com.biz.crm.button.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.button.mapper.MdmButtonRoleMapper;
import com.biz.crm.button.model.MdmButtonRoleEntity;
import com.biz.crm.button.service.MdmButtonRoleService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.buttonrole.MdmButtonRoleReqVo;
import com.biz.crm.nebular.mdm.buttonrole.MdmButtonRoleRespVo;
import com.biz.crm.nebular.mdm.role.req.MdmRoleSubButtonVo;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 按钮-角色关联接口实现
 *
 * @author zxw
 * @date 2020-11-18 11:51:48
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmButtonRoleServiceExpandImpl")
public class MdmButtonRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmButtonRoleMapper, MdmButtonRoleEntity> implements MdmButtonRoleService {

    @Resource
    private MdmButtonRoleMapper mdmButtonRoleMapper;

    /**
     * 列表
     * @param reqVo vo
     * @return page
     */
    @Override
    public PageResult<MdmButtonRoleRespVo> findList(MdmButtonRoleReqVo reqVo) {
        Page<MdmButtonRoleRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmButtonRoleRespVo> list = mdmButtonRoleMapper.findList(page, reqVo);
        return PageResult.<MdmButtonRoleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo vo
     * @return mdmButtonRoleRespVo
     */
    @Override
    public MdmButtonRoleRespVo query(MdmButtonRoleReqVo reqVo) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmButtonRoleReqVo reqVo) {
        MdmButtonRoleEntity entity = CrmBeanUtil.copy(reqVo, MdmButtonRoleEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmButtonRoleReqVo reqVo) {
        MdmButtonRoleEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param reqVo vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmButtonRoleReqVo reqVo) {

    }

    @Override
    public List<String> findButtonCodeByRole(List<String> roleCodeList, String parentCode, String functionCode) {
        Assert.notEmpty(roleCodeList, "角色编码不能为空");
        Assert.hasText(parentCode, "菜单编码不能为空");
        Assert.hasText(functionCode, "功能编码不能为空");
        return this.lambdaQuery()
                .in(MdmButtonRoleEntity::getRoleCode, roleCodeList)
                .eq(MdmButtonRoleEntity::getParentCode, parentCode)
                .eq(MdmButtonRoleEntity::getFunctionCode, functionCode)
                .select(MdmButtonRoleEntity::getButtonCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getButtonCode()))
                .map(MdmButtonRoleEntity::getButtonCode)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> findButtonCodeByRole(String roleCode, String parentCode, String functionCode) {
        Assert.hasText(roleCode, "角色编码不能为空");
        Assert.hasText(parentCode, "菜单编码不能为空");
        Assert.hasText(functionCode, "功能编码不能为空");
        return this.lambdaQuery()
                .eq(MdmButtonRoleEntity::getRoleCode, roleCode)
                .eq(MdmButtonRoleEntity::getParentCode, parentCode)
                .eq(MdmButtonRoleEntity::getFunctionCode, functionCode)
                .select(MdmButtonRoleEntity::getButtonCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getButtonCode()))
                .map(MdmButtonRoleEntity::getButtonCode)
                .collect(Collectors.toList());
    }

    @Override
    public void setUp(String roleCode, List<MdmRoleSubButtonVo> mdmRoleSubButtonVos) {
        Assert.hasText(roleCode, "角色编码不能为空");
        if (CollectionUtils.isEmpty(mdmRoleSubButtonVos)) {
            return;
        }
        List<String> menuCodeList = mdmRoleSubButtonVos.stream()
                .map(MdmRoleSubButtonVo::getParentCode)
                .filter(parentCode -> !StringUtils.isEmpty(parentCode))
                .distinct()
                .collect(Collectors.toList());
        List<String> functionCodeList = mdmRoleSubButtonVos.stream()
                .filter(x -> !StringUtils.isEmpty(x.getParentCode()))
                .map(MdmRoleSubButtonVo::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
        this.lambdaUpdate()
                .eq(MdmButtonRoleEntity::getRoleCode, roleCode)
                .in(!CollectionUtils.isEmpty(menuCodeList), MdmButtonRoleEntity::getParentCode, menuCodeList)
                .in(!CollectionUtils.isEmpty(functionCodeList), MdmButtonRoleEntity::getFunctionCode, functionCodeList)
                .remove();
        List<MdmRoleSubButtonVo> collect = mdmRoleSubButtonVos.stream()
                .filter(x -> !StringUtils.isEmpty(x.getButtonSelect()) && YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getButtonSelect()))
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(collect)) {
            Set<String> repeatCheckSet = new HashSet<>();
            for (MdmRoleSubButtonVo mdmRoleSubButtonVo : collect) {
                String parentCode = mdmRoleSubButtonVo.getParentCode();
                Assert.hasText(parentCode, "按钮:{"+mdmRoleSubButtonVo.getButtonName()+"}菜单编码不能为空");
                String functionCode = mdmRoleSubButtonVo.getFunctionCode();
                Assert.hasText(functionCode, "按钮:{"+mdmRoleSubButtonVo.getButtonName()+"}功能编码不能为空");
                String uniqueCode = roleCode + parentCode + functionCode + mdmRoleSubButtonVo.getButtonCode();
                Assert.isTrue(repeatCheckSet.add(uniqueCode), "同一菜单功能列表下，按钮重复:" + mdmRoleSubButtonVo.getButtonName());
            }
            List<MdmButtonRoleEntity> mdmButtonRoleEntities = collect.stream().map(x -> {
                MdmButtonRoleEntity mdmButtonRoleEntity = new MdmButtonRoleEntity();
                mdmButtonRoleEntity.setButtonCode(x.getButtonCode());
                mdmButtonRoleEntity.setFunctionCode(x.getFunctionCode());
                mdmButtonRoleEntity.setParentCode(x.getParentCode());
                mdmButtonRoleEntity.setRoleCode(roleCode);
                mdmButtonRoleEntity.setId(UUID.randomUUID().toString());
                return mdmButtonRoleEntity;
            }).collect(Collectors.toList());
            this.saveBatch(mdmButtonRoleEntities);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByRoleCodeList(List<String> roleCodeList) {
        if (CollectionUtils.isEmpty(roleCodeList)) {
            return;
        }
        this.lambdaUpdate()
                .in(MdmButtonRoleEntity::getRoleCode, roleCodeList)
                .remove();
    }


}
