package com.biz.crm.button.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.button.mapper.MdmButtonMapper;
import com.biz.crm.button.model.MdmButtonEntity;
import com.biz.crm.button.service.MdmButtonService;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.CacheManagerConstant;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.icon.service.IMdmIconService;
import com.biz.crm.nebular.mdm.button.MdmButtonReqVo;
import com.biz.crm.nebular.mdm.button.MdmButtonRespVo;
import com.biz.crm.nebular.mdm.icon.MdmIconRespVo;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.cache.TableConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 按钮接口实现
 *
 * @author zxw
 * @date 2020-11-18 11:33:08
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmButtonServiceExpandImpl")
@CacheConfig(cacheManager=CacheManagerConstant.CACHE_MANAGER,cacheNames = CacheManagerConstant.MDM_BUTTON)
public class MdmButtonServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmButtonMapper, MdmButtonEntity> implements MdmButtonService {

    @Resource
    private MdmButtonMapper mdmButtonMapper;

    @Resource
    private IMdmIconService mdmIconService;

    @Override
    public PageResult<MdmButtonRespVo> findList(MdmButtonReqVo reqVo) {
        Page<MdmButtonRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmButtonRespVo> list = mdmButtonMapper.findList(page, reqVo);
        return PageResult.<MdmButtonRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmButtonRespVo query(String id,String buttonCode) {
        if (StringUtils.isEmpty(id)&&StringUtils.isEmpty(buttonCode)) {
            return null;
        }
        MdmButtonEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(id), MdmButtonEntity::getId, id)
                .eq(!StringUtils.isEmpty(buttonCode), MdmButtonEntity::getButtonCode, buttonCode)
                .one();
        return CrmBeanUtil.copy(one, MdmButtonRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void save(MdmButtonReqVo reqVo) {
        valid(reqVo);
        validIcon(reqVo);
        Integer count = this.lambdaQuery()
                .eq(MdmButtonEntity::getButtonCode, reqVo.getButtonCode())
                .count();
        Assert.isTrue(count == null || count < 1, "按钮编码重复");
        this.save(CrmBeanUtil.copy(reqVo, MdmButtonEntity.class));
    }

    /**
     * 图标校验
     * @param reqVo vo
     */
    protected void validIcon(MdmButtonReqVo reqVo) {
        String iconCode = reqVo.getIconCode();
        if(!StringUtils.isEmpty(iconCode)){
            MdmIconRespVo mdmIconRespVo = mdmIconService.query(null, iconCode);
            Assert.notNull(mdmIconRespVo, "图标不存在");
        }
    }

    /**
     * 按钮校验
     * @param reqVo vo
     */
    protected void valid(MdmButtonReqVo reqVo) {
        String buttonCode = reqVo.getButtonCode();
        Assert.hasText(buttonCode, "按钮编码不能为空");
        String buttonName = reqVo.getButtonName();
        Assert.hasText(buttonName, "按钮名称不能为空");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void update(MdmButtonReqVo reqVo) {
        valid(reqVo);
        validIcon(reqVo);
        String id = reqVo.getId();
        Assert.hasText(id, "主键不能为空");
        Assert.notNull(this.getById(id), "按钮不存在");
        long count = this.lambdaQuery()
                .eq(MdmButtonEntity::getButtonCode, reqVo.getButtonCode())
                .select(MdmButtonEntity::getId)
                .list()
                .stream().filter(x -> !reqVo.getId().equals(x.getId()))
                .count();
        Assert.isTrue(count < 1, "按钮编码重复");
        this.updateById(CrmBeanUtil.copy(reqVo, MdmButtonEntity.class));
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmButtonEntity::getId, ids)
                .remove();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmButtonEntity::getId, ids)
                .set(MdmButtonEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmButtonEntity::getId, ids)
                .set(MdmButtonEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Cacheable(sync = true)
    public List<MdmButtonRespVo> listCondition(MdmButtonReqVo mdmButtonReqVo) {
        return mdmButtonMapper.listCondition(mdmButtonReqVo);
    }
}
