package com.biz.crm.coderule.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.coderule.mapper.MdmCodeRuleMapper;
import com.biz.crm.coderule.model.MdmCodeRuleEntity;
import com.biz.crm.coderule.service.MdmCodeRuleService;
import com.biz.crm.common.PageResult;
import com.biz.crm.nebular.mdm.coderule.req.MdmCodeRuleReqVo;
import com.biz.crm.nebular.mdm.coderule.resp.MdmCodeRuleRespVo;
import com.biz.crm.util.*;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.List;

/**
 * 编码规则接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-05 10:47:13
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmCodeRuleServiceExpandImpl")
public class MdmCodeRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmCodeRuleMapper, MdmCodeRuleEntity> implements MdmCodeRuleService {

    @Resource
    private MdmCodeRuleMapper mdmCodeRuleMapper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmCodeRuleRespVo> findList(MdmCodeRuleReqVo reqVo) {
        Page<MdmCodeRuleRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCodeRuleRespVo> list = mdmCodeRuleMapper.findList(page, reqVo);
        return PageResult.<MdmCodeRuleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return mdmCodeRuleRespVo
     */
    @Override
    public MdmCodeRuleRespVo query(MdmCodeRuleReqVo reqVo) {
        MdmCodeRuleRespVo respVo = new MdmCodeRuleRespVo();
        if (StringUtils.isNotEmpty(reqVo.getId()) || StringUtils.isNotEmpty(reqVo.getRuleCode())) {
            MdmCodeRuleEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(reqVo.getId()), MdmCodeRuleEntity::getId, reqVo.getId())
                    .eq(StringUtils.isNotEmpty(reqVo.getRuleCode()), MdmCodeRuleEntity::getRuleCode, reqVo.getRuleCode())
                    .one();
            if (one != null) {
                respVo = CrmBeanUtil.copy(one, MdmCodeRuleRespVo.class);
            }
        }
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmCodeRuleReqVo reqVo) {
        Assert.notNull(reqVo, "空参数");
        reqVo.setId(null);
        verify(reqVo);
        MdmCodeRuleEntity one = this.lambdaQuery().eq(MdmCodeRuleEntity::getRuleCode, reqVo.getRuleCode()).one();
        Assert.isTrue(one == null, "编码规则key已存在");
        MdmCodeRuleEntity entity = CrmBeanUtil.copy(reqVo, MdmCodeRuleEntity.class);
        if (entity.getCurrentValue() == null) {
            entity.setCurrentValue(entity.getInitialValue() - 1);
        }
        UserRedis user = UserUtils.getUser();
        entity.setCreateName(user != null ? user.getRealname() : "");
        entity.setCreateDate(DateUtil.dateNow2Str());
        entity.setCreateDateSecond(DateUtil.dateNowStrHms());
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmCodeRuleReqVo reqVo) {
        Assert.notNull(reqVo, "空参数");
        Assert.hasText(reqVo.getId(), "id不能为空");
        verify(reqVo);
        MdmCodeRuleEntity entity = this.getById(reqVo.getId());
        Assert.isTrue(entity != null, "无效的id");
        if (reqVo.getCurrentValue() != null) {
            Assert.isTrue(reqVo.getCurrentValue() >= entity.getCurrentValue(), "当前值不能改小");
        }
        Assert.isTrue(entity.getRuleCode().equals(reqVo.getRuleCode()), "编码规则key不能修改");
        entity.setPrefix(reqVo.getPrefix());
        entity.setDateFormat(reqVo.getDateFormat());
        entity.setCodeLength(reqVo.getCodeLength());
        entity.setInitialValue(reqVo.getInitialValue());
        entity.setCurrentValue(reqVo.getCurrentValue() != null ? reqVo.getCurrentValue() : entity.getCurrentValue());
        entity.setRuleDescription(reqVo.getRuleDescription());
        entity.setRuleModule(reqVo.getRuleModule());
        UserRedis user = UserUtils.getUser();
        entity.setUpdateName(user != null ? user.getRealname() : "");
        entity.setUpdateDate(DateUtil.dateNow2Str());
        entity.setUpdateDateSecond(DateUtil.dateNowStrHms());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        org.springframework.util.Assert.notEmpty(ids, "请选择一条数据");
        Assert.notEmpty(ids, "缺失参数");
        this.removeByIds(ids);
    }

    /**
     * 重置
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void reset(MdmCodeRuleReqVo reqVo) {
        Assert.notNull(reqVo, "空参数");
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "缺失参数");
        Assert.isTrue(ids.size() == 1, "该操作影响较大，暂不支持批量重置");
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmCodeRuleEntity one = this.getById(id);
            Assert.notNull(one, "无效的id");
            one.setCurrentValue(one.getInitialValue() - 1);
            this.updateById(one);
        });
    }

    /**
     * 新增编辑数据校验
     *
     * @param reqVo
     */
    private void verify(MdmCodeRuleReqVo reqVo) {
        Assert.hasText(reqVo.getRuleCode(), "编码规则key不能为空");
        Assert.notNull(reqVo.getCodeLength(), "长度不能为空");
        Assert.isTrue(reqVo.getCodeLength() <= 32, "长度不能超过32");
        Assert.notNull(reqVo.getInitialValue(), "起始值不能为空");
        Assert.isTrue(reqVo.getInitialValue() > 0, "起始值必须大于0");
        int lengthWithoutNumber = (StringUtils.isEmpty(reqVo.getPrefix()) ? 0 : reqVo.getPrefix().length()) + (StringUtils.isEmpty(reqVo.getDateFormat()) ? 0 : reqVo.getDateFormat().length());
        Assert.isTrue(reqVo.getCodeLength() > lengthWithoutNumber, "编码长度必须大于前缀和时间戳长度之和");
        if (StringUtils.isNotEmpty(reqVo.getDateFormat())) {
            try {
                SimpleDateFormat sdf = new SimpleDateFormat(reqVo.getDateFormat());
            } catch (Exception e) {
                throw new BusinessException("时间戳格式" + reqVo.getDateFormat() + "不合法");
            }
        }
    }
}
