package com.biz.crm.customer.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.customer.mapper.MdmCustomerROrgMapper;
import com.biz.crm.customer.model.MdmCustomerROrgEntity;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.org.model.MdmOrgEntity;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 客户与组织关联表接口实现
 *
 * @author zxw
 * @date 2020-12-21 09:50:12
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmCustomerROrgServiceExpandImpl")
public class MdmCustomerROrgServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmCustomerROrgMapper, MdmCustomerROrgEntity> implements MdmCustomerROrgService {

    @Resource
    private MdmOrgService mdmOrgService;

    @Override
    public Map<String, List<String>> findOrgCodeList(List<String> customerCodeList) {
        if (CollectionUtils.isEmpty(customerCodeList)) {
            return Collections.emptyMap();
        }
        return this.lambdaQuery()
                .in(MdmCustomerROrgEntity::getCustomerCode,customerCodeList)
                .select(MdmCustomerROrgEntity::getOrgCode, MdmCustomerROrgEntity::getCustomerCode)
                .list()
                .stream()
                .collect(Collectors.groupingBy(MdmCustomerROrgEntity::getCustomerCode
                        , Collectors.mapping(MdmCustomerROrgEntity::getOrgCode, Collectors.toList())));
    }

    @Override
    public Map<String, List<MdmOrgRespVo>> findOrgGroupByCustomerCodeList(List<String> customerCodeList) {
        if (CollectionUtil.listEmpty(customerCodeList)) {
            return Collections.emptyMap();
        }
        List<MdmCustomerROrgEntity> list = this.lambdaQuery()
                .in(MdmCustomerROrgEntity::getCustomerCode, customerCodeList)
                .select(MdmCustomerROrgEntity::getCustomerCode, MdmCustomerROrgEntity::getOrgCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Map<String, List<MdmOrgRespVo>> map = new HashMap<>(16);
            final Map<String, MdmOrgRespVo> orgMap = mdmOrgService.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, list.stream().map(MdmCustomerROrgEntity::getOrgCode).collect(Collectors.toSet()))
                    .list()
                    .stream().collect(Collectors.toMap(MdmOrgEntity::getOrgCode, v -> CrmBeanUtil.copy(v,MdmOrgRespVo.class)));
            Map<String, List<String>> groupMap = list.stream().collect(Collectors.groupingBy(MdmCustomerROrgEntity::getCustomerCode, Collectors.mapping(MdmCustomerROrgEntity::getOrgCode, Collectors.toList())));
            for (Map.Entry<String, List<String>> entry :
                    groupMap.entrySet()) {
                List<MdmOrgRespVo> orgList = new ArrayList<>();
                for (String item :
                        entry.getValue()) {
                    if (orgMap.containsKey(item)) {
                        orgList.add(orgMap.get(item));
                    }
                }
                map.put(entry.getKey(), orgList);
            }
            return map;
        }
        return Collections.emptyMap();
    }

    @Override
    public List<String> findOrgCodeByCustomerCode(String customerCode) {
        if(StringUtils.isEmpty(customerCode)){
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .eq(MdmCustomerROrgEntity::getCustomerCode, customerCode)
                .select(MdmCustomerROrgEntity::getOrgCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getOrgCode()))
                .map(MdmCustomerROrgEntity::getOrgCode)
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public List<String> findCustomerByOrgCodeList(List<String> orgCodeList) {
        if(CollectionUtils.isEmpty(orgCodeList)){
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmCustomerROrgEntity::getOrgCode,orgCodeList)
                .select(MdmCustomerROrgEntity::getCustomerCode)
                .list()
                .stream()
                .filter(x -> !StringUtils.isEmpty(x))
                .map(MdmCustomerROrgEntity::getCustomerCode)
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public List<MdmCustomerROrgEntity> findByOrgCodeList(List<String> orgCodeList) {
        if(CollectionUtils.isEmpty(orgCodeList)){
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmCustomerROrgEntity::getOrgCode,orgCodeList)
                .list();
    }

    @Override
    public void setUpOrg(MdmCustomerMsgReqVo mdmCustomerMsgReqVo) {
        String customerCode = mdmCustomerMsgReqVo.getCustomerCode();
        Assert.hasText(customerCode, "客户编码不能为空");
        this.lambdaUpdate()
                .eq(MdmCustomerROrgEntity::getCustomerCode, customerCode)
                .remove();
        String orgCode = mdmCustomerMsgReqVo.getOrgCode();
        if (!StringUtils.isEmpty(orgCode)) {
            List<String> orgCodeList = Arrays.asList(orgCode.split(","));
            if(!CollectionUtils.isEmpty(orgCodeList)){
                List<MdmCustomerROrgEntity> entityList = orgCodeList.stream().map(x -> {
                    MdmCustomerROrgEntity mdmCustomerROrgEntity = new MdmCustomerROrgEntity();
                    mdmCustomerROrgEntity.setOrgCode(x);
                    mdmCustomerROrgEntity.setCustomerCode(customerCode);
                    return mdmCustomerROrgEntity;
                }).collect(Collectors.toList());
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    public void setUpOrg(List<String> customerCodeList, String orgCode) {
        if(CollectionUtils.isEmpty(customerCodeList)){
            return;
        }
        this.lambdaUpdate()
                .in(MdmCustomerROrgEntity::getCustomerCode, customerCodeList)
                .remove();
        if(StringUtils.isEmpty(orgCode)){
            return;
        }
        List<String> orgCodeList = Arrays.asList(orgCode.split(","));
        List<MdmCustomerROrgEntity> list = new ArrayList<>();
        for (String customerCode : customerCodeList) {
            for (String currentOrgCode : orgCodeList) {
                MdmCustomerROrgEntity mdmCustomerROrgEntity = new MdmCustomerROrgEntity();
                mdmCustomerROrgEntity.setCustomerCode(customerCode);
                mdmCustomerROrgEntity.setOrgCode(currentOrgCode);
                list.add(mdmCustomerROrgEntity);
            }
        }
        if(!CollectionUtils.isEmpty(list)){
            this.saveBatch(list);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceOrg(List<String> customerCodeList, String originOrgCode, String orgCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList) && !StringUtils.isEmpty(originOrgCode) && !StringUtils.isEmpty(orgCode)) {
            this.lambdaUpdate()
                    .in(MdmCustomerROrgEntity::getCustomerCode, customerCodeList)
                    .eq(MdmCustomerROrgEntity::getOrgCode, originOrgCode)
                    .remove();
            Set<String> collect = this.lambdaQuery()
                    .in(MdmCustomerROrgEntity::getCustomerCode, customerCodeList)
                    .eq(MdmCustomerROrgEntity::getOrgCode, orgCode)
                    .select(MdmCustomerROrgEntity::getCustomerCode, MdmCustomerROrgEntity::getOrgCode)
                    .list()
                    .stream().map(MdmCustomerROrgEntity::getCustomerCode).collect(Collectors.toSet());

            List<MdmCustomerROrgEntity> entityList = customerCodeList.stream().filter(item -> !collect.contains(item)).map(item -> {
                MdmCustomerROrgEntity entity = new MdmCustomerROrgEntity();
                entity.setCustomerCode(item);
                entity.setOrgCode(orgCode);
                return entity;
            }).collect(Collectors.toList());
            if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                this.saveBatch(entityList);
            }
        }
    }
}
