package com.biz.crm.customer.service.impl;


import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.customer.mapper.MdmCustomerSupplyMapper;
import com.biz.crm.customer.model.MdmCustomerSupplyDetailEntity;
import com.biz.crm.customer.service.IMdmCustomerSupplyDetailService;
import com.biz.crm.customer.service.MdmCustomerSupplyService;
import com.biz.crm.nebular.mdm.customer.*;
import com.biz.crm.position.model.MdmCustomerSupplyEntity;
import com.biz.crm.position.model.MdmPositionEntity;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.bizunited.platform.common.vo.UuidVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位与用户关联表接口实现
 *
 * @author zxw
 * @date 2020-10-10 13:59:04
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionCustomerServiceExpandImpl")
public class MdmCustomerSupplyServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmCustomerSupplyMapper, MdmCustomerSupplyEntity> implements MdmCustomerSupplyService {

    @Autowired
    private MdmCustomerSupplyMapper mdmCustomerSupplyMapper;

    @Autowired
    private IMdmCustomerSupplyDetailService mdmCustomerSupplyDetailService;

    @Autowired
    private MdmPositionService mdmPositionService;

    @Override
    public List<String> findCustomerCodeByPosition(List<String> positionCodeList) {
        if (CollectionUtils.isEmpty(positionCodeList)) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmCustomerSupplyEntity::getPositionCode, positionCodeList)
                .select(MdmCustomerSupplyEntity::getCustomerCode)
                .list()
                .stream()
                .filter(x -> !StringUtils.isEmpty(x.getCustomerCode()))
                .map(MdmCustomerSupplyEntity::getCustomerCode)
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addRelation(String customerCode, String positionCode) {
        if (StringUtils.isNotEmpty(customerCode) && StringUtils.isNotEmpty(positionCode)) {
            List<MdmCustomerSupplyEntity> list = this.lambdaQuery()
                    .eq(MdmCustomerSupplyEntity::getCustomerCode, customerCode)
                    .eq(MdmCustomerSupplyEntity::getPositionCode, positionCode)
                    .list();
            if (CollectionUtil.listEmpty(list)) {
                MdmCustomerSupplyEntity entity = new MdmCustomerSupplyEntity();
                entity.setCustomerCode(customerCode);
                entity.setPositionCode(positionCode);
                this.save(entity);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addRelation(String customerCode, String positionCode, String orgCode) {
        MdmCustomerSupplyEntity entity = new MdmCustomerSupplyEntity();
        entity.setCustomerCode(customerCode);
        entity.setPositionCode(positionCode);
        entity.setOrgCode(orgCode);
        this.save(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replacePosition(List<String> ids, String positionCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) && StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity one = mdmPositionService.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, positionCode)
                    .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getOrgCode)
                    .one();
            if (one != null) {
                this.replacePosition(ids, positionCode, one.getOrgCode());
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replacePosition(List<String> ids, String positionCode, String orgCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) && StringUtils.isNotEmpty(positionCode)) {
            this.lambdaUpdate()
                    .in(MdmCustomerSupplyEntity::getId, ids)
                    .set(MdmCustomerSupplyEntity::getPositionCode, positionCode)
                    .set(MdmCustomerSupplyEntity::getOrgCode, orgCode)
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeRelations(List<String> ids) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            this.removeByIds(ids);
            mdmCustomerSupplyDetailService.lambdaUpdate()
                    .in(MdmCustomerSupplyDetailEntity::getSupplyId, ids)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByCustomer(String customerCode) {
        if (StringUtils.isNotEmpty(customerCode)) {
            this.removeByCustomers(Collections.singletonList(customerCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByCustomers(List<String> customerCodes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodes)) {
            List<MdmCustomerSupplyEntity> list = this.lambdaQuery()
                    .in(MdmCustomerSupplyEntity::getCustomerCode, customerCodes)
                    .select(MdmCustomerSupplyEntity::getId)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeRelations(list.stream().map(MdmCustomerSupplyEntity::getId).collect(Collectors.toList()));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPosition(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            this.removeByPositions(Collections.singletonList(positionCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPositions(List<String> positionCodes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodes)) {
            List<MdmCustomerSupplyEntity> list = this.lambdaQuery()
                    .in(MdmCustomerSupplyEntity::getPositionCode, positionCodes)
                    .select(MdmCustomerSupplyEntity::getId)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeRelations(list.stream().map(MdmCustomerSupplyEntity::getId).collect(Collectors.toList()));
            }
        }
    }

    @Override
    public void setUpDocking(MdmCustomerMsgReqVo mdmCustomerMsgReqVo) {
        String customerCode = mdmCustomerMsgReqVo.getCustomerCode();
        Assert.hasText(customerCode, "客户编码不能为空");
        //删除对接人及供货关系
        removeByCustomerCodeList(Collections.singletonList(customerCode));

        if (CollectionUtil.listEmpty(mdmCustomerMsgReqVo.getMdmCustomerDockingVos())) {
            return;
        }
        //过滤职位编码为空的数据
        List<MdmCustomerDockingReqVo> mdmCustomerDockingVos = mdmCustomerMsgReqVo.getMdmCustomerDockingVos().stream().filter(item -> StringUtils.isNotEmpty(item.getPositionCode())).collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(mdmCustomerDockingVos)) {
            List<MdmCustomerSupplyDetailReqVo> detailReqVos = new ArrayList<>();
            for (MdmCustomerDockingReqVo mdmCustomerDockingVo : mdmCustomerDockingVos) {
                mdmCustomerDockingVo.setCustomerCode(customerCode);
                String id = UUID.randomUUID().toString();
                mdmCustomerDockingVo.setId(id);
                List<MdmCustomerSupplyDetailReqVo> mdmCustomerSupplyDetailVos = mdmCustomerDockingVo.getMdmCustomerSupplyDetailVos();
                if (!CollectionUtils.isEmpty(mdmCustomerSupplyDetailVos)) {
                    for (MdmCustomerSupplyDetailReqVo mdmCustomerSupplyDetailVo : mdmCustomerSupplyDetailVos) {
                        mdmCustomerSupplyDetailVo.setId(null);
                        mdmCustomerSupplyDetailVo.setSupplyId(id);
                    }
                    detailReqVos.addAll(mdmCustomerSupplyDetailVos);
                }
            }
            //客户供货关系保存
            if (!CollectionUtils.isEmpty(detailReqVos)) {
                List<MdmCustomerSupplyDetailEntity> detailEntities = CrmBeanUtil.copyList(detailReqVos, MdmCustomerSupplyDetailEntity.class);
                mdmCustomerSupplyDetailService.saveBatch(detailEntities);
            }
            //客户对接人保存
            List<MdmCustomerSupplyEntity> mdmPositionCustomerEntities = CrmBeanUtil.copyList(mdmCustomerDockingVos, MdmCustomerSupplyEntity.class);
            this.saveBatch(mdmPositionCustomerEntities);
        }
    }

    @Override
    public void removeByCustomerCodeList(List<String> customerCodeList) {
        this.removeByCustomers(customerCodeList);
    }

    @Override
    public List<MdmCustomerDockingRespVo> findDockingListByCustomerCode(MdmCustomerDockingReqVo mdmCustomerDockingReqVo) {
        List<MdmCustomerDockingRespVo> list = mdmCustomerSupplyMapper.findDockingListByCustomerCode(mdmCustomerDockingReqVo);
        if (!CollectionUtils.isEmpty(list)) {
            List<String> supplyIdList = list.stream()
                    .filter(x -> x != null && !StringUtils.isEmpty(x.getId()))
                    .map(UuidVo::getId)
                    .collect(Collectors.toList());
            List<MdmCustomerSupplyDetailEntity> detailEntities = mdmCustomerSupplyDetailService.lambdaQuery()
                    .in(MdmCustomerSupplyDetailEntity::getSupplyId, supplyIdList)
                    .list();
            if (!CollectionUtils.isEmpty(detailEntities)) {
                List<MdmCustomerSupplyDetailRespVo> mdmCustomerSupplyDetailRespVos = CrmBeanUtil.copyList(detailEntities, MdmCustomerSupplyDetailRespVo.class);
                Map<String, List<MdmCustomerSupplyDetailRespVo>> detailMap = mdmCustomerSupplyDetailRespVos
                        .stream()
                        .collect(Collectors.groupingBy(MdmCustomerSupplyDetailRespVo::getSupplyId));
                for (MdmCustomerDockingRespVo mdmCustomerDockingRespVo : list) {
                    mdmCustomerDockingRespVo.setMdmCustomerSupplyDetailVos(detailMap.get(mdmCustomerDockingRespVo.getId()));
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmCustomerDockingRespVo> findCustomerSupplyByCustomerCodeList(List<String> customerCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
            List<MdmCustomerSupplyEntity> list = this.lambdaQuery()
                    .in(MdmCustomerSupplyEntity::getCustomerCode, customerCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmCustomerDockingRespVo.class);
            }
        }
        return new ArrayList<>();
    }

}
