package com.biz.crm.customer.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.customer.mapper.MdmCustomerTagMapper;
import com.biz.crm.customer.model.MdmCustomerTagEntity;
import com.biz.crm.customer.service.MdmCustomerTagService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.customer.MdmCustomerTagRefreshVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerTagReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerTagRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.assertj.core.util.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 客户标签实现
 *
 * @author yangshan
 * @date 2021/3/4 11:57
 */
@Slf4j
@Service
public class MdmCustomerTagServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmCustomerTagMapper, MdmCustomerTagEntity> implements MdmCustomerTagService {

    @Autowired
    private MdmCustomerTagMapper mdmCustomerTagMapper;
    @Autowired
    private RocketMQProducer rocketMQProducer;

    @Override
    public List<MdmCustomerTagRespVo> findList(MdmCustomerTagReqVo reqVo) {
        List<MdmCustomerTagRespVo> list = mdmCustomerTagMapper.findList(reqVo);
        return list;
    }

    @Override
    @Transactional
    public void save(MdmCustomerTagReqVo reqVo) {
        this.doCheck(reqVo);
        List<MdmCustomerTagEntity> list = this.lambdaQuery().in(MdmCustomerTagEntity::getClientCode, Lists.newArrayList(reqVo.getClientCode())).list();
        MdmCustomerTagEntity entity = CrmBeanUtil.copy(reqVo, MdmCustomerTagEntity.class);
        this.save(entity);
        if (CollectionUtils.isEmpty(list)) {
            list = Lists.newArrayList();
        }
        list.add(entity);
        MdmCustomerTagRefreshVo refreshVo = new MdmCustomerTagRefreshVo();
        refreshVo.setClientCode(reqVo.getClientCode());
        refreshVo.setTagList(CrmBeanUtil.copyList(list,MdmCustomerTagRespVo.class));
        //发送mq
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(refreshVo));
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.CUSTOMER_TAG_REFRESH);
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
    }

    @Override
    public void saveBatch(List<MdmCustomerTagReqVo> reqVos) {
        if(CollectionUtil.listNotEmpty(reqVos)) {
            List<MdmCustomerTagEntity> list = this.lambdaQuery().in(MdmCustomerTagEntity::getClientCode, Lists.newArrayList(reqVos.get(0).getClientCode())).list();
            List<MdmCustomerTagEntity> all = new ArrayList<>();
            if (CollectionUtils.isEmpty(list)) {
                list = Lists.newArrayList();
            }
            for (MdmCustomerTagReqVo reqVo : reqVos) {
                this.doCheck(reqVo);
                MdmCustomerTagEntity entity = CrmBeanUtil.copy(reqVo, MdmCustomerTagEntity.class);
                all.add(entity);
                list.add(entity);
            }
            this.saveBatch(all);
            MdmCustomerTagRefreshVo refreshVo = new MdmCustomerTagRefreshVo();
            refreshVo.setClientCode(reqVos.get(0).getClientCode());
            refreshVo.setTagList(CrmBeanUtil.copyList(list,MdmCustomerTagRespVo.class));
            //发送mq
            RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
            rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(refreshVo));
            rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.CUSTOMER_TAG_REFRESH);
            rocketMQProducer.convertAndSend(rocketMQMessageBody);
        }
    }

    @Override
    @Transactional
    public void delete(String id) {
        AssertUtils.isNotEmpty(id,"主键Id不能为空");
        MdmCustomerTagEntity entity = mdmCustomerTagMapper.selectById(id);
        AssertUtils.isNotNull(entity, "标签不存在");
        if(YesNoEnum.yesNoEnum.ZERO.getValue().equals(entity.getTagType())){
            throw new BusinessException("此标签不可删除");
        }
        List<MdmCustomerTagEntity> list = this.lambdaQuery().in(MdmCustomerTagEntity::getClientCode, Lists.newArrayList(entity.getClientCode())).list();
        if (CollectionUtils.isEmpty(list)) {
            list = Lists.newArrayList();
        } else {
            list = list.stream().filter(mdmCustomerTagEntity -> !id.equals(mdmCustomerTagEntity.getId())).collect(Collectors.toList());
        }
        this.removeById(id);
        MdmCustomerTagRefreshVo refreshVo = new MdmCustomerTagRefreshVo();
        refreshVo.setClientCode(entity.getClientCode());
        refreshVo.setTagList(CollectionUtils.isEmpty(list) ? null : CrmBeanUtil.copyList(list,MdmCustomerTagRespVo.class));
        //发送mq
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(refreshVo));
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.CUSTOMER_TAG_REFRESH);
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
    }

    public void doCheck(MdmCustomerTagReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getClientCode(), "客户编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getTagType(), "标签类型不能为空");
        AssertUtils.isNotEmpty(reqVo.getTagDescription(), "标签描述不能为空");
    }


    @Override
    public List<MdmCustomerTagRespVo> queryTagByCodeList(List<String> customerCodeList) {
        List<MdmCustomerTagRespVo> voList = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)){
            List<MdmCustomerTagEntity> entityList = this.lambdaQuery().in(MdmCustomerTagEntity::getClientCode, customerCodeList).list();
            if (CollectionUtil.listNotEmptyNotSizeZero(entityList)){
                voList = CrmBeanUtil.copyList(entityList, MdmCustomerTagRespVo.class);
            }
        }
        return voList;
    }
}
