package com.biz.crm.customermaterial.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customermaterial.mapper.MdmCustomerMaterialMapper;
import com.biz.crm.customermaterial.model.MdmCustomerMaterialEntity;
import com.biz.crm.customermaterial.service.IMdmCustomerMaterialService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.mdm.MdmCusMaterialException;
import com.biz.crm.nebular.mdm.customermaterial.req.MdmCustomerMaterialReqVo;
import com.biz.crm.nebular.mdm.customermaterial.resp.MdmCustomerMaterialRespVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 客户物料接口实现
 *
 * @author huang.long
 * @date 2020-09-03 10:26:02
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="MdmCustomerMaterialServiceExpandImpl")
public class MdmCustomerMaterialServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<MdmCustomerMaterialMapper, MdmCustomerMaterialEntity> implements IMdmCustomerMaterialService {

    @Resource
    private MdmCustomerMaterialMapper mdmCustomerMaterialMapper;
    @Autowired
    private MdmCustomerMaterialServiceHelper customerMaterialServiceHelper;
    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Override
    public PageResult<MdmCustomerMaterialRespVo> findList(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo){
        Page<MdmCustomerMaterialRespVo> page = PageUtil.buildPage(mdmCustomerMaterialReqVo.getPageNum(), mdmCustomerMaterialReqVo.getPageSize());
        List<MdmCustomerMaterialRespVo> list = mdmCustomerMaterialMapper.findList(page, mdmCustomerMaterialReqVo);
        return PageResult.<MdmCustomerMaterialRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmCustomerMaterialRespVo query(String id){
        MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo = new MdmCustomerMaterialReqVo();
        mdmCustomerMaterialReqVo.setId(id);
        return mdmCustomerMaterialMapper.query(mdmCustomerMaterialReqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo){
        //校验数据
        customerMaterialServiceHelper.saveCheck(mdmCustomerMaterialReqVo);
        MdmCustomerMaterialEntity entity = CrmBeanUtil.copy(mdmCustomerMaterialReqVo,MdmCustomerMaterialEntity.class);
        this.save(entity);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(((String) menuObject), entity.getId(),
                    mdmCustomerMaterialReqVo.getCustomerOrgName() + ":" + mdmCustomerMaterialReqVo.getMaterialName()
                    , CrmBeanUtil.copy(mdmCustomerMaterialReqVo, MdmCustomerMaterialRespVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo){
        if(StringUtils.isEmpty(mdmCustomerMaterialReqVo.getId())){
            throw new BusinessException("数据主键不能为空");
        }
        MdmCustomerMaterialEntity entity = this.getById(mdmCustomerMaterialReqVo.getId());
        if(Objects.isNull(entity)){
            throw new BusinessException(MdmCusMaterialException.DATA_NOT_EXIST);
        }
        MdmCustomerMaterialRespVo oldObject = CrmBeanUtil.copy(entity, MdmCustomerMaterialRespVo.class);
        //校验数据
        customerMaterialServiceHelper.saveCheck(mdmCustomerMaterialReqVo);
        CrmBeanUtil.copyProperties(mdmCustomerMaterialReqVo,entity);
        this.updateById(entity);

        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForUpdate(((String) menuObject), entity.getId(),
                    mdmCustomerMaterialReqVo.getCustomerOrgName() + ":" + mdmCustomerMaterialReqVo.getMaterialName()
                    , oldObject, CrmBeanUtil.copy(entity, MdmCustomerMaterialRespVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo){
        List<String> ids = mdmCustomerMaterialReqVo.getIds();
        Assert.notEmpty(ids,"id集合不能为空");
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            for (String id : ids) {
                MdmCustomerMaterialRespVo entity = query(id);
                if (entity != null) {
                    crmLogSendUtil.sendForDel(((String) menuObject), entity.getId(),
                            mdmCustomerMaterialReqVo.getCustomerOrgName() + ":" + mdmCustomerMaterialReqVo.getMaterialName()
                            , CrmBeanUtil.copy(entity, MdmCustomerMaterialRespVo.class));
                }
            }
        }
        this.lambdaUpdate()
                .in(MdmCustomerMaterialEntity::getId, ids)
                .remove();
    }

    private List<String> valid(MdmCustomerMaterialReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        io.jsonwebtoken.lang.Assert.notEmpty(ids,"id集合不能为空");
        String id = ids.get(0);
        io.jsonwebtoken.lang.Assert.hasText(id, "id不能为空");
        return ids;
    }

    /**
     * 启用
     * @param mdmCustomerMaterialReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo){
        List<String> ids = valid(mdmCustomerMaterialReqVo);
        List<MdmCustomerMaterialEntity> list = this.lambdaQuery()
                .in(MdmCustomerMaterialEntity::getId, ids)
                .select(MdmCustomerMaterialEntity::getId, MdmCustomerMaterialEntity::getEnableStatus)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            this.lambdaUpdate()
                    .in(MdmCustomerMaterialEntity::getId, list.stream().map(MdmCustomerMaterialEntity::getId).collect(Collectors.toSet()))
                    .set(MdmCustomerMaterialEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmCustomerMaterialReqVo mdmCustomerMaterialReqVo) {
        List<String> ids = valid(mdmCustomerMaterialReqVo);
        List<MdmCustomerMaterialEntity> list = this.lambdaQuery()
                .in(MdmCustomerMaterialEntity::getId, ids)
                .select(MdmCustomerMaterialEntity::getId)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.DISABLE.getCode().equals(x.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            this.lambdaUpdate()
                    .in(MdmCustomerMaterialEntity::getId, list.stream().map(MdmCustomerMaterialEntity::getId).collect(Collectors.toSet()))
                    .set(MdmCustomerMaterialEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                    .update();
        }
    }
}
