package com.biz.crm.dict.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dict.mapper.MdmDictAttrConfMapper;
import com.biz.crm.dict.model.MdmDictAttrConfEntity;
import com.biz.crm.dict.service.MdmDictAttrConfService;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.dict.service.MdmDictTypeService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.dict.req.MdmDictAttrConfReqVo;
import com.biz.crm.nebular.mdm.dict.resp.MdmDictAttrConfRespVo;
import com.biz.crm.nebular.mdm.dict.resp.MdmDictTypeRespVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典扩展字段配置表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-20 15:00:54
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmDictAttrConfServiceExpandImpl")
public class MdmDictAttrConfServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmDictAttrConfMapper, MdmDictAttrConfEntity> implements MdmDictAttrConfService {

    @Resource
    private MdmDictAttrConfMapper mdmDictAttrConfMapper;
    @Resource
    private MdmDictTypeService mdmDictTypeService;
    @Resource
    private MdmDictDataService mdmDictDataService;

    @Override
    public List<MdmDictAttrConfRespVo> findList(String dictTypeCode) {
        Assert.hasText(dictTypeCode, "字典类型编码不能为空");
        List<MdmDictAttrConfEntity> list = this.lambdaQuery()
                .eq(MdmDictAttrConfEntity::getDictTypeCode, dictTypeCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return list.stream().map(x -> {
                MdmDictAttrConfRespVo resp = CrmBeanUtil.copy(x, MdmDictAttrConfRespVo.class);
                if (StringUtils.isEmpty(resp.getSelectSingleOnly())) {
                    resp.setSelectSingleOnly(YesNoEnum.yesNoEnum.ZERO.getValue());
                }
                return resp;
            }).sorted((x1, x2) -> new BigDecimal(x1.getShowOrder()).compareTo(new BigDecimal(x2.getShowOrder()))).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public MdmDictAttrConfRespVo detail(String id) {
        if (StringUtils.isNotEmpty(id)) {
            MdmDictAttrConfEntity one = this.getById(id);
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmDictAttrConfRespVo.class);
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmDictAttrConfReqVo reqVo) {
        reqVo.setId(null);
        verify(reqVo);
        //分配一个扩展字段
        String extField = "";
        //总共有哪些扩展字段
        List<String> defaultExtFieldList = Arrays.asList("ext1", "ext2", "ext3", "ext4", "ext5", "ext6", "ext7", "ext8", "ext9", "ext10");

        //已经占用的
        Set<String> extFieldList = new HashSet<>(16);
        List<MdmDictAttrConfEntity> list = this.lambdaQuery()
                .eq(MdmDictAttrConfEntity::getDictTypeCode, reqVo.getDictTypeCode())
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            extFieldList.addAll(list.stream().map(MdmDictAttrConfEntity::getExtField).collect(Collectors.toSet()));
        }
        for (String item :
                defaultExtFieldList) {
            if (!extFieldList.contains(item)) {
                extField = item;
                break;
            }
        }
        Assert.hasText(extField, "扩展字段已经用尽，每个字典最多添加10个扩展字段");
        MdmDictAttrConfEntity entity = CrmBeanUtil.copy(reqVo, MdmDictAttrConfEntity.class);
        entity.setExtField(extField);
        mdmDictDataService.deleteExtFieldVal(entity.getDictTypeCode(), Collections.singletonList(extField));
        this.save(entity);
        DictUtil.deleteRedisCache(entity.getDictTypeCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmDictAttrConfReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "id不能为空");
        MdmDictAttrConfEntity entity = this.getById(reqVo.getId());
        Assert.notNull(entity, "无效的id");
        verify(reqVo);
        Assert.hasText(reqVo.getExtField(), "占用数据表的扩展字段不能为空");
        Assert.isTrue(entity.getExtField().equals(reqVo.getExtField()), "占用数据表的扩展字段不能修改");
        if ((reqVo.getSelectDictTypeCode() != null && !reqVo.getSelectDictTypeCode().equals(entity.getSelectDictTypeCode())) || (entity.getSelectDictTypeCode() != null && !entity.getSelectDictTypeCode().equals(reqVo.getSelectDictTypeCode()))) {
            mdmDictDataService.deleteExtFieldVal(entity.getDictTypeCode(), Collections.singletonList(entity.getExtField()));
        }
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        DictUtil.deleteRedisCache(entity.getDictTypeCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmDictAttrConfEntity> list = this.lambdaQuery().in(MdmDictAttrConfEntity::getId, ids).list();
        Map<String, List<MdmDictAttrConfEntity>> map = list.stream().collect(Collectors.groupingBy(MdmDictAttrConfEntity::getDictTypeCode));
        for (Map.Entry<String, List<MdmDictAttrConfEntity>> entry :
                map.entrySet()) {
            List<String> extFieldList = entry.getValue().stream().map(MdmDictAttrConfEntity::getExtField).collect(Collectors.toList());
            mdmDictDataService.deleteExtFieldVal(entry.getKey(), extFieldList);
            DictUtil.deleteRedisCache(entry.getKey());
        }
        this.removeByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByDictTypeCode(String dictTypeCode) {
        Assert.hasText(dictTypeCode, "字典类型编码不能为空");
        this.lambdaUpdate()
                .eq(MdmDictAttrConfEntity::getDictTypeCode, dictTypeCode)
                .remove();
    }

    /**
     * 校验
     *
     * @param reqVo
     */
    private void verify(MdmDictAttrConfReqVo reqVo) {
        Assert.hasText(reqVo.getDictTypeCode(), "缺失数据字典");
        Assert.hasText(reqVo.getFieldCode(), "字段编码不能为空");
        Assert.hasText(reqVo.getFieldName(), "字段名称不能为空");
        Assert.notNull(reqVo.getShowOrder(), "显示顺序不能为空");
        Assert.notNull(reqVo.getRequired(), "是否必填不能为空");
        if (StringUtils.isNotEmpty(reqVo.getSelectDictTypeCode())) {
            MdmDictTypeRespVo selectDict = mdmDictTypeService.query(null, reqVo.getSelectDictTypeCode());
            Assert.notNull(selectDict, "下拉框数据字典类型编码无效");
        }
        MdmDictTypeRespVo one = mdmDictTypeService.query(null, reqVo.getDictTypeCode());
        Assert.notNull(one, "未找到关联数据字典类型");
        List<MdmDictAttrConfEntity> list = this.lambdaQuery()
                .eq(MdmDictAttrConfEntity::getDictTypeCode, reqVo.getDictTypeCode())
                .list()
                .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> fieldList = list.stream().map(MdmDictAttrConfEntity::getFieldCode).collect(Collectors.toSet());
            Assert.isTrue(!fieldList.contains(reqVo.getFieldCode()), "当前数据字典下已经存在该字段：" + reqVo.getFieldCode());
        }
    }
}
