package com.biz.crm.dict.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.dict.mapper.MdmDictDataMapper;
import com.biz.crm.dict.model.MdmDictDataEntity;
import com.biz.crm.dict.service.MdmDictAttrConfService;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.dict.service.MdmDictTypeService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.kms.service.MdmUnitService;
import com.biz.crm.nebular.mdm.dict.req.MdmDictDataReqVo;
import com.biz.crm.nebular.mdm.dict.req.MdmDictDataSelectReqVo;
import com.biz.crm.nebular.mdm.dict.resp.*;
import com.biz.crm.util.*;
import java.util.Map.Entry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典明细表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-20 15:00:54
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmDictDataServiceExpandImpl")
public class MdmDictDataServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmDictDataMapper, MdmDictDataEntity> implements MdmDictDataService {

    @Resource
    private MdmDictDataMapper mdmDictDataMapper;
    @Resource
    private MdmDictTypeService mdmDictTypeService;
    @Resource
    private MdmDictAttrConfService mdmDictAttrConfService;
    @Resource
    private MdmUnitService mdmUnitService;

    @Override
    public List<MdmDictDataRespVo> treeList(MdmDictDataReqVo reqVo) {
        Assert.hasText(reqVo.getDictTypeCode(), "缺失字典类型编码");
        List<MdmDictDataRespVo> result = new ArrayList<>();
        List<MdmDictDataRespVo> list = mdmDictDataMapper.findList(null, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<MdmDictDataRespVo> allList = this.getAllParentList(reqVo.getDictTypeCode(), list);

            List<MdmDictAttrConfRespVo> confList = mdmDictAttrConfService.findList(reqVo.getDictTypeCode());
            Map<String, Map<String, String>> map = new HashMap<>(16);
            for (MdmDictAttrConfRespVo filed :
                    confList) {
                if (StringUtils.isNotEmpty(filed.getSelectDictTypeCode())) {
                    Map<String, String> dictMap = DictUtil.getDictValueMapsByCodes(filed.getSelectDictTypeCode());
                    if (!dictMap.isEmpty()) {
                        map.put(filed.getExtField(), dictMap);
                    }
                }
            }
            if (!map.isEmpty()) {
                allList.forEach(item -> {
                    map.forEach((k, v) -> {
                        Object fieldKy = ReflectUtil.getFiledValueByName(item, k);
                        if (fieldKy != null && StringUtils.isNotEmpty(fieldKy.toString())) {
                            ReflectUtil.setFieldValue(item, k, Arrays.asList(fieldKy.toString().split(",")).stream().map(x -> {
                                if (v.containsKey(x)) {
                                    return v.get(x);
                                }
                                return x;
                            }).collect(Collectors.joining(",")));
                        }
                    });
                });
            }
            result.addAll(generateTree(allList));
            this.setPath(reqVo.getDictTypeCode(), result);
        }
        return result;
    }

    @Override
    public List<DictDataVo> tree(MdmDictDataReqVo reqVo) {
        Assert.hasText(reqVo.getDictTypeCode(), "缺失字典类型编码");
        List<DictDataVo> tree = DictUtil.tree(reqVo.getDictTypeCode());
        if (CollectionUtil.listNotEmptyNotSizeZero(tree)) {
            return tree;
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmDictDataRespVo> queryByDictTypeCode(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            List<MdmDictDataEntity> list = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmDictDataRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, List<MdmDictDataSelectRespVo>> batchDictSelect(List<String> dictTypeCodeList) {
        Map<String, List<MdmDictDataSelectRespVo>> map = new HashMap<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(dictTypeCodeList)) {
            for (String dictTypeCode :
                    dictTypeCodeList) {
                List<DictDataVo> list = DictUtil.tree(dictTypeCode);
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    map.put(dictTypeCode, CrmBeanUtil.copyList(list, MdmDictDataSelectRespVo.class));
                }
            }
        }
        return map;
    }

    @Override
    public List<JSONObject> list(MdmDictDataSelectReqVo reqVo) {
        Assert.hasText(reqVo.getDictTypeCode(), "缺失字典类型编码");
        List<DictDataVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(reqVo.getParentDictCode())) {
            list.addAll(DictUtil.getChildrenList(reqVo.getDictTypeCode(), reqVo.getParentDictCode()));
        } else {
            list.addAll(DictUtil.tree(reqVo.getDictTypeCode()));
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return list.stream().filter(item -> {
                if (StringUtils.isNotEmpty(reqVo.getDictValue())) {
                    return item.getDictValue().contains(reqVo.getDictValue());
                }
                return true;
            }).map(item -> {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("dictCode", item.getDictCode());
                jsonObject.put("dictValue", item.getDictValue());
                Map<String, String> extendMap = item.getExtendMap();
                if (!extendMap.isEmpty()) {
                    for (Map.Entry<String, String> entry :
                            extendMap.entrySet()) {
                        jsonObject.put(entry.getKey(), entry.getValue());
                    }
                }
                return jsonObject;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public MdmDictDataRespVo query(String id) {
        if (StringUtils.isNotEmpty(id)) {
            MdmDictDataEntity entity = this.getById(id);
            if (entity != null) {
                MdmDictDataRespVo copy = CrmBeanUtil.copy(entity, MdmDictDataRespVo.class);
                if (StringUtils.isNotEmpty(copy.getParentDictCode())) {
                    //临时过渡
                    MdmDictDataEntity parent = this.lambdaQuery()
                            .eq(MdmDictDataEntity::getDictTypeCode, entity.getDictTypeCode())
                            .eq(MdmDictDataEntity::getDictCode, entity.getDictCode())
                            .select(MdmDictDataEntity::getId)
                            .one();
                    if (parent != null) {
                        copy.setParentId(parent.getId());
                    }
                }
                return copy;
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmDictDataReqVo reqVo) {
        reqVo.setId(null);
        Assert.hasText(reqVo.getDictCode(), "字典编码不能为空");
        Assert.isTrue(!reqVo.getDictCode().contains(DictUtil.PATH_SPLIT), "字典编码不能包含字符“" + DictUtil.PATH_SPLIT + "”");
        Assert.isTrue(!reqVo.getDictCode().contains(","), "字典编码不能包含字符“,”");
        Assert.hasText(reqVo.getDictValue(), "字典值不能为空");
        Assert.hasText(reqVo.getDictTypeCode(), "对应字典类型不能为空");
        MdmDictTypeRespVo dict = mdmDictTypeService.query(null, reqVo.getDictTypeCode());
        Assert.notNull(dict, "对应字典不存在");

        List<MdmDictDataEntity> uniqueList = this.lambdaQuery()
                .eq(MdmDictDataEntity::getDictTypeCode, reqVo.getDictTypeCode())
                .eq(MdmDictDataEntity::getDictCode, reqVo.getDictCode())
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(uniqueList), "字典编码已存在");

        if (StringUtils.isNotEmpty(reqVo.getParentDictCode())) {
            MdmDictDataEntity parent = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dict.getDictTypeCode())
                    .eq(MdmDictDataEntity::getDictCode, reqVo.getParentDictCode())
                    .one();
            Assert.notNull(parent, "上级字典不存在");
        } else {
            //临时过渡
            if (StringUtils.isNotEmpty(reqVo.getParentId())) {
                MdmDictDataEntity parent = this.getById(reqVo.getParentId());
                Assert.notNull(parent, "上级字典不存在");
                Assert.isTrue(parent.getDictTypeCode().equals(reqVo.getDictTypeCode()), "上级不属于当前字典");
                reqVo.setParentDictCode(parent.getDictCode());
            }
        }

        //校验扩展字段
        List<MdmDictAttrConfRespVo> confList = mdmDictAttrConfService.findList(reqVo.getDictTypeCode());
        if (CollectionUtil.listNotEmptyNotSizeZero(confList)) {
            for (MdmDictAttrConfRespVo conf :
                    confList) {
                if (YesNoEnum.yesNoEnum.ONE.getValue().equals(conf.getRequired())) {
                    Object filedValueByName = ReflectUtil.getFiledValueByName(reqVo, conf.getExtField());
                    if (filedValueByName == null || StringUtils.isEmpty(filedValueByName.toString())) {
                        throw new BusinessException(conf.getFieldName() + "是必填项");
                    }
                }
            }
        }
        reqVo.setId(UUID.randomUUID().toString());
        MdmDictDataEntity entity = CrmBeanUtil.copy(reqVo, MdmDictDataEntity.class);
        this.save(entity);
        DictUtil.deleteRedisCache(dict.getDictTypeCode());
        //同步kms
        mdmUnitService.add(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmDictDataReqVo reqVo) {

        Assert.hasText(reqVo.getId(), "缺失id");
        MdmDictDataEntity entity = this.getById(reqVo.getId());
        Assert.notNull(entity, "无效的id");
        Assert.hasText(reqVo.getDictCode(), "字典编码不能为空");
        Assert.hasText(reqVo.getDictValue(), "字典值不能为空");
        Assert.hasText(reqVo.getDictTypeCode(), "对应字典类型不能为空");
        Assert.isTrue(entity.getDictCode().equals(reqVo.getDictCode()), "字典编码不能修改");
        Assert.isTrue(entity.getDictTypeCode().equals(reqVo.getDictTypeCode()), "字典类型编码不能修改");

        List<MdmDictDataEntity> uniqueList = this.lambdaQuery()
                .eq(MdmDictDataEntity::getDictTypeCode, reqVo.getDictTypeCode())
                .eq(MdmDictDataEntity::getDictCode, reqVo.getDictCode())
                .select(MdmDictDataEntity::getId)
                .list()
                .stream().filter(x -> !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(uniqueList), "字典编码已存在");

        if (StringUtils.isNotEmpty(reqVo.getParentDictCode())) {
            MdmDictDataEntity parent = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, entity.getDictTypeCode())
                    .eq(MdmDictDataEntity::getDictCode, reqVo.getParentDictCode())
                    .one();
            Assert.notNull(parent, "上级字典不存在");
        } else {
            //临时过渡
            if (StringUtils.isNotEmpty(reqVo.getParentId())) {
                MdmDictDataEntity parent = this.getById(reqVo.getParentId());
                Assert.notNull(parent, "上级字典不存在");
                Assert.isTrue(parent.getDictTypeCode().equals(reqVo.getDictTypeCode()), "上级不属于当前字典");
                reqVo.setParentDictCode(parent.getDictCode());
            }
        }

        //校验扩展字段
        List<MdmDictAttrConfRespVo> confList = mdmDictAttrConfService.findList(reqVo.getDictTypeCode());
        if (CollectionUtil.listNotEmptyNotSizeZero(confList)) {
            for (MdmDictAttrConfRespVo conf :
                    confList) {
                if (YesNoEnum.yesNoEnum.ONE.getValue().equals(conf.getRequired())) {
                    Object filedValueByName = ReflectUtil.getFiledValueByName(reqVo, conf.getExtField());
                    if (filedValueByName == null || StringUtils.isEmpty(filedValueByName.toString())) {
                        throw new BusinessException(conf.getFieldName() + "是必填项");
                    }
                }
            }
        }
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        DictUtil.deleteRedisCache(reqVo.getDictTypeCode());
        //同步修改kms
        mdmUnitService.update(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "空参数");
        List<MdmDictDataEntity> list = this.lambdaQuery()
                .in(MdmDictDataEntity::getId, ids)
                .select(MdmDictDataEntity::getId, MdmDictDataEntity::getDictTypeCode, MdmDictDataEntity::getDictCode)
                .list();
        if (list != null) {
            List<String> dictCodeList = list.stream().map(MdmDictDataEntity::getDictCode).collect(Collectors.toList());
            List<MdmDictDataEntity> children = this.lambdaQuery()
                    .in(MdmDictDataEntity::getParentDictCode, dictCodeList)
                    .select(MdmDictDataEntity::getId)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(children)) {
                final Set<String> idsSet = new HashSet<>(ids);
                List<String> collect = children.stream().map(MdmDictDataEntity::getId).filter(idsSet::contains).collect(Collectors.toList());
                Assert.isTrue(CollectionUtil.listEmpty(collect), "字典存在下级不能删除");
            }
            list.stream().map(MdmDictDataEntity::getDictTypeCode).distinct().collect(Collectors.toList()).forEach(DictUtil::deleteRedisCache);
            this.removeByIds(ids);
            //同步删除kms
            mdmUnitService.remove(ids);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByDictTypeCode(String dictTypeCode) {
        Assert.hasText(dictTypeCode, "字典类型编码不能为空");
        this.lambdaUpdate()
                .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                .remove();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteExtFieldVal(String dictTypeCode, List<String> extList) {
        if (StringUtils.isNotEmpty(dictTypeCode) && CollectionUtil.listNotEmptyNotSizeZero(extList)) {
            List<MdmDictDataEntity> list = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                list.forEach(item -> {
                    for (String extField :
                            extList) {
                        ReflectUtil.setFieldValue(item, extField, "");
                    }
                });
                this.updateBatchById(list);
            }
        }
    }

    @Override
    public Map<String, String> getDictDataMap(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            return this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                    .select(MdmDictDataEntity::getDictCode, MdmDictDataEntity::getDictValue)
                    .list()
                    .stream().collect(Collectors.toMap(MdmDictDataEntity::getDictCode, MdmDictDataEntity::getDictValue));
        }
        return Collections.emptyMap();
    }

    @Override
    public Map<String, DictDataVo> getDictDataDetailMap(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            List<MdmDictDataEntity> dataEntityList = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(dataEntityList)) {
                final Map<String, MdmDictAttrConfRespVo> attrConfigMap = mdmDictAttrConfService.findList(dictTypeCode)
                        .stream()
                        .collect(Collectors.toMap(MdmDictAttrConfRespVo::getExtField, v -> v));
                Map<String, DictDataVo> collect = dataEntityList.stream().collect(Collectors.toMap(MdmDictDataEntity::getDictCode, v -> {
                    DictDataVo copy = CrmBeanUtil.copy(v, DictDataVo.class);
                    if (!attrConfigMap.isEmpty()) {
                        Map<String, String> map = new HashMap<>();
                        for (Entry<String, MdmDictAttrConfRespVo> entry :
                            attrConfigMap.entrySet()) {
                            Object obj = ReflectUtil.getFiledValueByName(v, entry.getKey());
                            map.put(entry.getValue().getFieldCode(), obj == null ? "" : (String) obj);
                        }
                        copy.setExtendMap(map);
                    }
                    return copy;
                }));
                return collect;
            }
        }
        return Collections.emptyMap();
    }

    /**
     * 获取全部上级
     *
     * @param dictTypeCode
     * @param list
     * @return
     */
    private List<MdmDictDataRespVo> getAllParentList(String dictTypeCode, List<MdmDictDataRespVo> list) {
        List<MdmDictDataRespVo> result = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> dictCodeSet = new HashSet<>();
            for (MdmDictDataRespVo item :
                    list) {
                dictCodeSet.add(item.getDictCode());
                setParent(dictTypeCode, item.getParentDictCode(), dictCodeSet);
            }
            if (!dictCodeSet.isEmpty()) {
                List<MdmDictDataEntity> parentList = this.lambdaQuery()
                        .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                        .in(MdmDictDataEntity::getDictCode, dictCodeSet)
                        .list();
                result.addAll(CrmBeanUtil.copyList(parentList, MdmDictDataRespVo.class));
            }
        }
        return result;
    }

    /**
     * 递归获取上级编码
     *
     * @param dictTypeCode
     * @param parentDictCode
     * @param dictCodeSet
     */
    private void setParent(String dictTypeCode, String parentDictCode, Set<String> dictCodeSet) {
        if (StringUtils.isNotEmpty(parentDictCode)) {
            if (dictCodeSet.contains(parentDictCode)) {
                return;
            }
            MdmDictDataEntity one = this.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, dictTypeCode)
                    .eq(MdmDictDataEntity::getDictCode, parentDictCode)
                    .select(MdmDictDataEntity::getDictCode, MdmDictDataEntity::getParentDictCode)
                    .one();
            if (one != null) {
                dictCodeSet.add(parentDictCode);
                setParent(dictTypeCode, one.getParentDictCode(), dictCodeSet);
            }
        }
    }

    /**
     * 构建树形结构
     *
     * @param totalList
     * @return
     */
    private List<MdmDictDataRespVo> generateTree(List<MdmDictDataRespVo> totalList) {

        totalList = totalList.stream().sorted(Comparator.comparing(MdmDictDataRespVo::getDictSort)).collect(Collectors.toList());

        //构建树list
        List<MdmDictDataRespVo> treeList = new ArrayList<>();
        //当前操作层级数据
        List<MdmDictDataRespVo> curLevelList = new ArrayList<>();
        //未操作数据
        List<MdmDictDataRespVo> restList = new ArrayList<>();

        //key:id
        Map<String, MdmDictDataRespVo> totalMap = totalList.stream().collect(Collectors.toMap(MdmDictDataRespVo::getDictCode, v -> v));

        //查找第一层
        for (MdmDictDataRespVo item : totalList) {
            if (StringUtils.isEmpty(item.getParentDictCode()) || !totalMap.containsKey(item.getParentDictCode())) {
                treeList.add(item);
                curLevelList.add(item);
            } else {
                restList.add(item);
            }
        }

        //构建数据，从第二层开始
        while (curLevelList.size() > 0 && restList.size() > 0) {
            List<MdmDictDataRespVo> restTempList = new ArrayList<>();
            List<MdmDictDataRespVo> curLevelTempList = new ArrayList<>();
            Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(MdmDictDataRespVo::getDictCode, MdmDictDataRespVo::getDictCode));
            Map<String, List<MdmDictDataRespVo>> curLevelChildrenMap = new LinkedHashMap<>(16);

            for (MdmDictDataRespVo item : restList) {
                if (curLevelMap.containsKey(item.getParentDictCode())) {
                    curLevelTempList.add(item);

                    List<MdmDictDataRespVo> childrenList = new ArrayList<>();
                    if (curLevelChildrenMap.containsKey(item.getParentDictCode())) {
                        childrenList.addAll(curLevelChildrenMap.get(item.getParentDictCode()));
                    }
                    childrenList.add(item);
                    curLevelChildrenMap.put(item.getParentDictCode(), childrenList);
                } else {
                    restTempList.add(item);
                }
            }

            for (MdmDictDataRespVo item : curLevelList) {
                if (curLevelChildrenMap.containsKey(item.getDictCode())) {
                    item.setChildren(curLevelChildrenMap.get(item.getDictCode()));
                }
            }

            curLevelList.clear();
            curLevelList.addAll(curLevelTempList);
            restList.clear();
            restList.addAll(restTempList);
        }

        return treeList;
    }

    private void setPath(String dictTypeCode, List<MdmDictDataRespVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            for (MdmDictDataRespVo item :
                    list) {
                item.setPath(dictTypeCode + DictUtil.PATH_SPLIT + item.getDictCode());
                if (CollectionUtil.listNotEmptyNotSizeZero(item.getChildren())) {
                    this.setPath(item.getPath(), item.getChildren());
                }
            }
        }
    }
}
