package com.biz.crm.function.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.function.mapper.MdmFunctionRoleMapper;
import com.biz.crm.function.model.MdmFunctionRoleEntity;
import com.biz.crm.function.service.MdmFunctionRoleService;
import com.biz.crm.nebular.mdm.functionrole.MdmFunctionRoleReqVo;
import com.biz.crm.nebular.mdm.functionrole.MdmFunctionRoleRespVo;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 职位级别-角色关联表接口实现
 *
 * @author zxw
 * @date 2020-11-18 10:42:36
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmFunctionRoleServiceExpandImpl")
public class MdmFunctionRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmFunctionRoleMapper, MdmFunctionRoleEntity> implements MdmFunctionRoleService {

    @Resource
    private MdmFunctionRoleMapper mdmFunctionRoleMapper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmFunctionRoleRespVo> findList(MdmFunctionRoleReqVo reqVo) {
        Page<MdmFunctionRoleRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmFunctionRoleRespVo> list = mdmFunctionRoleMapper.findList(page, reqVo);
        return PageResult.<MdmFunctionRoleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return mdmFunctionRoleRespVo
     */
    @Override
    public MdmFunctionRoleRespVo query(MdmFunctionRoleReqVo reqVo) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmFunctionRoleReqVo reqVo) {
        MdmFunctionRoleEntity entity = CrmBeanUtil.copy(reqVo, MdmFunctionRoleEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmFunctionRoleReqVo reqVo) {
        MdmFunctionRoleEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByRoleCodeList(List<String> roleCodeList) {
        if(CollectionUtils.isEmpty(roleCodeList)){
            return;
        }
        this.lambdaUpdate()
                .in(MdmFunctionRoleEntity::getRoleCode, roleCodeList)
                .remove();
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmFunctionRoleReqVo reqVo) {
        this.removeById(reqVo.getId());
    }

    @Override
    public List<String> findFunctionCodeByRole(String roleCode) {
        Assert.hasText(roleCode, "角色编码不能为空");
        return this.lambdaQuery()
                .eq(MdmFunctionRoleEntity::getRoleCode, roleCode)
                .list()
                .stream()
                .filter(x->x!=null&&!StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmFunctionRoleEntity::getFunctionCode)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> findFunctionCodeByRole(List<String> roleCodeList) {
        if(CollectionUtils.isEmpty(roleCodeList)){
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmFunctionRoleEntity::getRoleCode, roleCodeList)
                .list()
                .stream()
                .map(MdmFunctionRoleEntity::getFunctionCode)
                .filter(functionCode ->!StringUtils.isEmpty(functionCode))
                .collect(Collectors.toList());
    }

    @Override
    public List<String> findRoleCodeByFunction(String functionCode) {
        Assert.hasText(functionCode, "菜单编码不能为空");
        return this.lambdaQuery()
                .eq(MdmFunctionRoleEntity::getFunctionCode, functionCode)
                .list()
                .stream()
                .map(MdmFunctionRoleEntity::getRoleCode)
                .filter(roleCode ->!StringUtils.isEmpty(roleCode))
                .collect(Collectors.toList());
    }

    @Override
    public void update(String roleCode, List<String> functionCodeList) {
        Assert.hasText(roleCode, "角色编码不能为空");
        this.lambdaUpdate()
                .eq(MdmFunctionRoleEntity::getRoleCode, roleCode)
                .remove();
        if (CollectionUtils.isEmpty(functionCodeList)) {
            return;
        }
        List<MdmFunctionRoleEntity> collect = functionCodeList.stream().map(x -> {
            MdmFunctionRoleEntity mdmFunctionRoleEntity = new MdmFunctionRoleEntity();
            mdmFunctionRoleEntity.setFunctionCode(x);
            mdmFunctionRoleEntity.setRoleCode(roleCode);
            mdmFunctionRoleEntity.setId(UUID.randomUUID().toString());
            return mdmFunctionRoleEntity;
        }).collect(Collectors.toList());
        this.saveBatch(collect);
    }

    @Override
    public List<MdmFunctionRoleEntity> findFunctionCodeByPositionCode(String positionCode) {
        if(StringUtils.isEmpty(positionCode)){
            return new ArrayList<>();
        }
        return mdmFunctionRoleMapper.findFunctionCodeByPositionCode(positionCode);
    }

}
