package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.dict.model.MdmDictDataEntity;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.kms.manager.KmsConverter;
import com.biz.crm.kms.service.MdmProService;
import com.biz.crm.nebular.mdm.kms.api.KmsProInsertVo;
import com.biz.crm.nebular.mdm.kms.api.KmsProVo;
import com.biz.crm.nebular.mdm.kms.api.KmsResultVo;
import com.biz.crm.nebular.mdm.kms.api.KmsUnitVo;
import com.biz.crm.nebular.mdm.product.req.MdmProductReqVo;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author zxw
 * @date 2021-03-09 16:46
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmProServiceExpandImpl")
public class MdmProServiceImpl implements MdmProService {

    @Value("${jobId:3726962913e5877a3ed3a2eae59728c2}")
    private String jobId;

    @Value("${proBrand:f00b26660f74a2810dea8652534cfdad}")
    private String proBrand;

    @Autowired
    private KmsProperties kmsProperties;

    @Autowired
    private MdmDictDataService mdmDictDataService;

    @Resource
    private MdmProductService mdmProductService;


    @Override
    public void add(KmsProVo kmsProVo) {
        KmsProInsertVo kmsProInsertVo = new KmsProInsertVo();
        kmsProInsertVo.setMap(kmsProVo);
        kmsProInsertVo.setId(kmsProVo.getId());
        kmsProInsertVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProInsertVo.setUserId(kmsProperties.getUserId());
        kmsProInsertVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/KmsProDetailsController/addProDetails?access_token=" + kmsProperties.getAccessToken();

        ParameterizedTypeReference<KmsResultVo<Object>> reference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
        };
        Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProInsertVo, reference))
                .map(HttpEntity::getBody)
                .map(x -> {
                    log.info("[kms商品][新增]entity:{}", x);
                    return x;
                })
                .filter(x -> MapperConfig.ERROR.equals(x.getCode()))
                .ifPresent(x -> {
                    throw new BusinessException(MapperConfig.ERROR_MSG);
                });

    }

    @Override
    public void add(MdmProductReqVo mdmProductReqVo) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        KmsProVo kmsProVo = KmsConverter.INSTANCE.product2Kms(mdmProductReqVo);
        kmsProVo.setJob(jobId);
        kmsProVo.setProBrand(proBrand);
        kmsProVo.setUnit(convertUnit(mdmProductReqVo));
        add(kmsProVo);
    }

    protected List<KmsUnitVo> convertUnit(MdmProductReqVo mdmProductReqVo) {
        String saleUnit = mdmProductReqVo.getSaleUnit();
        if (StringUtils.isEmpty(saleUnit)) {
            return new ArrayList<>();
        }
        MdmDictDataEntity one = mdmDictDataService.lambdaQuery()
                .eq(MdmDictDataEntity::getDictTypeCode, "product_sale_unit")
                .eq(MdmDictDataEntity::getDictCode, saleUnit)
                .select(MdmDictDataEntity::getId)
                .one();
        if (one == null) {
            return new ArrayList<>();
        }
        KmsUnitVo kmsUnitVo = new KmsUnitVo();
        kmsUnitVo.setSupplementaryUnitId(one.getId());
        return Collections.singletonList(kmsUnitVo);
    }

    @Override
    public void update(KmsProVo kmsProVo) {

    }

    @Override
    public void update(MdmProductReqVo mdmProductReqVo) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        String url = kmsProperties.getUrl() + "/KmsProDetailsController/updateProDetails?access_token=" + kmsProperties.getAccessToken();
        KmsProVo kmsProVo = KmsConverter.INSTANCE.product2Kms(mdmProductReqVo);
        kmsProVo.setJob(jobId);
        kmsProVo.setProBrand(proBrand);
        kmsProVo.setUnit(convertUnit(mdmProductReqVo));

        KmsProInsertVo kmsProInsertVo = new KmsProInsertVo();
        kmsProInsertVo.setMap(kmsProVo);
        kmsProInsertVo.setId(kmsProVo.getId());
        kmsProInsertVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProInsertVo.setUserId(kmsProperties.getUserId());
        kmsProInsertVo.setUsername(kmsProperties.getUsername());

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsProInsertVo, String.class);
        log.info("[kms商品][update]entity:{}", entity);
    }

    @Override
    public void remove(List<String> productCodeList) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        if (CollectionUtils.isEmpty(productCodeList)) {
            return;
        }
        for (String id : productCodeList) {
            if (!StringUtils.isEmpty(id)) {
                Map<String, String> paramMap = new HashMap<>();
                paramMap.put("tenantryId", kmsProperties.getTenantryId());
                paramMap.put("userId", kmsProperties.getUserId());
                paramMap.put("username", kmsProperties.getUsername());
                paramMap.put("id", id);

                String url = kmsProperties.getUrl() + "/KmsProDetailsController/deleteById?access_token=" + kmsProperties.getAccessToken();

                ResponseEntity<String> enitty = RestTemplateUtils.postForEntity(url, null, paramMap, String.class);
                log.info("[kms商品][删除]entity:{}", enitty);
            }
        }
    }

    @Override
    public void sync() {
        try {
            mdmProductService.lambdaQuery()
                    .list()
                    .parallelStream()
                    .map(x -> CrmBeanUtil.copy(x, MdmProductReqVo.class))
                    .forEach(this::add);
        } catch (Exception e) {
            log.error("kms商品同步失败：{}", e);
        }
    }

    @Override
    public void add(List<MdmProductReqVo> list) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        if (!CollectionUtils.isEmpty(list)) {
            try {
                list.parallelStream().forEach(this::add);
            } catch (Exception e) {
                log.error("批量新增商品失败:{}", e);
            }
        }
    }

}
