package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.manager.KmsConverter;
import com.biz.crm.kms.service.MdmSellPartyService;
import com.biz.crm.nebular.mdm.kms.MdmSellPartyVo;
import com.biz.crm.nebular.mdm.kms.api.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-02-19 10:53
 **/
@Service
@ConditionalOnMissingBean(name = "MdmSellPartyServiceExpandImpl")
@Slf4j
public class MdmSellPartyServiceImpl implements MdmSellPartyService {

    @Autowired
    private KmsProperties kmsProperties;

    @Override
    public PageResult<MdmSellPartyVo> page(MdmSellPartyVo mdmSellPartyVo) {
        Integer current = Optional.ofNullable(mdmSellPartyVo.getPageNum()).orElse(1);
        Integer size = Optional.ofNullable(mdmSellPartyVo.getPageSize()).orElse(25);
        KmsPageVo kmsPageVo = new KmsPageVo(current, size);

        KmsSellPartyReqVo kmsSellPartyReqVo = new KmsSellPartyReqVo();
        String directSystemId = mdmSellPartyVo.getDirectSystemId();
        Assert.hasText(directSystemId, "直营体系id不能为空");
        kmsSellPartyReqVo.setDirectSystemId(directSystemId);
        kmsSellPartyReqVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyReqVo.setSearchValue(mdmSellPartyVo.getSellPartyName());

        KmsSellPartyPageReqVo kmsSellPartyPageReqVo = new KmsSellPartyPageReqVo();
        kmsSellPartyPageReqVo.setPage(kmsPageVo);
        kmsSellPartyPageReqVo.setReqVo(kmsSellPartyReqVo);

        kmsSellPartyPageReqVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyPageReqVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyPageReqVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/sell/party/findByPage";

        ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsSellPartyVo>>> reference = new ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsSellPartyVo>>>() {
        };

        ResponseEntity<KmsResultVo<KmsPageResultVo<KmsSellPartyVo>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsSellPartyPageReqVo, reference);
        KmsPageResultVo<KmsSellPartyVo> mdmSellPartyVoKmsPageResultVo = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if (mdmSellPartyVoKmsPageResultVo == null) {
            PageResult<MdmSellPartyVo> mdmSellPartyVoPageResult = new PageResult<>();
            mdmSellPartyVoPageResult.setCount(0L);
            mdmSellPartyVoPageResult.setData(new ArrayList<>());
            return mdmSellPartyVoPageResult;
        }
        PageResult<MdmSellPartyVo> mdmSellPartyVoPageResult = KmsConverter.INSTANCE.sellPartyPage(mdmSellPartyVoKmsPageResultVo);
        if (mdmSellPartyVoPageResult != null) {
            List<MdmSellPartyVo> data = mdmSellPartyVoPageResult.getData();
            if(!CollectionUtils.isEmpty(data)){
                data.forEach(x->{
                    x.setDirectSystemId(mdmSellPartyVo.getDirectSystemId());
                    x.setDirectSystemName(mdmSellPartyVo.getDirectSystemName());
                });
            }
        }
        return mdmSellPartyVoPageResult;
    }

    @Override
    public void save(MdmSellPartyVo mdmSellPartyVo) {
        Assert.hasText(mdmSellPartyVo.getDirectSystemId(), "直营体系id不能为空");
        Assert.hasText(mdmSellPartyVo.getSellPartyCode(), "售达方编码不能为空");
        Assert.hasText(mdmSellPartyVo.getSellPartyName(), "售达方名称不能为空");
        List<MdmSellPartyVo> list = list(mdmSellPartyVo);
        if(!CollectionUtils.isEmpty(list)){
            for (MdmSellPartyVo sellPartyVo : list) {
                if (mdmSellPartyVo.getSellPartyCode().equals(sellPartyVo.getSellPartyCode())) {
                    throw new BusinessException("当前售达方编码已存在，请重新输入");
                }
            }
        }
        KmsSellPartyVo kmsSellPartyVo = KmsConverter.INSTANCE.mdm2SellParty(mdmSellPartyVo);
        kmsSellPartyVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyVo.setUsername(kmsProperties.getUsername());
        kmsSellPartyVo.setId(null);

        String url = kmsProperties.getUrl() + "/sell/party/saveSellParty?access_token="+kmsProperties.getAccessToken();
        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsSellPartyVo, String.class);
        String body = entity.getBody();
        log.info("body:{}",body);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("新增失败");
        }
    }

    @Override
    public void update(MdmSellPartyVo mdmSellPartyVo) {
        Assert.hasText(mdmSellPartyVo.getId(), "id不能为空");
        Assert.hasText(mdmSellPartyVo.getDirectSystemId(), "直营体系id不能为空");
        Assert.hasText(mdmSellPartyVo.getSellPartyCode(), "售达方编码不能为空");
        Assert.hasText(mdmSellPartyVo.getSellPartyName(), "售达方名称不能为空");
        List<MdmSellPartyVo> list = list(mdmSellPartyVo);
        if(!CollectionUtils.isEmpty(list)){
            for (MdmSellPartyVo sellPartyVo : list) {
                if (mdmSellPartyVo.getSellPartyCode().equals(sellPartyVo.getSellPartyCode())) {
                    throw new BusinessException("售达方编码已存在，请重新输入");
                }
            }
        }
        KmsSellPartyVo kmsSellPartyVo = KmsConverter.INSTANCE.mdm2SellParty(mdmSellPartyVo);
        kmsSellPartyVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/sell/party/saveSellParty?access_token="+kmsProperties.getAccessToken();
        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsSellPartyVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("编辑失败");
        }
    }

    @Override
    public void delete(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        KmsSellPartyVo kmsSellPartyVo = new KmsSellPartyVo();
        kmsSellPartyVo.setIds(ids);
        kmsSellPartyVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/sell/party/deleteBatch?access_token="+kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsSellPartyVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("删除失败");
        }
    }

    @Override
    public void enable(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        KmsSellPartyVo kmsSellPartyVo = new KmsSellPartyVo();
        kmsSellPartyVo.setIds(ids);
        kmsSellPartyVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyVo.setUsername(kmsProperties.getUsername());
        kmsSellPartyVo.setFieldEn("status");
        kmsSellPartyVo.setStatus("ENABLE");

        String url = kmsProperties.getUrl() + "/sell/party/editBatch?access_token="+kmsProperties.getAccessToken();
        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsSellPartyVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("启用失败");
        }
    }

    @Override
    public void disable(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        KmsSellPartyVo kmsSellPartyVo = new KmsSellPartyVo();
        kmsSellPartyVo.setIds(ids);
        kmsSellPartyVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyVo.setUsername(kmsProperties.getUsername());
        kmsSellPartyVo.setFieldEn("status");
        kmsSellPartyVo.setStatus("DISABLE");

        String url = kmsProperties.getUrl() + "/sell/party/editBatch?access_token="+kmsProperties.getAccessToken();
        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsSellPartyVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("禁用失败");
        }
    }

    protected boolean valid() {
        return !StringUtils.isEmpty(kmsProperties.getAccessToken());
    }

    @Override
    public List<MdmSellPartyVo> list(MdmSellPartyVo mdmSellPartyVo) {
        if (!valid()) {
            return new ArrayList<>();
        }
        String directSystemId = mdmSellPartyVo.getDirectSystemId();
        if(StringUtils.isEmpty(directSystemId)){
            return new ArrayList<>();
        }
        KmsSellPartyPageReqVo kmsSellPartyPageReqVo = new KmsSellPartyPageReqVo();
        kmsSellPartyPageReqVo.setDirectSystemId(directSystemId);

        kmsSellPartyPageReqVo.setTenantryId(kmsProperties.getTenantryId());
        kmsSellPartyPageReqVo.setUserId(kmsProperties.getUserId());
        kmsSellPartyPageReqVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/sell/party/findByList";
        ParameterizedTypeReference<KmsResultVo<List<KmsSellPartyVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsSellPartyVo>>>() {
        };

        ResponseEntity<KmsResultVo<List<KmsSellPartyVo>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsSellPartyPageReqVo, reference);
        List<KmsSellPartyVo> kmsSellPartyVos = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if(CollectionUtils.isEmpty(kmsSellPartyVos)){
            return new ArrayList<>();
        }
        List<MdmSellPartyVo> mdmSellPartyVos = KmsConverter.INSTANCE.sellParty2Mdm(kmsSellPartyVos);
        String sellPartyName = mdmSellPartyVo.getSellPartyName();
        return mdmSellPartyVos.stream()
                .filter(x -> StringUtils.isEmpty(sellPartyName) || x.getSellPartyName().contains(sellPartyName))
                .collect(Collectors.toList());
    }

}
