package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.kms.service.MdmDirectSystemService;
import com.biz.crm.kms.service.MdmStoreService;
import com.biz.crm.nebular.mdm.kms.MdmDirectSystemVo;
import com.biz.crm.nebular.mdm.kms.api.KmsResultVo;
import com.biz.crm.nebular.mdm.kms.api.KmsStoreReplicaVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.util.CrmBeanUtil;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Optional;

/**
 * @author zxw
 * @date 2021-03-13 23:52
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmStoreServiceExpandImpl")
public class MdmStoreServiceImpl implements MdmStoreService {

    @Autowired
    private KmsProperties kmsProperties;

    @Autowired
    private MdmDirectSystemService mdmDirectSystemService;

    @Autowired
    private MdmTerminalService mdmTerminalService;

    @Override
    public void add(KmsStoreReplicaVo kmsStoreReplicaVo) {
        kmsStoreReplicaVo.setTenantryId(kmsProperties.getTenantryId());
        kmsStoreReplicaVo.setUserId(kmsProperties.getUserId());
        kmsStoreReplicaVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/storeController/onlySave?access_token=" + kmsProperties.getAccessToken();

        ParameterizedTypeReference<KmsResultVo<Object>> reference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
        };
        log.info("[kms门店][add]kmsStoreReplicaVo:{}", kmsStoreReplicaVo);
        ResponseEntity<KmsResultVo<Object>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsStoreReplicaVo, reference);
        log.info("[kms门店][add]entity:{}", entity);
        Optional.of(entity)
                .map(HttpEntity::getBody)
                .filter(x -> MapperConfig.ERROR.equals(x.getCode()))
                .ifPresent(x -> {
                    throw new BusinessException(x.getMessage());
                });
    }

    @Override
    public void add(MdmTerminalVo mdmTerminalVo) {
        if (!MapperConfig.validStore(kmsProperties)) {
            return;
        }
        KmsStoreReplicaVo check = check(mdmTerminalVo);
        if (check != null) {
            add(check);
        }
    }

    protected KmsStoreReplicaVo check(MdmTerminalVo mdmTerminalVo) {
        log.info("[kms门店][check]:{}", mdmTerminalVo);
        KmsStoreReplicaVo kmsStoreReplicaVo = new KmsStoreReplicaVo();
        kmsStoreReplicaVo.setId(mdmTerminalVo.getTerminalCode());
        kmsStoreReplicaVo.setStoreCode(mdmTerminalVo.getTerminalCode());
        kmsStoreReplicaVo.setStoreName(mdmTerminalVo.getTerminalName());
        kmsStoreReplicaVo.setDirectSystemId(mdmTerminalVo.getDirectSystemId());
        kmsStoreReplicaVo.setBsStoreId(mdmTerminalVo.getStoreId());

        if (StringUtils.isEmpty(mdmTerminalVo.getDirectSystemId()) || StringUtils.isEmpty(mdmTerminalVo.getStoreId())) {
            return null;
        }

        List<MdmDirectSystemVo> mdmDirectSystemVos = mdmDirectSystemService.directSystemSelect(new MdmDirectSystemVo());
        Assert.notEmpty(mdmDirectSystemVos, "直营体系不存在");
        Assert.isTrue(mdmDirectSystemVos.stream().anyMatch(x -> mdmTerminalVo.getDirectSystemId().equals(x.getId())), "直营体系不存在");
        for (MdmDirectSystemVo mdmDirectSystemVo : mdmDirectSystemVos) {
            if (mdmTerminalVo.getDirectSystemId().equals(mdmDirectSystemVo.getId())) {
                kmsStoreReplicaVo.setDirectSystemName(mdmDirectSystemVo.getDirectSystemName());
            }
        }
        return kmsStoreReplicaVo;
    }

    @Override
    public void update(MdmTerminalVo mdmTerminalVo) {
        if (!MapperConfig.validStore(kmsProperties)) {
            return;
        }
        KmsStoreReplicaVo check = check(mdmTerminalVo);
        if (check != null) {
            update(check);
        }
    }

    @Override
    public void remove(List<String> ids) {
        if (!MapperConfig.validStore(kmsProperties)) {
            return;
        }
        if (!CollectionUtils.isEmpty(ids)) {
            for (String id : ids) {
                if (!StringUtils.isEmpty(id)) {
                    ObjectNode instance = MapperConfig.instance(kmsProperties);
                    instance.put("storeId", id);
                    String url = kmsProperties.getUrl() + "/storeController/deleteStore?access_token=" + kmsProperties.getAccessToken();
                    ParameterizedTypeReference<KmsResultVo<Object>> reference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
                    };
                    Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, instance, reference))
                            .map(x -> {
                                log.info("[kms门店][remove]entity:{}", x);
                                return x;
                            })
                            .map(HttpEntity::getBody)
                            .filter(x -> MapperConfig.ERROR.equals(x.getCode()))
                            .ifPresent(x -> {
                                throw new BusinessException(x.getMessage());
                            });
                }
            }
        }
    }

    @Override
    public void update(KmsStoreReplicaVo kmsStoreReplicaVo) {
        kmsStoreReplicaVo.setTenantryId(kmsProperties.getTenantryId());
        kmsStoreReplicaVo.setUserId(kmsProperties.getUserId());
        kmsStoreReplicaVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/storeController/saveStore?access_token=" + kmsProperties.getAccessToken();

        ParameterizedTypeReference<KmsResultVo<Object>> reference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
        };
        ResponseEntity<KmsResultVo<Object>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsStoreReplicaVo, reference);
        log.info("[kms门店][add]entity:{}", entity);
        Optional.of(entity)
                .map(HttpEntity::getBody)
                .filter(x -> MapperConfig.ERROR.equals(x.getCode()))
                .ifPresent(x -> {
                    throw new BusinessException(x.getMessage());
                });
    }

    @Override
    public void sync() {
        try {
            mdmTerminalService.lambdaQuery()
                    .list()
                    .parallelStream()
                    .filter(x -> !StringUtils.isEmpty(x.getTerminalCode()))
                    .filter(x -> !StringUtils.isEmpty(x.getDirectSystemId()) && !StringUtils.isEmpty(x.getStoreId()))
                    .map(x -> CrmBeanUtil.copy(x, MdmTerminalVo.class))
                    .forEach(this::add);
        } catch (Exception e) {
            log.error("kms门店同步失败:{}", e);
        }
    }

    @Override
    public void add(List<MdmTerminalVo> list) {
        if (!MapperConfig.validStore(kmsProperties)) {
            return;
        }
        if (!CollectionUtils.isEmpty(list)) {
            try {
                list.parallelStream().forEach(this::add);
            } catch (Exception e) {
                log.error("批量同步kms门店失败:{}", e);
            }
        }
    }

}
