package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.dict.model.MdmDictDataEntity;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.kms.service.MdmUnitService;
import com.biz.crm.nebular.mdm.dict.req.MdmDictDataReqVo;
import com.biz.crm.nebular.mdm.kms.api.KmsResultVo;
import com.biz.crm.nebular.mdm.kms.api.KmsUnitVo;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @author zxw
 * @date 2021-03-10 10:05
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmUnitServiceExpandImpl")
public class MdmUnitServiceImpl implements MdmUnitService {

    @Value("${proUnitLevel:BASE}")
    private String proUnitLevel;

    @Autowired
    private KmsProperties kmsProperties;

    @Resource
    private MdmDictDataService mdmDictDataService;

    private static final String PRODUCT_SALE_UNIT = "product_sale_unit";

    @Override
    public void add(KmsUnitVo kmsUnitVo) {
        kmsUnitVo.setTenantryId(kmsProperties.getTenantryId());
        kmsUnitVo.setUserId(kmsProperties.getUserId());
        kmsUnitVo.setUsername(kmsProperties.getUsername());
        kmsUnitVo.setProUnitLevel(proUnitLevel);

        String url = kmsProperties.getUrl() + "/KmsProUnitController/addProUnit?access_token=" + kmsProperties.getAccessToken();

        ParameterizedTypeReference<KmsResultVo<Object>> reference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
        };
        Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsUnitVo, reference))
                .map(HttpEntity::getBody)
                .map(x -> {
                    log.info("[kms单位][add]entity:{}", x);
                    return x;
                })
                .filter(x -> MapperConfig.ERROR.equals(x.getCode()))
                .ifPresent(x -> {
                    throw new BusinessException("kms数据字典同步失败");
                });

    }

    @Override
    public void add(MdmDictDataReqVo mdmDictDataReqVo) {
        if (!valid(mdmDictDataReqVo)) {
            return;
        }
        String id = mdmDictDataReqVo.getId();
        Assert.hasText(id, "id不能为空");
        KmsUnitVo kmsUnitVo = new KmsUnitVo();
        kmsUnitVo.setId(id);
        kmsUnitVo.setProUnitName(mdmDictDataReqVo.getDictValue());
        add(kmsUnitVo);
    }

    protected boolean valid(MdmDictDataReqVo mdmDictDataReqVo) {
        if (!MapperConfig.valid(kmsProperties)) {
            return false;
        }
        String dictCode = mdmDictDataReqVo.getDictCode();
        if (StringUtils.isEmpty(dictCode) || !PRODUCT_SALE_UNIT.equals(mdmDictDataReqVo.getDictTypeCode())) {
            return false;
        }
        return true;
    }

    @Override
    public void update(MdmDictDataReqVo mdmDictDataReqVo) {
        if (!valid(mdmDictDataReqVo)) {
            return;
        }
        KmsUnitVo kmsUnitVo = new KmsUnitVo();
        kmsUnitVo.setId(mdmDictDataReqVo.getId());
        kmsUnitVo.setProUnitName(mdmDictDataReqVo.getDictValue());
        update(kmsUnitVo);
    }

    @Override
    public void update(KmsUnitVo kmsUnitVo) {
        kmsUnitVo.setTenantryId(kmsProperties.getTenantryId());
        kmsUnitVo.setUserId(kmsProperties.getUserId());
        kmsUnitVo.setUsername(kmsProperties.getUsername());
        kmsUnitVo.setProUnitLevel(proUnitLevel);

        String url = kmsProperties.getUrl() + "/KmsProUnitController/updateProUnit?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsUnitVo, String.class);
        log.info("[kms单位][update]entity:{}", entity);
    }

    @Override
    public void remove(List<String> ids) {
        if (StringUtils.isEmpty(kmsProperties.getAccessToken())) {
            return;
        }
        if (!CollectionUtils.isEmpty(ids)) {
            for (String id : ids) {
                if (!StringUtils.isEmpty(id)) {
                    Map<String, String> paramMap = new HashMap<>();
                    paramMap.put("tenantryId", kmsProperties.getTenantryId());
                    paramMap.put("userId", kmsProperties.getUserId());
                    paramMap.put("username", kmsProperties.getUsername());
                    paramMap.put("id", id);

                    String url = kmsProperties.getUrl() + "/KmsProUnitController/deleteById?access_token=" + kmsProperties.getAccessToken();

                    ResponseEntity<String> enitty = RestTemplateUtils.postForEntity(url, null, paramMap, String.class);
                    log.info("[kms单位][remove]entity:{}", enitty);
                }
            }
        }
    }

    @Override
    public void sync() {
        try {
            mdmDictDataService.lambdaQuery()
                    .eq(MdmDictDataEntity::getDictTypeCode, PRODUCT_SALE_UNIT)
                    .list()
                    .parallelStream()
                    .map(x -> CrmBeanUtil.copy(x, MdmDictDataReqVo.class))
                    .forEach(this::add);
        } catch (Exception e) {
            log.error("商品单位同步kms失败:{}", e);
        }
    }

}
