package com.biz.crm.login.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.login.service.MdmDmsMallLoginService;
import com.biz.crm.login.service.MdmLoginHelpService;
import com.biz.crm.login.service.MdmSystemLoginService;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.login.MdmLoginReqVo;
import com.biz.crm.nebular.mdm.login.MdmLoginRespVo;
import com.biz.crm.nebular.mdm.login.MdmLoginTypeControlVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserLoginLogRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.user.service.MdmUserLoginLogAsyncService;
import com.biz.crm.user.service.MdmUserLoginLogService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginHelpUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 系统登录业务层
 *
 * @author zxw
 * @date 2020-11-19 15:48
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmDmsMallLoginServiceExpandImpl")
public class MdmDmsMallLoginServiceImpl implements MdmDmsMallLoginService {

    @Autowired
    private MdmUserService mdmUserService;
    @Autowired
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Resource
    private MdmUserLoginLogService mdmUserLoginLogService;
    @Resource
    private MdmUserLoginLogAsyncService mdmUserLoginLogAsyncService;
    @Resource
    private MdmSystemLoginService mdmSystemLoginService;
    @Resource
    private MdmCustomerROrgService mdmCustomerROrgService;
    @Resource
    private MdmOrgService mdmOrgService;
    @Resource
    private MdmLoginHelpService mdmLoginHelpService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo login(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失账号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(reqVo.getUserName(), null, null);
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhone(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmail(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhoneVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //根据验证码+手机号获取当时的用户
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmailVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //根据验证码+邮箱获取当时的用户
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    public void sendVerificationForLoginByMessage(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送短信
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    public void sendVerificationForLoginByEmail(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送邮箱验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    public void checkPhoneExistAndUsable(MdmLoginReqVo reqVo) {
        mdmSystemLoginService.checkPhoneExistAndUsable(reqVo);
    }

    @Override
    public void checkEmailExistAndUsable(MdmLoginReqVo reqVo) {
        mdmSystemLoginService.checkEmailExistAndUsable(reqVo);
    }

    @Override
    public void sendVerificationForLoginAndResetByMessage(MdmLoginReqVo reqVo) {
        mdmSystemLoginService.sendVerificationForLoginAndResetByMessage(reqVo);
    }

    @Override
    public void sendVerificationForLoginAndResetByEmail(MdmLoginReqVo reqVo) {
        mdmSystemLoginService.sendVerificationForLoginAndResetByEmail(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByPhoneVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), false, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByEmailVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), false, true);
    }

    @Override
    public MdmLoginTypeControlVo getLoginTypeControlConfig() {
        return mdmSystemLoginService.getLoginTypeControlConfig();
    }

    /**
     * 执行客户用户登录
     *
     * @param user            用户信息
     * @param fromType        登录入口
     * @param getLastLoginLog 查询上一次登录日志
     * @param resetPassword   强制登录后修改密码
     * @return
     */
    protected MdmLoginRespVo doLoginThisSystem(MdmUserRespVo user, String fromType, boolean getLastLoginLog, boolean resetPassword) {

        MdmCustomerMsgRespVo customer = mdmCustomerMsgService.getUserCurrentCustomer(user.getUserName());
        if (customer == null) {
            throw new BusinessException("当前账号未关联可用的客户，请联系管理员");
        }
        Map<String, List<String>> orgCodeListMap = mdmCustomerROrgService.findOrgCodeList(Collections.singletonList(customer.getCustomerCode()));
        String orgCode = null;
        String orgName = null;
        if (orgCodeListMap.containsKey(customer.getCustomerCode())) {
            List<String> orgCodeList = orgCodeListMap.get(customer.getCustomerCode());
            Map<String, String> orgNameMap = mdmOrgService.findOrgNameMap(orgCodeList);
            orgName = orgCodeList.stream().map(orgNameMap::get).collect(Collectors.joining(","));
            orgCode = String.join(",", orgCodeList);
        }

        //生成登录信息
        String token = UUID.randomUUID().toString().replaceAll("-", "");
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(user.getUserName());
        userRedis.setRealname(user.getFullName());
        userRedis.setUsertype(user.getUserType());
        userRedis.setCustcode(customer.getCustomerCode());
        userRedis.setCustname(customer.getCustomerName());
        userRedis.setOrgcode(orgCode);
        userRedis.setOrgname(orgName);
        userRedis.setFromtype(fromType);
        UserUtils.setUser(token, userRedis);
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        HttpServletResponse response = HttpServletRequestUtil.getResponse();
        CookiesUtil.doCoke(request, response, token, GlobalParam.TOKEN);

        MdmLoginRespVo mdmLoginRespVo = new MdmLoginRespVo();
        mdmLoginRespVo.setUserCode(user.getUserCode());
        mdmLoginRespVo.setLoginUserToken(token);
        mdmLoginRespVo.setUserName(user.getUserName());
        mdmLoginRespVo.setFullName(user.getFullName());
        mdmLoginRespVo.setUserType(user.getUserType());
        mdmLoginRespVo.setCustomerCode(customer.getCustomerCode());
        mdmLoginRespVo.setCustomerName(customer.getCustomerName());
        mdmLoginRespVo.setOrgCode(orgCode);
        mdmLoginRespVo.setOrgName(orgName);
        mdmLoginRespVo.setLanguage(userRedis.getLanguage());
        mdmLoginRespVo.setMdmCustomerMsgRespVo(customer);

        if (getLastLoginLog) {
            MdmUserLoginLogRespVo lastLoginLog = mdmUserLoginLogService.getLastLoginLog(user.getUserName(), null);
            mdmLoginRespVo.setLastLoginLog(lastLoginLog);
        }

        if (resetPassword) {
            mdmUserService.setUserForceChangePassword(user.getUserName());
            mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
        } else {
            if (YesNoEnum.yesNoEnum.ONE.getValue().equals(user.getForceChangePassword())) {
                mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
            }
        }

        mdmUserLoginLogAsyncService.saveLoginLog(request, user.getUserName(), user.getFullName(), fromType, DateUtil.dateNowHms());

        return mdmLoginRespVo;
    }
}
