package com.biz.crm.login.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.login.service.MdmLoginHelpService;
import com.biz.crm.login.service.MdmSfaAppletLoginService;
import com.biz.crm.login.service.MdmSystemLoginService;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.login.MdmAppletLoginReqVo;
import com.biz.crm.nebular.mdm.login.MdmLoginRespVo;
import com.biz.crm.nebular.mdm.login.MdmLoginTypeControlVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserRelWeChatReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserLoginLogRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRelWeChatRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.user.service.MdmUserLoginLogAsyncService;
import com.biz.crm.user.service.MdmUserLoginLogService;
import com.biz.crm.user.service.MdmUserRelWeChatService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginHelpUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 系统登录业务层
 *
 * @author zxw
 * @date 2020-11-19 15:48
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmSfaAppletLoginServiceExpandImpl")
public class MdmSfaAppletLoginServiceImpl implements MdmSfaAppletLoginService {

    @Autowired
    private MdmUserRelWeChatService mdmUserRelWeChatService;
    @Resource
    private MdmUserService mdmUserService;
    @Resource
    private MdmPositionService mdmPositionService;
    @Resource
    private MdmUserLoginLogService mdmUserLoginLogService;
    @Resource
    private MdmUserLoginLogAsyncService mdmUserLoginLogAsyncService;
    @Resource
    private MdmSystemLoginService mdmSystemLoginService;
    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Resource
    private MdmCustomerROrgService mdmCustomerROrgService;
    @Resource
    private MdmOrgService mdmOrgService;
    @Resource
    private MdmLoginHelpService mdmLoginHelpService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo login(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失账号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(reqVo.getUserName(), null, null);
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo, true, false, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhone(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo, true, false, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmail(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        } else {
            mdmLoginHelpService.addError(user.getUserName(), user.getUserPhone(), user.getEmail());
        }
        return doLoginThisSystem(user, reqVo, true, false, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhoneVerification(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //根据验证码+手机号获取当时的用户
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo, true, false, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmailVerification(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //根据验证码+邮箱获取当时的用户
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo, true, false, true);
    }

    @Override
    public void sendVerificationForLoginByMessage(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送短信
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    public void sendVerificationForLoginByEmail(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送邮箱验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    public void checkPhoneExistAndUsable(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
    }

    @Override
    public void checkEmailExistAndUsable(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
    }

    @Override
    public void sendVerificationForLoginAndResetByMessage(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        //发送短信验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
    }

    @Override
    public void sendVerificationForLoginAndResetByEmail(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        //发送邮箱验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByPhoneVerification(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo, false, true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByEmailVerification(MdmAppletLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo, false, true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByOpenId(MdmAppletLoginReqVo reqVo) {
        String openId = reqVo.getOpenId();
        Assert.hasText(openId, "缺失openId");
        List<MdmUserRelWeChatRespVo> list = mdmUserRelWeChatService.getBindUserListByOpenId(openId, LoginFromTypeEnum.APPLET_SFA.getValue());
        Assert.notEmpty(list, "微信未绑定任何用户");
        Assert.isTrue(list.size() == 1, "微信绑定了多个用户");
        MdmUserRelWeChatRespVo one = list.get(0);
        if (YesNoEnum.yesNoEnum.Y.getValue().equals(ParamUtil.getParameterValue(ParameterParam.SFA_APPLET_USER_UNIQUE_FLAG))) {
            List<MdmUserRelWeChatRespVo> userRelList = mdmUserRelWeChatService.getBindWeChatListByUserName(one.getUserName(), LoginFromTypeEnum.APPLET_SFA.getValue());
            Assert.isTrue(userRelList.size() == 1, "用户绑定了多个微信，请先在后台解绑");
        }
        Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(one.getLoginStatus()), "处于退出登录状态，需要重新登录");

        mdmLoginHelpService.checkLock(one.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        MdmUserRespVo user = mdmUserService.getUserForLogin(one.getUserName(), null, null);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo, true, false, false);
    }

    @Override
    public MdmLoginTypeControlVo getLoginTypeControlConfig() {
        return mdmSystemLoginService.getLoginTypeControlConfig();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void logout() {
        UserRedis user = UserUtils.getUser();
        if (user != null) {
            if (StringUtils.isNotEmpty(user.getOpenId())) {
                mdmUserRelWeChatService.changeLoginStatus(LoginFromTypeEnum.APPLET_SFA.getValue(), user.getOpenId(), YesNoEnum.yesNoEnum.ZERO.getValue());
            }
            UserUtils.logout();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void logoutFromType() {
        UserRedis user = UserUtils.getUser();
        if (user != null) {
            if (StringUtils.isNotEmpty(user.getOpenId())) {
                mdmUserRelWeChatService.changeLoginStatus(LoginFromTypeEnum.APPLET_SFA.getValue(), user.getOpenId(), YesNoEnum.yesNoEnum.ZERO.getValue());
            }
            UserUtils.logout(LoginFromTypeEnum.APPLET_SFA.getValue());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void logoutAll() {
        UserRedis user = UserUtils.getUser();
        if (user != null) {
            if (StringUtils.isNotEmpty(user.getOpenId())) {
                mdmUserRelWeChatService.changeLoginStatus(LoginFromTypeEnum.APPLET_SFA.getValue(), user.getOpenId(), YesNoEnum.yesNoEnum.ZERO.getValue());
            }
            UserUtils.deleteUser(user.getUsername());
        }
    }


    /**
     * 执行登录操作
     *
     * @param user            用户信息
     * @param reqVo           请求参数
     * @param getLastLoginLog 获取上一次登录日志
     * @param resetPassword   强制下一次登录修改密码
     * @param bind            绑定微信
     * @return
     */
    protected MdmLoginRespVo doLoginThisSystem(MdmUserRespVo user, MdmAppletLoginReqVo reqVo, boolean getLastLoginLog, boolean resetPassword, boolean bind) {

        //获取当前职位
        if (bind && StringUtils.isNotEmpty(reqVo.getOpenId())) {
            //绑定微信与用户
            MdmUserRelWeChatReqVo bindReq = new MdmUserRelWeChatReqVo();
            CrmBeanUtil.copyProperties(reqVo, bindReq);
            bindReq.setUserName(user.getUserName());
            bindReq.setOriginModule(reqVo.getFromType());
            bindReq.setLoginStatus(YesNoEnum.yesNoEnum.ONE.getValue());
            mdmUserRelWeChatService.bind(bindReq);
        }

        MdmLoginRespVo mdmLoginRespVo = new MdmLoginRespVo();

        String token = UUID.randomUUID().toString().replaceAll("-", "");
        //生成登录信息
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(user.getUserName());
        userRedis.setRealname(user.getFullName());
        userRedis.setUsertype(user.getUserType());
        mdmLoginRespVo.setUserCode(user.getUserCode());
        mdmLoginRespVo.setLoginUserToken(token);
        mdmLoginRespVo.setUserName(user.getUserName());
        mdmLoginRespVo.setFullName(user.getFullName());
        mdmLoginRespVo.setUserType(user.getUserType());
        mdmLoginRespVo.setLanguage(userRedis.getLanguage());
        if (UserTypeEnum.USER.getCode().equals(user.getUserType())) {
            //获取当前职位
            MdmPositionUserOrgRespVo positionUserOrg = mdmPositionService.getCurrentPositionByUserName(user.getUserName());
            userRedis.setPoscode(positionUserOrg.getPositionCode());
            userRedis.setPosname(positionUserOrg.getPositionName());
            userRedis.setOrgcode(positionUserOrg.getOrgCode());
            userRedis.setOrgname(positionUserOrg.getOrgName());
            mdmLoginRespVo.setPositionCode(positionUserOrg.getPositionCode());
            mdmLoginRespVo.setPositionName(positionUserOrg.getPositionName());
            mdmLoginRespVo.setOrgCode(positionUserOrg.getOrgCode());
            mdmLoginRespVo.setOrgName(positionUserOrg.getOrgName());
        } else if (UserTypeEnum.CUSTOMER.getCode().equals(user.getUserType()) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(user.getUserType())) {
            MdmCustomerMsgRespVo customer = mdmCustomerMsgService.getUserCurrentCustomer(user.getUserName());
            if (customer == null) {
                throw new BusinessException("当前账号未关联可用的客户，请联系管理员");
            }
            Map<String, List<String>> orgCodeListMap = mdmCustomerROrgService.findOrgCodeList(Collections.singletonList(customer.getCustomerCode()));
            String orgCode = null;
            String orgName = null;
            if (orgCodeListMap.containsKey(customer.getCustomerCode())) {
                List<String> orgCodeList = orgCodeListMap.get(customer.getCustomerCode());
                Map<String, String> orgNameMap = mdmOrgService.findOrgNameMap(orgCodeList);
                orgName = orgCodeList.stream().map(orgNameMap::get).collect(Collectors.joining(","));
                orgCode = String.join(",", orgCodeList);
            }
            userRedis.setCustcode(customer.getCustomerCode());
            userRedis.setCustname(customer.getCustomerName());
            userRedis.setOrgcode(orgCode);
            userRedis.setOrgname(orgName);
            mdmLoginRespVo.setCustomerCode(customer.getCustomerCode());
            mdmLoginRespVo.setCustomerName(customer.getCustomerName());
            mdmLoginRespVo.setOrgCode(orgCode);
            mdmLoginRespVo.setOrgName(orgName);
            mdmLoginRespVo.setMdmCustomerMsgRespVo(customer);
        } else {

        }
        userRedis.setFromtype(reqVo.getFromType());
        UserUtils.setUser(token, userRedis);
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        HttpServletResponse response = HttpServletRequestUtil.getResponse();
        CookiesUtil.doCoke(request, response, token, GlobalParam.TOKEN);
        if (getLastLoginLog) {
            MdmUserLoginLogRespVo lastLoginLog = mdmUserLoginLogService.getLastLoginLog(user.getUserName(), null);
            mdmLoginRespVo.setLastLoginLog(lastLoginLog);
        }
        if (resetPassword) {
            mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
            mdmUserService.setUserForceChangePassword(user.getUserName());
        } else {
            if (YesNoEnum.yesNoEnum.ONE.getValue().equals(user.getForceChangePassword())) {
                mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
            }
        }
        mdmUserLoginLogAsyncService.saveLoginLog(request, user.getUserName(), user.getFullName(), reqVo.getFromType(), DateUtil.dateNowHms());
        return mdmLoginRespVo;
    }
}
