package com.biz.crm.login.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.login.service.MdmLoginHelpService;
import com.biz.crm.login.service.MdmSystemLoginService;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.login.MdmLoginReqVo;
import com.biz.crm.nebular.mdm.login.MdmLoginRespVo;
import com.biz.crm.nebular.mdm.login.MdmLoginTypeControlVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserLoginLogRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.user.model.MdmUserEntity;
import com.biz.crm.user.service.MdmUserLoginLogAsyncService;
import com.biz.crm.user.service.MdmUserLoginLogService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginHelpUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 系统登录业务层
 *
 * @author zxw
 * @date 2020-11-19 15:48
 **/
@Service
@Slf4j
@ConditionalOnMissingBean(name = "MdmSystemLoginServiceExpandImpl")
public class MdmSystemLoginServiceImpl implements MdmSystemLoginService {

    @Autowired
    private MdmUserService mdmUserService;

    @Autowired
    private MdmPositionService mdmPositionService;

    @Resource
    private MdmUserLoginLogService mdmUserLoginLogService;

    @Resource
    private MdmUserLoginLogAsyncService mdmUserLoginLogAsyncService;

    @Resource
    private MdmDictDataService mdmDictDataService;

    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;

    @Resource
    private MdmCustomerROrgService mdmCustomerROrgService;

    @Resource
    private MdmOrgService mdmOrgService;

    @Resource
    private MdmLoginHelpService mdmLoginHelpService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo login(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失账号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        String userName = reqVo.getUserName();
        String userPhone = null;
        String email = null;
        String accountContext = "账号";

        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        //判断设备码不为空，则进行修改
        if (StringUtils.isNotEmpty(reqVo.getRegistrationId())) {
            mdmUserService.lambdaUpdate()
                    .eq(MdmUserEntity::getUserName, user.getUserName())
                    .set(MdmUserEntity::getRegistrationId, reqVo.getRegistrationId())
                    .update();
        }
        Assert.notNull(user, "用户名不存在或密码错误");
        userPhone = user.getUserPhone();
        email = user.getEmail();
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(userName, userPhone, email);
        } else {
            mdmLoginHelpService.addError(userName, userPhone, email);
        }

        return doLoginThisSystem(user, reqVo.getFromType(), true, false);

    }

    @Override
    public MdmLoginRespVo loginByOnlyUserName(MdmLoginReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getUjlrwebjruzddjnu())) {
            reqVo.setUserName(reqVo.getUjlrwebjruzddjnu());
        }
        if (StringUtils.isEmpty(reqVo.getFromType())) {
            reqVo.setFromType(LoginFromTypeEnum.EXTERNAL.getValue());
        }
        Assert.hasText(reqVo.getUserName(), "缺失账号");
        String userName = reqVo.getUserName();
        String userPhone = null;
        String email = null;
        String accountContext = "账号";

        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        Assert.notNull(user, accountContext + "不存在");
        userPhone = user.getUserPhone();
        email = user.getEmail();
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(userName, userPhone, email);
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhone(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        String userName = null;
        String userPhone = reqVo.getUserPhone();
        String email = null;
        String accountContext = "手机号";
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        Assert.notNull(user, "手机号不存在或密码错误");
        userName = user.getUserName();
        email = user.getEmail();
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(userName, userPhone, email);
        } else {
            mdmLoginHelpService.addError(userName, userPhone, email);
        }
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmail(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        String userName = null;
        String userPhone = null;
        String email = reqVo.getEmail();
        String accountContext = "邮箱";
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        Assert.notNull(user, "邮箱不存在或密码错误");
        userName = user.getUserName();
        userPhone = user.getUserPhone();
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //校验密码
        if (Md5EncryptionAndDecryption.checkPwd(reqVo.getPassword(), user.getUserPassword())) {
            mdmLoginHelpService.unlock(userName, userPhone, email);
        } else {
            mdmLoginHelpService.addError(userName, userPhone, email);
        }
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByPhoneVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");

        String userName = null;
        String userPhone = reqVo.getUserPhone();
        String email = null;
        String accountContext = "手机号";
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //根据验证码+手机号获取当时的用户
        userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        userPhone = user.getUserPhone();
        email = user.getEmail();
        Assert.notNull(user, "手机号不存在或密码错误");
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(userName, userPhone, email);
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginByEmailVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = null;
        String userPhone = null;
        String email = reqVo.getEmail();
        String accountContext = "邮箱";
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //根据验证码+邮箱获取当时的用户
        userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        Assert.notNull(user, "用户名不存在或密码错误");
        userName = user.getUserName();
        userPhone = user.getUserPhone();
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        //解锁
        mdmLoginHelpService.unlock(userName, userPhone, email);
        return doLoginThisSystem(user, reqVo.getFromType(), true, false);

    }

    @Override
    public void checkPhoneExistAndUsable(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        Assert.notNull(user, "手机号不存在或密码错误");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
    }

    @Override
    public void checkEmailExistAndUsable(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        Assert.notNull(user, "邮箱不存在或密码错误");
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        //校验用户状态
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
    }

    @Override
    public void sendVerificationForLoginByMessage(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        Assert.notNull(user, "手机号不存在或密码错误");
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送短信
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    public void sendVerificationForLoginByEmail(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        Assert.notNull(user, "邮箱不存在或密码错误");
        mdmLoginHelpService.checkLock(user.getUserName(), LoginHelpUtil.AccountType.USER_NAME);
        mdmLoginHelpService.checkLock(user.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
        //发送邮箱验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByPhoneVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.notNull(user, "手机号不存在或密码错误");
        Assert.isTrue(reqVo.getUserPhone().equals(user.getUserPhone()), "手机号关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), false, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MdmLoginRespVo loginAndResetByEmailVerification(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        Assert.hasText(reqVo.getVerificationCode(), "缺失验证码");
        String userName = mdmLoginHelpService.checkVerificationCode(reqVo.getFromType(), reqVo.getVerificationCode(), reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, null, null);
        Assert.notNull(user, "邮箱不存在或密码错误");
        Assert.isTrue(reqVo.getEmail().equals(user.getEmail()), "邮箱关联用户发生变化，请重新登录");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        mdmLoginHelpService.unlock(user.getUserName(), user.getUserPhone(), user.getEmail());
        return doLoginThisSystem(user, reqVo.getFromType(), false, false);
    }

    @Override
    public void sendVerificationForLoginAndResetByMessage(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getUserPhone(), "缺失手机号");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getUserPhone(), LoginHelpUtil.AccountType.PHONE);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, reqVo.getUserPhone(), null);
        Assert.notNull(user, "手机号不存在或密码错误");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        //发送短信验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getUserPhone(), LoginHelpUtil.AccountType.PHONE, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
    }

    @Override
    public void sendVerificationForLoginAndResetByEmail(MdmLoginReqVo reqVo) {
        Assert.hasText(reqVo.getEmail(), "缺失邮箱");
        //校验锁定状态
        mdmLoginHelpService.checkLock(reqVo.getEmail(), LoginHelpUtil.AccountType.EMAIL);
        MdmUserRespVo user = mdmUserService.getUserForLogin(null, null, reqVo.getEmail());
        Assert.notNull(user, "邮箱不存在或密码错误");
        mdmLoginHelpService.checkUserType(user.getUserType(), reqVo.getFromType());
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());
        String verification = mdmLoginHelpService.saveLoginVerificationCodeRelUser(reqVo.getFromType(), user.getUserName(), user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
        //发送邮箱验证码
        mdmLoginHelpService.sendVerificationCode(reqVo.getFromType(), verification, user.getEmail(), LoginHelpUtil.AccountType.EMAIL, LoginHelpUtil.VerificationCodeType.LOGIN_AND_RESET);
    }

    @Override
    public MdmLoginTypeControlVo getLoginTypeControlConfig() {
        MdmLoginTypeControlVo controlVo = new MdmLoginTypeControlVo();
        try {
            Map<String, String> map = mdmDictDataService.getDictDataMap(DictConstant.LOGIN_CONTROL_CONFIG);
            controlVo.setLoginByUserName(YesNoEnum.yesNoEnum.ZERO.getValue().equals(map.get("system_login_by_user_name_pwd")) ? YesNoEnum.yesNoEnum.ZERO.getValue() : YesNoEnum.yesNoEnum.ONE.getValue());
            controlVo.setLoginByPhone(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_login_by_phone_pwd")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setLoginByPhoneVerification(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_login_by_phone_verification")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setLoginByEmail(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_login_by_email_pwd")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setLoginByEmailVerification(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_login_by_email_verification")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setLoginByWeChatWork(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_login_by_we_chat_work")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setResetByPhone(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_reset_by_phone")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
            controlVo.setResetByEmail(YesNoEnum.yesNoEnum.ONE.getValue().equals(map.get("system_reset_by_email")) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
        } catch (Exception e) {
            log.error("获取登录方式控制失败：{}", e);
        }
        return controlVo;
    }

    @Override
    public void logout() {
        UserUtils.logout();
    }

    @Override
    public void logoutFromType() {
        UserUtils.logout(LoginFromTypeEnum.CONSOLE.getValue());
    }

    @Override
    public void logoutAll() {
        UserRedis user = UserUtils.getUser();
        if (user != null && StringUtils.isNotEmpty(user.getUsername())) {
            UserUtils.deleteUser(user.getUsername());
        }
    }

    /**
     * 执行登录操作
     *
     * @param user            用户信息
     * @param fromType        登录入口
     * @param getLastLoginLog 是否查询上一次登录日志
     * @param resetPassword   强制重设密码
     * @return
     */
    protected MdmLoginRespVo doLoginThisSystem(MdmUserRespVo user, String fromType, boolean getLastLoginLog, boolean resetPassword) {

        MdmLoginRespVo mdmLoginRespVo = new MdmLoginRespVo();

        String token = UUID.randomUUID().toString().replaceAll("-", "");
        //生成登录信息
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(user.getUserName());
        userRedis.setRealname(user.getFullName());
        userRedis.setUsertype(user.getUserType());
        mdmLoginRespVo.setUserCode(user.getUserCode());
        mdmLoginRespVo.setLoginUserToken(token);
        mdmLoginRespVo.setUserName(user.getUserName());
        mdmLoginRespVo.setFullName(user.getFullName());
        mdmLoginRespVo.setUserType(user.getUserType());
        mdmLoginRespVo.setLanguage(userRedis.getLanguage());
        if (UserTypeEnum.USER.getCode().equals(user.getUserType())) {
            //获取当前职位
            MdmPositionUserOrgRespVo positionUserOrg = mdmPositionService.getCurrentPositionByUserName(user.getUserName());
            userRedis.setPoscode(positionUserOrg.getPositionCode());
            userRedis.setPosname(positionUserOrg.getPositionName());
            userRedis.setOrgcode(positionUserOrg.getOrgCode());
            userRedis.setOrgname(positionUserOrg.getOrgName());
            mdmLoginRespVo.setPositionCode(positionUserOrg.getPositionCode());
            mdmLoginRespVo.setPositionName(positionUserOrg.getPositionName());
            mdmLoginRespVo.setOrgCode(positionUserOrg.getOrgCode());
            mdmLoginRespVo.setOrgName(positionUserOrg.getOrgName());
        } else if (UserTypeEnum.CUSTOMER.getCode().equals(user.getUserType()) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(user.getUserType())) {
            MdmCustomerMsgRespVo customer = mdmCustomerMsgService.getUserCurrentCustomer(user.getUserName());
            if (customer == null) {
                throw new BusinessException("当前账号未关联可用的客户，请联系管理员");
            }
            Map<String, List<String>> orgCodeListMap = mdmCustomerROrgService.findOrgCodeList(Collections.singletonList(customer.getCustomerCode()));
            String orgCode = null;
            String orgName = null;
            if (orgCodeListMap.containsKey(customer.getCustomerCode())) {
                List<String> orgCodeList = orgCodeListMap.get(customer.getCustomerCode());
                Map<String, String> orgNameMap = mdmOrgService.findOrgNameMap(orgCodeList);
                orgName = orgCodeList.stream().map(orgNameMap::get).collect(Collectors.joining(","));
                orgCode = String.join(",", orgCodeList);
            }
            //小程序以用户名替换职位
            mdmLoginRespVo.setPositionCode(user.getUserName());
            userRedis.setRealname(user.getFullName());
            userRedis.setPoscode(user.getUserName());

            userRedis.setCustcode(customer.getCustomerCode());
            userRedis.setCustname(customer.getCustomerName());
            userRedis.setOrgcode(orgCode);
            userRedis.setOrgname(orgName);
            mdmLoginRespVo.setCustomerCode(customer.getCustomerCode());
            mdmLoginRespVo.setCustomerName(customer.getCustomerName());
            mdmLoginRespVo.setOrgCode(orgCode);
            mdmLoginRespVo.setOrgName(orgName);
            mdmLoginRespVo.setMdmCustomerMsgRespVo(customer);
        } else {

        }
        userRedis.setFromtype(fromType);
        UserUtils.setUser(token, userRedis);
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        HttpServletResponse response = HttpServletRequestUtil.getResponse();
        CookiesUtil.doCoke(request, response, token, GlobalParam.TOKEN);
        if (getLastLoginLog) {
            MdmUserLoginLogRespVo lastLoginLog = mdmUserLoginLogService.getLastLoginLog(user.getUserName(), null);
            mdmLoginRespVo.setLastLoginLog(lastLoginLog);
        }
        if (resetPassword) {
            mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
            mdmUserService.setUserForceChangePassword(user.getUserName());
        } else {
            if (YesNoEnum.yesNoEnum.ONE.getValue().equals(user.getForceChangePassword())) {
                mdmLoginRespVo.setResetPassword(YesNoEnum.yesNoEnum.ONE.getValue());
            }
        }

        // 校验密码是否超过三个月时间没有修改
        String updatePasswordDate = user.getUpdatePasswordDate();
        if(StringUtils.isNotEmpty(updatePasswordDate)){
            int monthSpace = DateUtil.getMonthSpace(DateUtil.getYYYYMMDD(), updatePasswordDate, DateUtil.DEFAULT_DATE_FORMAT);
            if(monthSpace>3){
                mdmLoginRespVo.setRemandMessage("温馨提示：你的密码已超过三个月未修改，请修改密码。");
            }
        }
        mdmUserLoginLogAsyncService.saveLoginLog(request, user.getUserName(), user.getFullName(), fromType, DateUtil.dateNowHms());
        return mdmLoginRespVo;
    }

    /**
     * 通过token查询用户信息
     *
     * @return
     */
    @Override
    public MdmLoginRespVo findUserByToken() {
        UserRedis userRedis = UserUtils.getUser();
        String userName = userRedis.getUsername();
        String userPhone = null;
        String email = null;
        String accountContext = "账号";
        //校验锁定状态
        mdmLoginHelpService.checkLock(userName, LoginHelpUtil.AccountType.USER_NAME);
        //获取用户
        MdmUserRespVo user = mdmUserService.getUserForLogin(userName, userPhone, email);
        Assert.notNull(user, "账号不存在或密码错误");
        userPhone = user.getUserPhone();
        email = user.getEmail();
        //校验用户类型
        mdmLoginHelpService.checkUserType(user.getUserType(), LoginFromTypeEnum.EXTERNAL.getValue());
        //校验用户可用性
        mdmLoginHelpService.checkUserLogin(user.getEnableStatus(), user.getStartTime(), user.getEndTime());

        MdmLoginRespVo mdmLoginRespVo = new MdmLoginRespVo();
        mdmLoginRespVo.setUserCode(user.getUserCode());
        mdmLoginRespVo.setLoginUserToken(UserUtils.getToken());
        mdmLoginRespVo.setUserName(user.getUserName());
        mdmLoginRespVo.setFullName(user.getFullName());
        mdmLoginRespVo.setUserType(user.getUserType());
        mdmLoginRespVo.setLanguage(userRedis.getLanguage());
        if (UserTypeEnum.USER.getCode().equals(user.getUserType())) {
            //获取当前职位
            MdmPositionUserOrgRespVo positionUserOrg = mdmPositionService.getCurrentPositionByUserName(user.getUserName());
            mdmLoginRespVo.setPositionCode(positionUserOrg.getPositionCode());
            mdmLoginRespVo.setPositionName(positionUserOrg.getPositionName());
            mdmLoginRespVo.setOrgCode(positionUserOrg.getOrgCode());
            mdmLoginRespVo.setOrgName(positionUserOrg.getOrgName());
        }
        return mdmLoginRespVo;
    }
}
