package com.biz.crm.newhope.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.FilterEnum;
import com.biz.crm.nebular.mdm.newhope.req.NewHopeReqVo;
import com.biz.crm.nebular.mdm.newhope.resp.NewHopeRespVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.newhope.service.OrgSyncService;
import com.biz.crm.newhope.util.Body;
import com.biz.crm.newhope.util.HopeResult;
import com.biz.crm.newhope.util.NewHopeUtils;
import com.biz.crm.org.mapper.MdmOrgMapper;
import com.biz.crm.org.model.MdmOrgEntity;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.util.*;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.assertj.core.util.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author maoshen
 * @date 2021/6/3.
 */
@Slf4j
@Service
public class OrgSyncServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmOrgMapper, MdmOrgEntity> implements OrgSyncService {
    @Autowired
    private MdmOrgService service;

    @Resource
    private MdmOrgMapper mdmOrgMapper;

    private static AtomicInteger count = new AtomicInteger();


    @Override
    public Body orgSyncWithCookie(NewHopeReqVo reqVo) throws UnsupportedEncodingException {
        AssertUtils.isNotEmpty(reqVo.getFilter(), "同步方式不能为空");
        AssertUtils.isTrue(FilterEnum.needOrgUrlSet().contains(reqVo.getFilter()), "组织同步方式与Url不匹配");
        if (StringUtils.equals(reqVo.getFilter(), FilterEnum.INCREMENT_ORG.getCode())) {
            AssertUtils.isNotEmpty(reqVo.getIncrementTime(), "查询组织增量，增量时间不能为空");
        }
        NewHopeRespVo respVo = NewHopeUtils.getNewHopeRespVo(reqVo);
        // 设置请求头 headers
        Map<String, String> headers = getStringStringMap(respVo);
        Result<String> result = HttpClientUtils.doGet(respVo.getUrl(), headers);
        HopeResult hopeResult = JsonPropertyUtil.toObject(result.getResult(), HopeResult.class);
        if (hopeResult == null) {
            return new Body();
        }
        return hopeResult.getBody();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void orgSyncAll(NewHopeReqVo reqVo) throws UnsupportedEncodingException {
        log.info("组织全量同步开始");
        reqVo.setFilter(FilterEnum.ORG.getCode());
        List<MdmOrgEntity> entityList = Lists.newArrayList();
        Set<String> orgCodes = Sets.newHashSet();
        String cookie = "";
        while ("".equals(cookie) || (cookie != null && cookie.length() > 1)) {
            cookie = null;
            NewHopeRespVo respVo = NewHopeUtils.getNewHopeRespVo(reqVo);
            // 设置请求头 headers
            Map<String, String> headers = getStringStringMap(respVo);
            Result<String> result = HttpClientUtils.doGet(respVo.getUrl(), headers);
            HopeResult hopeResult = JsonPropertyUtil.toObject(ApiResultUtil.objResult(result, true), HopeResult.class);
            log.info("请求结果：{} | {}", hopeResult.getSuccess(), hopeResult.getResultCode());
            if (hopeResult == null) {
                return;
            }
            if (hopeResult.getBody() == null) {
                return;
            }
            if (CollectionUtils.isEmpty(hopeResult.getBody().getOrgs())) {
                return;
            }
            hopeResult.getBody().getOrgs().forEach(x -> {
                if (x != null) {
                    if (x.getOrgId() == null) {
                        return;
                    }
                    MdmOrgEntity entity = new MdmOrgEntity();
                    if (orgCodes.contains(x.getOrgId())) {
                        throw new BusinessException(x.getOrgId() + "组织编号orgId不能重复");
                    }
                    entity.setOrgCode(x.getOrgId());
                    if ("00000000".equals(x.getParentOrgId())) {
                        entity.setParentCode(null);
                    } else {
                        entity.setParentCode(x.getParentOrgId());
                    }
                    entity.setOrgName(x.getOrgShortName());
                    if (StringUtils.isNotBlank(x.getOrglevel())) {
                        entity.setLevelNum(Integer.parseInt(x.getOrglevel()));
                    }
//                    entity.setRuleCode(generateRuleCodeByParentCode(entity.getParentCode()));
                    if ("1".equals(x.getOrgstatus())) {
                        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                    }
                    if ("0".equals(x.getOrgstatus())) {
                        entity.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                    }
                    orgCodes.add(x.getOrgId());
                    entityList.add(entity);
                }
            });
            cookie = hopeResult.getBody().getCookie();
            reqVo.setCookie(hopeResult.getBody().getCookie());
        }
        if (CollectionUtils.isEmpty(orgCodes)) {
            log.info("本次同步数据为零");
            return;
        }
        //
        List<MdmOrgRespVo> orgRespVos = buildOrg(null, entityList);
        log.info("数据同步写入数据库");
        saveOrgBatch(orgRespVos);
        log.info("组织全量同步结束");
        log.info("共同步" + count.get() + "条数据");
    }

    private void saveOrgBatch(List<MdmOrgRespVo> orgRespVos) {
        if (!org.springframework.util.CollectionUtils.isEmpty(orgRespVos)) {
            orgRespVos.forEach(e -> {
                e.setRuleCode(generateRuleCodeByParentCode(e.getParentCode()));
                MdmOrgEntity en = CrmBeanUtil.copy(e, MdmOrgEntity.class);
                saveOrUpdate(en);
                count.getAndIncrement();
                if (!org.springframework.util.CollectionUtils.isEmpty(e.getRespVos())) {
                    saveOrgBatch(e.getRespVos());
                }
            });
        }
    }

    private List<MdmOrgRespVo> buildOrg(MdmOrgRespVo respVo, List<MdmOrgEntity> entityList) {
        List<MdmOrgRespVo> respVos = Lists.newArrayList();
        entityList.forEach(o -> {
            if (org.springframework.util.StringUtils.isEmpty(o.getParentCode())) {
                o.setLevelNum(1);
                o.setId(o.getOrgCode());
                respVos.add(CrmBeanUtil.copy(o, MdmOrgRespVo.class));
            }
        });
        //
        buildOrgLevel(respVos, entityList);
        return respVos;
    }

    private void buildOrgLevel(List<MdmOrgRespVo> respVos, List<MdmOrgEntity> entityList) {
        respVos.forEach(o -> {
            String orgCode = o.getOrgCode();
            List<MdmOrgEntity> entities = entityList.stream()
                    .filter(e -> orgCode.equals(e.getParentCode())).collect(Collectors.toList());
            entities.forEach(e -> {
                e.setId(e.getOrgCode());
                e.setLevelNum(o.getLevelNum() + 1);
            });
            if (!org.springframework.util.CollectionUtils.isEmpty(entities)) {
                o.setRespVos(CrmBeanUtil.copyList(entities, MdmOrgRespVo.class));
                buildOrgLevel(o.getRespVos(), entityList);
            }
        });
    }

    private Map<String, String> getStringStringMap(NewHopeRespVo respVo) {
        Map<String, String> headers = Maps.newHashMap();
        headers.put("sign", respVo.getSign());
        headers.put("encodekey", respVo.getEncodekey());
        headers.put("randomcode", respVo.getRandomcode());
        headers.put("appuser", respVo.getAppuser());
        headers.put("timestamp", respVo.getTimestamp());
        return headers;
    }

    private String generateRuleCodeByParentCode(String parentCode) {
        String ruleCode = "";
        boolean top = true;
        String prefix = "";
        if (com.biz.crm.util.StringUtils.isNotEmpty(parentCode)) {
            MdmOrgEntity parent = mdmOrgMapper.selectOne(Wrappers.<MdmOrgEntity>lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, parentCode));
            if (parent != null) {
                top = false;
                prefix = parent.getRuleCode();
            }
        }
        List<MdmOrgEntity> list = null;
        if (top) {
            list = this.lambdaQuery().isNull(MdmOrgEntity::getParentCode).or().eq(MdmOrgEntity::getParentCode, "").list();
        } else {
            list = this.lambdaQuery().eq(MdmOrgEntity::getParentCode, parentCode).list();
        }
        if (CollectionUtil.listEmpty(list)) {
            ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(1);
        } else {
            Set<Integer> set = new LinkedHashSet<>();
            for (MdmOrgEntity item :
                    list) {
                if (com.biz.crm.util.StringUtils.isNotEmpty(item.getRuleCode())) {
                    try {
                        Integer integer = TreeRuleCodeUtil.getCurLevelCodeValue(item.getRuleCode());
                        set.add(integer);
                    } catch (BusinessException e) {
                        throw new BusinessException("企业组织" + e.getMsg());
                    }
                }
            }
            int maxRuleCodeValueInLevel = TreeRuleCodeUtil.getMaxRuleCodeValueInLevel();
            for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
                if (i == maxRuleCodeValueInLevel) {
                    throw new BusinessException("降维编码越界，请联系管理员处理");
                }
                if (!set.contains(i)) {
                    ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(i);
                    break;
                }
            }
        }
        return ruleCode;
    }


    /**
     * 增量数据同步
     *
     * @param reqVo
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void orgSyncIncrement(NewHopeReqVo reqVo) throws UnsupportedEncodingException {
        log.info("组织增量同步开始");
        reqVo.setFilter(FilterEnum.INCREMENT_ORG.getCode());
        if (StringUtils.isBlank(reqVo.getIncrementTime())) {
            reqVo.setIncrementTime(new SimpleDateFormat("yyyy-MM-dd").format(DateUtil.yesterday()) + " " + DateUtil.DAY_EARLIEST_TIME);
        }
        reqVo.setIncrementTime(format(reqVo.getIncrementTime()) + "Z");
        List<MdmOrgEntity> mdmOrgEntitiesHasParentCode = Lists.newArrayList();
        List<MdmOrgEntity> mdmOrgEntities = Lists.newArrayList();
        Set<String> orgCodes = Sets.newHashSet();
        String cookie = "";
        while ("".equals(cookie) || (cookie != null && cookie.length() > 1)) {
            cookie = null;
            NewHopeRespVo respVo = NewHopeUtils.getNewHopeRespVo(reqVo);
            Map<String, String> header = getStringStringMap(respVo);
            Result<String> result = HttpClientUtils.doGet(respVo.getUrl(), header);
            HopeResult hopeResult = JsonPropertyUtil.toObject(ApiResultUtil.objResult(result, true), HopeResult.class);
            if (hopeResult == null) {
                return;
            }
            if (hopeResult.getBody() == null) {
                return;
            }
            if (CollectionUtils.isEmpty(hopeResult.getBody().getOrgs())) {
                return;
            }
            hopeResult.getBody().getOrgs().forEach(x -> {
                if (x != null) {
                    if (x.getOrgId() == null) {
                        return;
                    }
                    MdmOrgEntity entity = new MdmOrgEntity();
                    if (orgCodes.contains(x.getOrgId())) {
                        throw new BusinessException(x.getOrgId() + "组织编号orgId不能重复");
                    }
                    entity.setOrgCode(x.getOrgId());
                    entity.setParentCode(x.getParentOrgId());
                    if ("00000000".equals(x.getParentOrgId())) {
                        entity.setParentCode(null);
                    }
                    entity.setOrgName(x.getOrgShortName());
                    if (StringUtils.isNotBlank(x.getOrglevel())) {
                        entity.setLevelNum(Integer.parseInt(x.getOrglevel()));
                    }
                    if (StringUtils.isBlank(entity.getParentCode())) {
                        entity.setLevelNum(1);
                    }
                    if ("1".equals(x.getOrgstatus())) {
                        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                    }
                    if ("0".equals(x.getOrgstatus())) {
                        entity.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                    }
                    entity.setId(x.getOrgId());
                    orgCodes.add(x.getOrgId());

                    if (StringUtils.isNotBlank(entity.getParentCode())) {
                        mdmOrgEntitiesHasParentCode.add(entity);
                    } else {
                        mdmOrgEntities.add(entity);
                    }
                }
            });
            if (StringUtils.isBlank(hopeResult.getBody().getCookie())) {
                break;
            }
            reqVo.setCookie(hopeResult.getBody().getCookie());
            cookie = reqVo.getCookie();
        }
        // 写入最上级父类
        saveData(mdmOrgEntitiesHasParentCode, mdmOrgEntities);
        log.info("全量同步结束");
        log.info("共同步" + count.get() + "条组织数据");
    }

    /**
     * 同步数据
     *
     * @param mdmOrgEntitiesHasParentCode
     * @param mdmOrgEntities
     */
    private void saveData(List<MdmOrgEntity> mdmOrgEntitiesHasParentCode, List<MdmOrgEntity> mdmOrgEntities) {
        if (CollectionUtils.isNotEmpty(mdmOrgEntities)) {
            mdmOrgEntities.forEach(x -> {
                if (mdmOrgMapper.selectById(x.getId()) == null) {
                    x.setRuleCode(generateRuleCodeByParentCode(x.getParentCode()));
                    this.save(CrmBeanUtil.copy(x, MdmOrgEntity.class));
                } else {
                    service.update(CrmBeanUtil.copy(x, MdmOrgReqVo.class));
                }
                count.getAndIncrement();
            });
        }
        // 写入子级 组织
        while (true) {
            if (mdmOrgEntitiesHasParentCode.size() == 1) {
                saveOrgUpdateOrgEntity(mdmOrgEntitiesHasParentCode.get(0));
                count.getAndIncrement();
                break;
            }
            MdmOrgEntity saveEntity = new MdmOrgEntity();
            MdmOrgEntity entity = mdmOrgEntitiesHasParentCode.get(0);
            // 取出list(0)相对最上级父类 如果没有，list（0）同步与该组数组没有任何关系 如果有，先增加他的父类
            for (int i = 0; i < mdmOrgEntitiesHasParentCode.size() - 1; i++) {

                saveEntity = entity;
                if (StringUtils.equals(entity.getParentCode()
                        , mdmOrgEntitiesHasParentCode.get(i + 1).getOrgCode())) {

                    entity = mdmOrgEntitiesHasParentCode.get(i + 1);
                    saveEntity = entity;
                }
            }
            mdmOrgEntitiesHasParentCode.remove(saveEntity);
            saveOrgUpdateOrgEntity(saveEntity);
        }

    }

    private void saveOrgUpdateOrgEntity(MdmOrgEntity saveEntity) {
        // 待同步数据，在待同步集合中，已经处于最高层级了
        // 判断数据库中，他的父类是否存在，如果存在，进行 新增或者修改操作，如果不存在，不进行操作

        MdmOrgEntity parentEntityByDb = mdmOrgMapper.selectById(saveEntity.getParentCode());
        if (parentEntityByDb == null) {
            return;
        }
        MdmOrgEntity entityByDb = mdmOrgMapper.selectById(saveEntity.getId());
        saveEntity.setLevelNum(parentEntityByDb.getLevelNum() + 1);
        MdmOrgReqVo mdmOrgReqVo = CrmBeanUtil.copy(saveEntity, MdmOrgReqVo.class);
        if (entityByDb != null) {
            service.update(mdmOrgReqVo);
            count.getAndIncrement();
        } else {
            saveEntity.setRuleCode(generateRuleCodeByParentCode(saveEntity.getParentCode()));
            this.save(saveEntity);
            count.getAndIncrement();
        }
    }


    /**
     * 忽略标点符号；
     *
     * @param s
     * @return
     */
    public static String format(String s) {
        String str = s.replaceAll("[`qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM~!@#$%^&*()+=|{}':;',\\[\\].<>/?~！@#￥%……& amp;*（）——+|{}【】‘；：”“’。，、？|-]", "");
        return str;
    }

    public Map<String, MdmOrgEntity> getParentOrgMap(Set<String> orgCodes) {
        //Map<String,MdmOrgEntity> map = Maps.newHashMap();
        List<MdmOrgEntity> entities = mdmOrgMapper.selectList(Wrappers.<MdmOrgEntity>lambdaQuery()
                .in(MdmOrgEntity::getOrgCode, orgCodes));
        if (CollectionUtils.isEmpty(entities)) {
            return Maps.newHashMap();
        }
        return entities.stream().collect(Collectors.toMap(MdmOrgEntity::getOrgCode, Function.identity()));
    }
}
