package com.biz.crm.notice.service.impl;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.MdmCodeEnum;
import com.biz.crm.eunm.mdm.MdmNoticeStatusEnum;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.nebular.mdm.notice.req.*;
import com.biz.crm.nebular.mdm.notice.resp.*;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.notice.mapper.*;
import com.biz.crm.notice.model.*;
import com.biz.crm.notice.service.*;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 公告管理 接口实现
 *
 * @author liuhongming
 * @date 2020-09-17 17:14:42
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmNoticeServiceExpandImpl")
@Transactional(rollbackFor = Exception.class)
public class MdmNoticeServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmNoticeMapper, MdmNoticeEntity> implements IMdmNoticeService {

    @Resource
    private MdmNoticeMapper mdmNoticeMapper;
    @Resource
    private MdmNoticeFileMapper noticeFileMapper;
    @Resource
    private MdmNoticeReceivingMapper noticeReceivingMapper;
    @Resource
    private MdmNoticeCommentMapper noticeCommentMapper;
    @Resource
    private MdmNoticeCommentLikesMapper noticeCommentLikesMapper;

    @Resource
    private MdmNoticeReadMapper mdmNoticeReadMapper;

    @Autowired
    private IMdmNoticeFileService iMdmNoticeFileService;
    @Autowired
    private IMdmNoticeReceivingService iMdmNoticeReceivingService;
    @Autowired
    private IMdmNoticeReadService iMdmNoticeReadService;
    @Autowired
    private MdmPositionRoleService mdmPositionRoleService;
    @Autowired
    private IMdmNoticeCommentService iMdmNoticeCommentService;
    @Resource
    private MdmOrgFeign mdmOrgFeign;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmNoticeRespVo> findList(MdmNoticeReqVo reqVo) {
        Page<MdmNoticeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmNoticeRespVo> list = mdmNoticeMapper.findList(page, reqVo);
        this.setNoticeStatus(list);

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> noticeCodes = list.stream().map(MdmNoticeRespVo::getNoticeCode).collect(Collectors.toList());
            MdmNoticeReadReqVo readReqVo = new MdmNoticeReadReqVo();
            readReqVo.setNoticeCodes(noticeCodes);
            readReqVo.setPageSize(-1);
            List<MdmNoticeReadRespVo> readRespVoList = iMdmNoticeReadService.findList(readReqVo).getData();
            Map<String, List<MdmNoticeReadRespVo>> readRespMap = readRespVoList.stream()
                    .collect(Collectors.groupingBy(MdmNoticeReadRespVo::getNoticeCode));
            for (MdmNoticeRespVo noticeVo : list) {
                noticeVo.setNoticeOrg(noticeVo.getCreateOrgName());
                noticeVo.setNoticeDate(noticeVo.getCreateDate());
                noticeVo.setEnableStatusName(CrmEnableStatusEnum.getDesc(noticeVo.getEnableStatus()));
                List<MdmNoticeReadRespVo> noticeReadRespVos = readRespMap.get(noticeVo.getNoticeCode());
                if (noticeReadRespVos != null && noticeReadRespVos.size() > 0) {
                    //todo 访问人数是否进行去重处理
                    noticeVo.setVisitNum(noticeReadRespVos.size());
                    Set<String> readusers = noticeReadRespVos.stream().map(MdmNoticeReadRespVo::getUserName).collect(Collectors.toSet());
                    noticeVo.setReadUserNum(readusers.size());
                } else {
                    noticeVo.setVisitNum(0);
                    noticeVo.setReadUserNum(0);
                }
            }
        }
        return PageResult.<MdmNoticeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }


    public void setNoticeStatus(List<MdmNoticeRespVo> list){
        for (MdmNoticeRespVo noticeRespVo : list){
            noticeRespVo.setNoticeReleaseStatusName(YesNoEnum.yesNoEnum.getEnum(noticeRespVo.getNoticeReleaseStatus()).getDes());
            noticeRespVo.setNoticeTopStateName(MdmNoticeStatusEnum.NoticeTopStatus.GETMAP.get(noticeRespVo.getNoticeTopState()));
            noticeRespVo.setCommentStateName(MdmNoticeStatusEnum.NoticeCommentStatus.GETMAP.get(noticeRespVo.getCommentState()));
        }
    }

    @Override
    public PageResult<MdmNoticeRespVo> homePageList(MdmNoticeReqVo reqVo) {
        Page<MdmNoticeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        String orgCode = UserUtils.getUser().getOrgcode();
        if (StringUtils.isEmpty(orgCode)) {
            return PageResult.<MdmNoticeRespVo>builder()
                    .data(Lists.newArrayList())
                    .count(page.getTotal())
                    .build();
        }
        MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo() {{
            this.setOrgCode(orgCode);
        }};
        Result<List<MdmOrgRespVo>> result = mdmOrgFeign.findAllChildrenOrgList(mdmOrgReqVo);
        List<String> orgCodeList = null;
        if (result != null && result.isSuccess()) {
            if (CollectionUtil.listNotEmptyNotSizeZero(result.getResult())) {
                orgCodeList = result.getResult().stream().map(MdmOrgRespVo::getOrgCode).distinct().collect(Collectors.toList());
                reqVo.setOrgCodeList(orgCodeList);
            } else {
                orgCodeList = Lists.newArrayList(orgCode);
                reqVo.setOrgCodeList(orgCodeList);
            }
        }
        //角色集合
        UserRedis user = UserUtils.getUser();
        String positionCode = user.getPoscode();
        //通过职位编码查询对应权限角色
        List<String> positionCodeList = new ArrayList<>();
        positionCodeList.add(positionCode);
        List<String> roleList = mdmPositionRoleService.findRoleCodeByPosition(positionCodeList);
        if (CollectionUtils.isNotEmpty(roleList)){
            List<String> collect = reqVo.getOrgCodeList().stream().sequential().collect(Collectors.toCollection(() -> roleList));
            reqVo.setOrgCodeList(collect);
        }
        List<MdmNoticeRespVo> list = mdmNoticeMapper.findList(page, reqVo);
        this.setNoticeStatus(list);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> noticeCodes = list.stream().map(MdmNoticeRespVo::getNoticeCode).collect(Collectors.toList());
            MdmNoticeReadReqVo readReqVo = new MdmNoticeReadReqVo();
            readReqVo.setNoticeCodes(noticeCodes);
            readReqVo.setPageSize(-1);
            List<MdmNoticeReadRespVo> readRespVoList = iMdmNoticeReadService.findList(readReqVo).getData();
            Map<String, List<MdmNoticeReadRespVo>> readRespMap = readRespVoList.stream()
                    .collect(Collectors.groupingBy(MdmNoticeReadRespVo::getNoticeCode));
            for (MdmNoticeRespVo noticeVo : list) {
                noticeVo.setNoticeOrg(noticeVo.getCreateOrgName());
                noticeVo.setNoticeDate(noticeVo.getCreateDate());
                noticeVo.setEnableStatusName(CrmEnableStatusEnum.getDesc(noticeVo.getEnableStatus()));
                List<MdmNoticeReadRespVo> noticeReadRespVos = readRespMap.get(noticeVo.getNoticeCode());
                if (noticeReadRespVos != null && noticeReadRespVos.size() > 0) {
                    //todo 访问人数是否进行去重处理
                    noticeVo.setVisitNum(noticeReadRespVos.size());
                    Set<String> readusers = noticeReadRespVos.stream().map(MdmNoticeReadRespVo::getUserName).collect(Collectors.toSet());
                    noticeVo.setReadUserNum(readusers.size());
                } else {
                    noticeVo.setVisitNum(0);
                    noticeVo.setReadUserNum(0);
                }
            }
        }
        return PageResult.<MdmNoticeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 小程序公告列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmNoticeRespVo> findListByMiniPrg(MdmNoticeMiniReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        Page<MdmNoticeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        //账号
        reqVo.setUserName(user.getUsername());
        //当前组织code
        reqVo.setThisCode(user.getOrgcode());
        //职位
        reqVo.setThisPost(user.getPoscode());
        //通过职位编码查询对应权限角色
        List<String> positionCodeList = new ArrayList<>();
        positionCodeList.add(user.getPoscode());
        List<String> roleList = mdmPositionRoleService.findRoleCodeByPosition(positionCodeList);
        if (CollectionUtils.isNotEmpty(roleList)){
            roleList.add(user.getOrgcode());
            reqVo.setOrgs(roleList);
        }
        List<MdmNoticeRespVo> miniPrgList = this.mdmNoticeMapper.findListMiniPrg(page, reqVo, YesNoEnum.yesNoEnum.ONE.getValue());
        Long total = page.getTotal();
        if (this.isNoticeAdmin()){
            PageResult<MdmNoticeRespVo> list = this.findList(CrmBeanUtil.copy(reqVo, MdmNoticeReqVo.class));
            miniPrgList = list.getData();
            total = list.getCount();
        }
        if (CollectionUtil.listEmpty(miniPrgList)){
            return PageResult.<MdmNoticeRespVo>builder()
                    .data(miniPrgList)
                    .count(page.getTotal())
                    .build();
        }
        this.setNoticeStatus(miniPrgList);
        List<String> notices = new ArrayList<>();
        for (MdmNoticeRespVo mdmNoticeRespVo : miniPrgList) {
            notices.add(mdmNoticeRespVo.getNoticeCode());
        }
        if (!CollectionUtil.listEmpty(notices)) {
            List<MdmNoticeReadRespVo> isItRight = this.mdmNoticeReadMapper.findIsItRight(notices, reqVo.getUserName(), reqVo.getThisPost(), user.getOrgcode());
            if (CollectionUtil.listNotEmpty(isItRight)) {
                Map<String, MdmNoticeReadRespVo> map = isItRight.stream().collect(Collectors.toMap(MdmNoticeReadRespVo::getNoticeCode, Function.identity(), (k1,k2) -> k2));
                for (MdmNoticeRespVo mdmNoticeRespVo : miniPrgList) {
                    if (map.containsKey(mdmNoticeRespVo.getNoticeCode())) {
                        //是否已读
                        mdmNoticeRespVo.setHaveRead("1");
                        continue;
                    }
                    mdmNoticeRespVo.setHaveRead("0");
                }
            }else{
                for (MdmNoticeRespVo mdmNoticeRespVo : miniPrgList) {
                    mdmNoticeRespVo.setHaveRead("0");
                }
            }
        }
        return PageResult.<MdmNoticeRespVo>builder()
                .data(miniPrgList)
                .count(total)
                .build();
    }

    /**
     * 小程序公告最新
     *
     * @return
     */
    @Override
    public MdmNoticeRespVo queryMiniPrgNew() {
        UserRedis user = UserUtils.getUser();
        List<String> positionCodeList = new ArrayList<>();
        positionCodeList.add(user.getPoscode());
        List<String> roleList = mdmPositionRoleService.findRoleCodeByPosition(positionCodeList);
        if (CollectionUtils.isNotEmpty(roleList)){
            roleList.add(user.getOrgcode());
        }
        List<MdmNoticeRespVo> data = mdmNoticeMapper.findListMiniPrgNew(roleList);
        if (CollectionUtil.listEmpty(data)) {
            return new MdmNoticeRespVo();
        }
        List<MdmNoticeRespVo> collect = data.stream().filter(x -> LocalDateTime.parse(x.getEndDate(), CrmDateUtils.yyyyMMddHHmmss).isAfter(LocalDateTime.now()))
                .sorted(Comparator.comparing(MdmNoticeRespVo::getCreateDateTime).thenComparing(MdmNoticeRespVo::getCreateDateSecond).reversed())
                .collect(Collectors.toList());
        if(CollectionUtil.listNotEmptyNotSizeZero(collect)){
            this.setNoticeStatus(data);
            return collect.get(0);
        }
        return new MdmNoticeRespVo();
    }


    /**
     * 小程序详情
     *
     * @param id
     * @return
     */
    @Override
    public MdmNoticeRespVo queryMiniPrg(String id) {
        AssertUtils.isNotEmpty(id, "Id不能为空");
        MdmNoticeRespVo mdmNoticeRespVo = mdmNoticeMapper.queryMimiPrg(id);
        this.setNoticeStatus(Arrays.asList(mdmNoticeRespVo));
        AssertUtils.isNotNull(mdmNoticeRespVo, "数据不存在");
        String noticeCode = mdmNoticeRespVo.getNoticeCode();
        List<MdmNoticeFileRespVo> miniList = noticeFileMapper.findMiniList(noticeCode);
        if (!CollectionUtil.listEmpty(miniList)) {
            //截取文件类型后缀
            miniList.forEach(m -> m.setFileSuffix
                    (StringUtils.isEmpty(m.getFileName()) ? "" : m.getFileName().substring(m.getFileName().lastIndexOf(".") + 1)));
        }
        mdmNoticeRespVo.setNoticeFileRespVos(miniList);
        iMdmNoticeReadService.save(mdmNoticeRespVo.getNoticeCode());
        return mdmNoticeRespVo;
    }

    /**
     * 上一页下一页
     *
     * @param id
     * @param upBelow
     * @return
     */
    @Override
    public MdmNoticeRespVo queryMiniPrgByUpBelow(String id, String upBelow) {
        AssertUtils.isNotEmpty(id, "主键不能为空");
        AssertUtils.isNotEmpty(upBelow, "翻页类型不能为空");
        UserRedis user = UserUtils.getUser();
        MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo();
        mdmOrgReqVo.setOrgCode(user.getOrgcode());

        String noticeId = mdmNoticeMapper.queryMinPrgUpOrBelow(id, upBelow, user.getOrgcode());
        if (StringUtils.isEmpty(noticeId)) {
            if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(upBelow)) {
                AssertUtils.isNotEmpty(noticeId, "已经是最后一条了");
            } else {
                AssertUtils.isNotEmpty(noticeId, "已经是第一条了");
            }
        }
        if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(noticeId)){
            if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(upBelow)) {
                throw new BusinessException("已经是最后一条了");
            } else {
                throw new BusinessException("已经是第一条了");
            }
        }
        MdmNoticeRespVo mdmNoticeRespVo = this.queryMiniPrg(noticeId);
        this.setNoticeStatus(Arrays.asList(mdmNoticeRespVo));
        List<MdmNoticeFileRespVo> miniList = noticeFileMapper.findMiniList(mdmNoticeRespVo.getNoticeCode());
        mdmNoticeRespVo.setNoticeFileRespVos(miniList);
        iMdmNoticeReadService.save(mdmNoticeRespVo.getNoticeCode());
        return mdmNoticeRespVo;
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return MdmNoticeRespVo
     */
    @Override
    public MdmNoticeRespVo query(MdmNoticeReqVo reqVo) {
        if (null == reqVo || (StringUtils.isEmpty(reqVo.getId())) && CollectionUtil.listEmpty(reqVo.getIds())) {
            return new MdmNoticeRespVo();
        }
        List<MdmNoticeRespVo> list = this.findList(reqVo).getData();
        this.setNoticeStatus(list);
        if (CollectionUtils.isEmpty(list)) {
            return new MdmNoticeRespVo();
        }
        MdmNoticeRespVo mdmNoticeRespVo = list.get(0);
        //查询公告附件
        MdmNoticeFileReqVo fileReqVo = new MdmNoticeFileReqVo();
        fileReqVo.setNoticeCode(mdmNoticeRespVo.getNoticeCode());
        fileReqVo.setPageSize(-1);
        List<MdmNoticeFileRespVo> fileRespVos =
                iMdmNoticeFileService.findList(fileReqVo).getData();
        if (CollectionUtil.listNotEmptyNotSizeZero(fileRespVos)) {
            mdmNoticeRespVo.setNoticeFileRespVos(fileRespVos);
        }
        //查询公告发布范围
        MdmNoticeReceivingReqVo receivingReqVo = new MdmNoticeReceivingReqVo();
        receivingReqVo.setNoticeCode(mdmNoticeRespVo.getNoticeCode());
        receivingReqVo.setPageSize(-1);
        List<MdmNoticeReceivingRespVo> receivingRespVos = iMdmNoticeReceivingService.findList(receivingReqVo).getData();
        mdmNoticeRespVo.setNoticeReceivingRespVos(receivingRespVos);
        //查询评论
        MdmNoticeCommentReqVo noticeCommentReqVo = new MdmNoticeCommentReqVo();
        noticeCommentReqVo.setNoticeId(reqVo.getId());
        noticeCommentReqVo.setPageSize(-1);
        List<MdmNoticeCommentRespVo> commentRespVos = iMdmNoticeCommentService.findCommentList(noticeCommentReqVo).getData();
        mdmNoticeRespVo.setNoticeCommentRespVos(commentRespVos);
        iMdmNoticeReadService.save(mdmNoticeRespVo.getNoticeCode());
        return mdmNoticeRespVo;
    }

    /**
     * 新增,编辑,校验
     *
     * @param reqVo
     */
    public void saveCheck(MdmNoticeReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getNoticeTitle(), "请输入公告标题");
        AssertUtils.isNotEmpty(reqVo.getNoticeType(), "请选择公告类型");
        AssertUtils.isNotEmpty(reqVo.getBeginDate(), "有效开始时间不能为空");
        AssertUtils.isNotEmpty(reqVo.getNoticeContent(), "请输入公告内容");
        AssertUtils.isNotEmpty(reqVo.getNoticeReceivingReqVos(), "请选择公告范围");
        LocalDateTime beginDate = null;
        try {
            beginDate = LocalDateTime.parse(reqVo.getBeginDate(), CrmDateUtils.yyyyMMddHHmmss);
        } catch (Exception e) {
            throw new BusinessException("有效开始时间格式错误");
        }
        AssertUtils.isNotEmpty(reqVo.getEndDate(), "有效结束时间不能为空");
        LocalDateTime endDate = null;
        try {
            endDate = LocalDateTime.parse(reqVo.getEndDate(), CrmDateUtils.yyyyMMddHHmmss);
        } catch (Exception e) {
            throw new BusinessException("有效结束时间格式错误");
        }
        if (!(beginDate.isBefore(endDate))) {
            throw new BusinessException("有效开始时间不能大于有效结束时间");
        }
        AssertUtils.isNotEmpty(reqVo.getNoticeContent(), "公告内容不能为空");
        //公告置顶状态默认为否
        if(StringUtils.isEmpty(reqVo.getNoticeTopState())){
            reqVo.setNoticeTopState(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString());
        }
        //公告发布状态默认为发布
        if(StringUtils.isEmpty(reqVo.getNoticeReleaseStatus())){
            reqVo.setNoticeReleaseStatus(YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString());
        }
        //公告可评论状态默认为不可评论
        if(StringUtils.isEmpty(reqVo.getCommentState())){
            reqVo.setCommentState(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString());
        }
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmNoticeReqVo reqVo) {
        this.saveCheck(reqVo);
        MdmNoticeEntity entity = CrmBeanUtil.copy(reqVo, MdmNoticeEntity.class);
        //公告编码
        String code = CodeUtil.generateCode(MdmCodeEnum.NoticeCode.NOTICE_CODE.getVal());
        entity.setNoticeCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        //置顶状态
        entity.setNoticeReleaseStatus(reqVo.getNoticeReleaseStatus());
        if (YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString().equals(reqVo.getNoticeTopState())){
            String topTime = String.valueOf(System.currentTimeMillis());
            entity.setNoticeTopTime(topTime);
        }else {
            entity.setNoticeTopTime(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        //发布状态
        entity.setNoticeReleaseStatus(reqVo.getNoticeReleaseStatus());
        this.save(entity);
        this.saveData(reqVo, entity);
    }

    /**
     * 保存业务明细数据
     *
     * @param reqVo
     * @param entity
     */
    public void saveData(MdmNoticeReqVo reqVo, MdmNoticeEntity entity) {
        //公告发布范围
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getNoticeReceivingReqVos())) {
            List<MdmNoticeReceivingEntity> receivingEntityList = reqVo.getNoticeReceivingReqVos().stream().map(o -> {
                MdmNoticeReceivingEntity receivingEntity = new MdmNoticeReceivingEntity();
                CrmBeanUtil.copyProperties(o, receivingEntity);
                receivingEntity.setId(null);
                receivingEntity.setNoticeCode(entity.getNoticeCode());
                return receivingEntity;
            }).collect(Collectors.toList());
            iMdmNoticeReceivingService.saveBatch(receivingEntityList);
        }
        //公告附件
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getNoticeFileReqVos())) {
            List<MdmNoticeFileEntity> fileEntityList = reqVo.getNoticeFileReqVos().stream().map(o -> {
                MdmNoticeFileEntity fileEntity = new MdmNoticeFileEntity();
                CrmBeanUtil.copyProperties(o, fileEntity);
                fileEntity.setId(null);
                fileEntity.setNoticeCode(entity.getNoticeCode());
                return fileEntity;
            }).collect(Collectors.toList());
            iMdmNoticeFileService.saveBatch(fileEntityList);
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    public void update(MdmNoticeReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getId())) {
            throw new BusinessException("数据主键不能为空");
        }
        MdmNoticeEntity entity = this.getById(reqVo.getId());
        if (Objects.isNull(entity)) {
            throw new BusinessException("公告数据不存在");
        }
        this.saveCheck(reqVo);
        CrmBeanUtil.copyProperties(reqVo, entity);
        if (YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString().equals(reqVo.getNoticeTopState())){
            entity.setNoticeTopTime(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        if (YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString().equals(reqVo.getNoticeTopState())){
            String topTime = String.valueOf(System.currentTimeMillis());
            entity.setNoticeTopTime(topTime);
        }
        entity.setNoticeReleaseStatus(reqVo.getNoticeReleaseStatus());
        this.updateById(entity);
        reqVo.setIds(Lists.newArrayList(entity.getId()));
        this.deleteBatchDate(reqVo);
        this.saveData(reqVo, entity);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    public void deleteBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        MdmNoticeReqVo reqVo = new MdmNoticeReqVo() {{
            this.setIds(ids);
        }};
        reqVo.setPageSize(-1);
        List<MdmNoticeRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("公告数据不存在");
        }
        for (MdmNoticeRespVo mdmNoticeRespVo : list){
            String noticeReleaseStatus = mdmNoticeRespVo.getNoticeReleaseStatus();
            if (YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString().equals(noticeReleaseStatus)){
                throw new BusinessException("公告：" + mdmNoticeRespVo.getNoticeTitle() +  " 已经发布，不能删除");
            }
        }
        mdmNoticeMapper.deleteProductsByParams(reqVo);
        List<String> noticeCodes = list.stream().map(MdmNoticeRespVo::getNoticeCode).collect(Collectors.toList());
        reqVo.setNoticeCodes(noticeCodes);
        this.deleteBatchDate(reqVo);
    }


    public void deleteBatchDate(MdmNoticeReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getNoticeCode()) && CollectionUtil.listEmpty(reqVo.getNoticeCodes())) {
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isEmpty(reqVo.getId()) && CollectionUtil.listEmpty(reqVo.getIds())) {
            throw new BusinessException("数据主键不能为空");
        }

        MdmNoticeFileReqVo mdmNoticeFileReqVo = new MdmNoticeFileReqVo();
        mdmNoticeFileReqVo.setNoticeCode(reqVo.getNoticeCode());
        mdmNoticeFileReqVo.setNoticeCodes(reqVo.getNoticeCodes());
        noticeFileMapper.deleteProductsByParams(mdmNoticeFileReqVo);

        MdmNoticeReceivingReqVo mdmNoticeReceivingReqVo = new MdmNoticeReceivingReqVo();
        mdmNoticeReceivingReqVo.setNoticeCode(reqVo.getNoticeCode());
        mdmNoticeReceivingReqVo.setNoticeCodes(reqVo.getNoticeCodes());
        noticeReceivingMapper.deleteProductsByParams(mdmNoticeReceivingReqVo);

        //通过公告id查询出对应评论id
        List<String> noticeIds = reqVo.getIds();
        List<String> commentIds = noticeCommentMapper.selectCommentIds(noticeIds);
        if (!commentIds.isEmpty()){
            noticeCommentLikesMapper.deleteProductCommentIds(commentIds);
        }
        noticeCommentMapper.deleteProductsByNoticeIds(noticeIds);
    }


    /**
     * 小程序删除 校验发布状态，已发布状态不能被删除
     * @param id
     * @return
     */
    @Override
    public void deleteById(String id) {
        AssertUtils.isNotEmpty(id,"数据主键不能为空");
        MdmNoticeEntity mdmNoticeEntity = mdmNoticeMapper.selectById(id);
        AssertUtils.isNotNull(mdmNoticeEntity,"公告可能已经被删除了");
        if (YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString().equals(mdmNoticeEntity.getNoticeReleaseStatus())){
            throw new BusinessException("公告：" + mdmNoticeEntity.getNoticeTitle() +  " 已经发布，不能删除");
        }
        MdmNoticeReqVo reqVo = CrmBeanUtil.copy(mdmNoticeEntity, MdmNoticeReqVo.class);

        MdmNoticeFileReqVo mdmNoticeFileReqVo = new MdmNoticeFileReqVo();
        mdmNoticeFileReqVo.setNoticeCode(reqVo.getNoticeCode());
        noticeFileMapper.deleteProductsByParams(mdmNoticeFileReqVo);

        MdmNoticeReceivingReqVo mdmNoticeReceivingReqVo = new MdmNoticeReceivingReqVo();
        mdmNoticeReceivingReqVo.setNoticeCode(reqVo.getNoticeCode());
        noticeReceivingMapper.deleteProductsByParams(mdmNoticeReceivingReqVo);

        //通过公告id查询出对应评论id
        String noticeId = reqVo.getId();
        List<String> noticeIds = Arrays.asList(noticeId);
        List<String> commentIds = noticeCommentMapper.selectCommentIds(noticeIds);
        if (!commentIds.isEmpty()){
            noticeCommentLikesMapper.deleteProductCommentIds(commentIds);
        }
        noticeCommentMapper.deleteProductsByNoticeIds(noticeIds);
        mdmNoticeMapper.deleteProductsByParams(reqVo);
    }

    /**
     * 登录角色判断
     * @return
     */
    @Override
    public Boolean isNoticeAdmin() {
        UserRedis user = UserUtils.getUser();
        if ("admin".equals(user.getUsername())){
            return true;
        }
        String positionCode = user.getPoscode();
        //通过职位编码查询对应权限角色
        List<String> positionCodeList = new ArrayList<>();
        positionCodeList.add(positionCode);
        List<String> roleList = mdmPositionRoleService.findRoleCodeByPosition(positionCodeList);
        Map<String,String> roleMap = roleList.stream().collect(HashMap::new,(k, v) -> k.put(v, v), HashMap::putAll);
        if (roleMap.containsKey("ADMIN")||roleMap.containsKey("公告管理员")){
            return true;
        }
        return false;
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    public void enableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<MdmNoticeEntity> mdmNoticeEntities = mdmNoticeMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(mdmNoticeEntities)) {
            mdmNoticeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(mdmNoticeEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    public void disableBatch(List<String> ids) {
        MdmNoticeReqVo reqVo = new MdmNoticeReqVo();
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为禁用
        List<MdmNoticeEntity> mdmNoticeEntities = mdmNoticeMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(mdmNoticeEntities)) {
            mdmNoticeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(mdmNoticeEntities);
    }

    /**
     * 置顶
     * @param id
     */
    @Override
    public void top(String id) {
        MdmNoticeEntity mdmNoticeEntity = this.doCheck(id);
        String topTime = String.valueOf(System.currentTimeMillis());
        mdmNoticeEntity.setNoticeTopState(YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString());
        mdmNoticeEntity.setNoticeTopTime(topTime);
        UpdateWrapper<MdmNoticeEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        mdmNoticeMapper.update(mdmNoticeEntity,updateWrapper);
    }

    /**
     * 取消置顶
     * @param id
     */
    @Override
    public void cancelTop(String id) {
        MdmNoticeEntity mdmNoticeEntity = this.doCheck(id);
        mdmNoticeEntity.setNoticeTopState(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString());
        mdmNoticeEntity.setNoticeTopTime(YesNoEnum.yesNoEnum.ZERO.getValue());
        UpdateWrapper<MdmNoticeEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        mdmNoticeMapper.update(mdmNoticeEntity,updateWrapper);
    }

    /**
     * 取消发布
     * @param id
     */
    @Override
    public void cancelRelease(String id) {
        MdmNoticeEntity mdmNoticeEntity = this.doCheck(id);
        mdmNoticeEntity.setNoticeReleaseStatus(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString());
        UpdateWrapper<MdmNoticeEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        mdmNoticeMapper.update(mdmNoticeEntity,updateWrapper);
    }

    /**
     * 校验
     * @param id
     * @return
     */
    public MdmNoticeEntity doCheck(String id){
        AssertUtils.isNotEmpty(id, "数据主键不能为空");
        MdmNoticeEntity mdmNoticeEntity = mdmNoticeMapper.selectById(id);
        AssertUtils.isNotNull(mdmNoticeEntity, "公告数据不存在");
        return mdmNoticeEntity;
    }

    /**
     * 根据id发布
     * @param id
     */
    @Override
    public void releaseNoticeById(String id) {
        MdmNoticeEntity mdmNoticeEntity = this.doCheck(id);
        mdmNoticeEntity.setNoticeReleaseStatus(YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString());
        UpdateWrapper<MdmNoticeEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        mdmNoticeMapper.update(mdmNoticeEntity,updateWrapper);
    }
}
