package com.biz.crm.org.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.org.req.MdmOrgRegionReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRegionRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.org.mapper.MdmOrgRegionMapper;
import com.biz.crm.org.model.MdmOrgEntity;
import com.biz.crm.org.model.MdmOrgRegionEntity;
import com.biz.crm.org.service.MdmOrgRegionService;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.region.model.MdmRegionEntity;
import com.biz.crm.region.service.MdmRegionService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.ParamUtil;
import com.biz.crm.util.StringUtils;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 组织和行政区域关联表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 10:42:46
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmOrgRegionServiceExpandImpl")
public class MdmOrgRegionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmOrgRegionMapper, MdmOrgRegionEntity> implements MdmOrgRegionService {

    @Resource
    private MdmOrgRegionMapper mdmOrgRegionMapper;
    @Resource
    private MdmRegionService mdmRegionService;
    @Resource
    private MdmOrgService mdmOrgService;

    @Override
    public List<MdmOrgRegionRespVo> findOrgRegionList(MdmOrgRegionReqVo reqVo) {
        Assert.hasText(reqVo.getOrgCode(), "组织编码不能为空");
        return mdmOrgRegionMapper.findOrgRegionList(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrgRegion(MdmOrgRegionReqVo reqVo) {
        Assert.hasText(reqVo.getOrgCode(), "缺失组织编码");
        MdmOrgRespVo org = mdmOrgService.detail(null, reqVo.getOrgCode());
        Assert.notNull(org, "组织编码无效");
        this.lambdaUpdate()
                .eq(MdmOrgRegionEntity::getOrgCode, reqVo.getOrgCode())
                .remove();
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRegionList())) {
            List<MdmOrgRegionReqVo> regionList = this.verify(reqVo);
            List<MdmOrgRegionEntity> list = CrmBeanUtil.copyList(regionList, MdmOrgRegionEntity.class);
            this.saveBatch(list);
        }
    }

    @Override
    public List<MdmOrgRegionRespVo> findOrgRegionByOrgCodeList(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgRegionEntity> list = this.lambdaQuery()
                    .in(MdmOrgRegionEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmOrgRegionRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    /**
     * 校验参数
     *
     * @param reqVo
     */
    private List<MdmOrgRegionReqVo> verify(MdmOrgRegionReqVo reqVo) {

        List<MdmOrgRegionReqVo> regionList = reqVo.getRegionList();
        List<MdmOrgRegionReqVo> resultList = new ArrayList<>();
        //记录区域编码
        Set<String> regionCodeList = new HashSet<>(16);
        regionCodeList.addAll(regionList.stream().filter(item -> StringUtils.isNotEmpty(item.getCountryCode())).map(MdmOrgRegionReqVo::getCountryCode).collect(Collectors.toSet()));
        regionCodeList.addAll(regionList.stream().filter(item -> StringUtils.isNotEmpty(item.getProvinceCode())).map(MdmOrgRegionReqVo::getProvinceCode).collect(Collectors.toSet()));
        regionCodeList.addAll(regionList.stream().filter(item -> StringUtils.isNotEmpty(item.getCityCode())).map(MdmOrgRegionReqVo::getCityCode).collect(Collectors.toSet()));
        regionCodeList.addAll(regionList.stream().filter(item -> StringUtils.isNotEmpty(item.getDistrictCode())).map(MdmOrgRegionReqVo::getDistrictCode).collect(Collectors.toSet()));
        regionCodeList.addAll(regionList.stream().filter(item -> StringUtils.isNotEmpty(item.getTownshipCode())).map(MdmOrgRegionReqVo::getTownshipCode).collect(Collectors.toSet()));

        final Map<String, String> regionMap = mdmRegionService.getRegionCodeNameMap(new ArrayList<>(regionCodeList));

        String countryShow = ParamUtil.getParameterValue(ParameterParam.MDM_REGION_SHOW_CONTROL_COUNTRY);
        Assert.hasText(countryShow, "全局参数异常，请联系管理员");
        for (int i = 0; i < regionList.size(); i++) {
            try {
                MdmOrgRegionReqVo item = regionList.get(i);
                MdmOrgRegionReqVo newItem = new MdmOrgRegionReqVo();
                resultList.add(newItem);
                newItem.setOrgCode(reqVo.getOrgCode());

                if (YesNoEnum.yesNoEnum.Y.getValue().equals(countryShow)) {
                    Assert.hasText(item.getCountryCode(), "国家必填");
                    Assert.isTrue(regionMap.containsKey(item.getCountryCode()), "国家编码有误");
                    newItem.setCountryCode(item.getCountryCode());
                } else {
                    Assert.hasText(item.getProvinceCode(), "省份必填");
//                    Assert.isTrue(regionMap.containsKey(item.getProvinceCode()), "省份编码有误");
//                    newItem.setProvinceCode(item.getProvinceCode());
                }

                if (StringUtils.isEmpty(item.getProvinceCode())) {
                    continue;
                }
                Assert.isTrue(regionMap.containsKey(item.getProvinceCode()), "省份编码有误");
                newItem.setProvinceCode(item.getProvinceCode());

                if (StringUtils.isEmpty(item.getCityCode())) {
                    continue;
                }
                Assert.isTrue(regionMap.containsKey(item.getCityCode()), "市编码有误");
                newItem.setCityCode(item.getCityCode());
                if (StringUtils.isEmpty(item.getDistrictCode())) {
                    continue;
                }
                Assert.isTrue(regionMap.containsKey(item.getDistrictCode()), "区县编码有误");
                newItem.setDistrictCode(item.getDistrictCode());
                if (StringUtils.isEmpty(item.getTownshipCode())) {
                    continue;
                }
                Assert.isTrue(regionMap.containsKey(item.getTownshipCode()), "乡镇编码有误");
                newItem.setTownshipCode(item.getTownshipCode());

            } catch (BusinessException e) {
                throw new BusinessException("第" + (i + 1) + "行：" + e.getMsg());
            }
        }

        Map<String, Integer> uniqueKeyMap = new LinkedHashMap<>(16);
        for (int i = 0; i < resultList.size(); i++) {
            try {
                MdmOrgRegionReqVo item = resultList.get(i);
                String key = generateUniqueKey(item.getCountryCode(), item.getProvinceCode(), item.getCityCode(), item.getDistrictCode(), item.getTownshipCode());
                for (Map.Entry<String, Integer> entry :
                        uniqueKeyMap.entrySet()) {
                    if (entry.getKey().equals(key)) {
                        throw new BusinessException("与第" + entry.getValue() + "行重复");
                    }
                    if (entry.getKey().startsWith(key) || key.startsWith(entry.getKey())) {
                        throw new BusinessException("与第" + entry.getValue() + "行存在包含关系");
                    }
                }
                uniqueKeyMap.put(key, (i + 1));

                List<MdmOrgRegionEntity> list = this.lambdaQuery()
                        .eq(StringUtils.isNotEmpty(item.getCountryCode()), MdmOrgRegionEntity::getCountryCode, item.getCountryCode())
                        .eq(StringUtils.isNotEmpty(item.getProvinceCode()), MdmOrgRegionEntity::getProvinceCode, item.getProvinceCode())
                        .eq(StringUtils.isNotEmpty(item.getCityCode()), MdmOrgRegionEntity::getCityCode, item.getCityCode())
                        .eq(StringUtils.isNotEmpty(item.getDistrictCode()), MdmOrgRegionEntity::getDistrictCode, item.getDistrictCode())
                        .eq(StringUtils.isNotEmpty(item.getTownshipCode()), MdmOrgRegionEntity::getTownshipCode, item.getTownshipCode())
                        .list()
                        .stream().filter(x -> !reqVo.getOrgCode().equals(x.getOrgCode()))
                        .collect(Collectors.toList());

                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    Map<String, MdmOrgRegionEntity> uniqueKeyList = list.stream().collect(Collectors.toMap(k -> {
                        return generateUniqueKey(k.getCountryCode(), k.getProvinceCode(), k.getCityCode(), k.getDistrictCode(), k.getTownshipCode());
                    }, v -> v));
                    if (uniqueKeyList.containsKey(key)) {
                        MdmOrgRegionEntity orgRegion = uniqueKeyList.get(key);
                        String useOrgName = list.get(0).getOrgCode();
                        final Map<String, String> orgCodeNameMap = mdmOrgService.getOrgCodeNameMap(list.stream().map(MdmOrgRegionEntity::getOrgCode).collect(Collectors.toList()));
                        for (Map.Entry<String, String> entry :
                                orgCodeNameMap.entrySet()) {
                            useOrgName = entry.getValue();
                            break;
                        }
                        String name = (StringUtils.isEmpty(orgRegion.getCountryCode()) ? "" : regionMap.get(orgRegion.getCountryCode()))
                                + (StringUtils.isEmpty(orgRegion.getProvinceCode()) ? "" : regionMap.get(orgRegion.getProvinceCode()))
                                + (StringUtils.isEmpty(orgRegion.getCityCode()) ? "" : regionMap.get(orgRegion.getCityCode()))
                                + (StringUtils.isEmpty(orgRegion.getDistrictCode()) ? "" : regionMap.get(orgRegion.getDistrictCode()))
                                + (StringUtils.isEmpty(orgRegion.getTownshipCode()) ? "" : regionMap.get(orgRegion.getTownshipCode()));
                        throw new BusinessException(name + "已经被[" + useOrgName + "]关联");
                    }
                }
            } catch (BusinessException e) {
                throw new BusinessException("第" + (i + 1) + "行：" + e.getMsg());
            }

        }
        return resultList;
    }

    private String generateUniqueKey(String countryCode,String provinceCode,String cityCode,String districtCode,String townshipCode) {
        return "_"
                + (StringUtils.isEmpty(countryCode) ? "" : countryCode + "_")
                + (StringUtils.isEmpty(provinceCode) ? "" : provinceCode + "_")
                + (StringUtils.isEmpty(cityCode) ? "" : cityCode + "_")
                + (StringUtils.isEmpty(districtCode) ? "" : districtCode + "_")
                + (StringUtils.isEmpty(townshipCode) ? "" : townshipCode + "_");
    }


}
