package com.biz.crm.org.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.org.req.MdmOrgPageSelectReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgSearchReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgSelectReqVo;
import com.biz.crm.nebular.mdm.org.resp.*;
import com.biz.crm.nebular.mdm.position.req.MdmPositionUserOrgReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.org.mapper.MdmOrgMapper;
import com.biz.crm.org.model.MdmOrgEntity;
import com.biz.crm.org.service.MdmOrgRegionService;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.terminal.service.MdmTerminalROrgService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.assertj.core.util.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 组织表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmOrgServiceExpandImpl")
public class MdmOrgServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmOrgMapper, MdmOrgEntity> implements MdmOrgService {

    @Resource
    private MdmOrgMapper mdmOrgMapper;
    @Resource
    private MdmPositionService mdmPositionService;
    @Resource
    private MdmOrgRegionService mdmOrgRegionService;
    @Resource
    private MdmCustomerROrgService mdmCustomerROrgService;
    @Resource
    private MdmTerminalROrgService mdmTerminalROrgService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;


    @Override
    public PageResult<MdmOrgRespVo> findList(MdmOrgReqVo reqVo) {
        Page<MdmOrgRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (!StringUtils.isEmpty(reqVo.getCode())) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, reqVo.getCode())
                    .one();
            if (one != null) {
                reqVo.setRuleCode(one.getRuleCode());
            }
        }
        log.info("开始查询数据库");
        List<MdmOrgRespVo> list = mdmOrgMapper.findList(page, reqVo);
        log.info("结束查询数据库");
        return PageResult.<MdmOrgRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmOrgRespVo> findAllPage(MdmOrgReqVo reqVo) {
        Page<MdmOrgRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (!StringUtils.isEmpty(reqVo.getCode())) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, reqVo.getCode())
                    .one();
            if (one != null) {
                reqVo.setRuleCode(one.getRuleCode());
            }
        }
        log.info("开始查询数据库");
        List<MdmOrgRespVo> list = mdmOrgMapper.findAllPage(page, reqVo);
        log.info("结束查询数据库");
        return PageResult.<MdmOrgRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmOrgRespVo> pageSelectList(MdmOrgPageSelectReqVo reqVo) {
        Page<MdmOrgRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getAllUnderOrgCodeListIncludeSelf())) {
            List<String> listIncludeSelf = OrgUtil.getChildrenOrgCodeListIncludeSelf(reqVo.getAllUnderOrgCodeListIncludeSelf());
            if (CollectionUtil.listNotEmptyNotSizeZero(listIncludeSelf)) {
                reqVo.setAllUnderOrgCodeListIncludeSelf(listIncludeSelf);
            }
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getAllUnderOrgCodeListExcludeSelf())) {
            List<String> listExcludeSelf = OrgUtil.getChildrenOrgCodeListExcludeSelf(reqVo.getAllUnderOrgCodeListExcludeSelf());
            if (CollectionUtil.listNotEmptyNotSizeZero(listExcludeSelf)) {
                reqVo.setAllUnderOrgCodeListExcludeSelf(listExcludeSelf);
            }
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getAllUnderOrgCodeListExcludeAnySelf())) {
            List<String> listExcludeAnySelf = OrgUtil.getChildrenOrgCodeListExcludeAnySelf(reqVo.getAllUnderOrgCodeListExcludeAnySelf());
            if (CollectionUtil.listNotEmptyNotSizeZero(listExcludeAnySelf)) {
                reqVo.setAllUnderOrgCodeListExcludeAnySelf(listExcludeAnySelf);
            }
        }
        List<MdmOrgRespVo> list = mdmOrgMapper.pageSelectList(page, reqVo);
        return PageResult.<MdmOrgRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmOrgRespVo detail(String id, String orgCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmOrgEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                    .one();
            if (one != null) {
                MdmOrgRespVo org = CrmBeanUtil.copy(one, MdmOrgRespVo.class);
                if (StringUtils.isNotEmpty(org.getParentCode())) {
                    MdmOrgEntity parent = this.lambdaQuery()
                            .eq(MdmOrgEntity::getOrgCode, org.getParentCode())
                            .select(MdmOrgEntity::getOrgName)
                            .one();
                    if (parent != null) {
                        org.setParentName(parent.getOrgName());
                    }
                }
                return org;
            }
        }
        return null;
    }

    @Override
    public MdmOrgRespVo queryByOrgCode(String orgCode) {
        if (StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, orgCode)
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmOrgRespVo.class);
            }
        }
        return null;
    }

    @Override
    public List<MdmOrgRespVo> queryBatchByOrgCodeList(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> list = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmOrgRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> getOrgCodeNameMap(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> list = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.toMap(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName));
            }
        }
        return Collections.emptyMap();
    }

    @Override
    public String getOrgRuleCode(String orgCode) {
        if (StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, orgCode)
                    .select(MdmOrgEntity::getRuleCode)
                    .one();
            if (one != null) {
                return one.getRuleCode();
            }
        }
        return null;
    }

    @Override
    public List<MdmOrgRespVo> detailBatch(List<String> ids, List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> entityList = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmOrgEntity::getId, ids)
                    .or()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(orgCodeList), MdmOrgEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                List<MdmOrgRespVo> list = CrmBeanUtil.copyList(entityList, MdmOrgRespVo.class);
                List<String> parentCodeList = entityList.stream().map(MdmOrgEntity::getParentCode).distinct().filter(StringUtils::isNotEmpty).collect(Collectors.toList());
                if (CollectionUtil.listNotEmptyNotSizeZero(parentCodeList)) {
                    final Map<String, String> collect = this.getOrgCodeNameMap(parentCodeList);
                    if (!collect.isEmpty()) {
                        list.forEach(item -> {
                            if (StringUtils.isNotEmpty(item.getParentCode())) {
                                item.setParentName(collect.get(item.getParentCode()));
                            }
                        });
                    }
                }
                return list;
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgRedisVo> findRedisListByOrgCodeList(List<String> orgCodeList) {
        List<MdmOrgRespVo> list = detailBatch(null, orgCodeList);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return CrmBeanUtil.copyList(list, MdmOrgRedisVo.class);
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgParentChildrenRedisVo> findRedisOrgParentListByCodeList(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> list = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(list.stream().map(MdmOrgEntity::getRuleCode).collect(Collectors.toList()));
                List<MdmOrgEntity> parentList = this.lambdaQuery()
                        .in(MdmOrgEntity::getRuleCode, parentRuleCodes)
                        .list();
                Set<String> collect = parentList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmOrgEntity::getParentCode).collect(Collectors.toSet());
                Map<String, String> parentOrgCodeNameMap = new HashMap<>(16);
                if (!collect.isEmpty()) {
                    parentOrgCodeNameMap.putAll(this.lambdaQuery()
                            .in(MdmOrgEntity::getOrgCode, collect)
                            .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName)
                            .list()
                            .stream()
                            .collect(Collectors.toMap(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName)));
                }
                List<MdmOrgRedisVo> parentRedisVoList = parentList.stream().map(item -> {
                    MdmOrgRedisVo copy = CrmBeanUtil.copy(item, MdmOrgRedisVo.class);
                    if (StringUtils.isNotEmpty(copy.getParentCode())) {
                        if (parentOrgCodeNameMap.containsKey(copy.getParentCode())) {
                            copy.setParentName(parentOrgCodeNameMap.get(copy.getParentCode()));
                        }
                    }
                    return copy;
                }).collect(Collectors.toList());
                List<MdmOrgParentChildrenRedisVo> result = new ArrayList<>();
                for (MdmOrgEntity item :
                        parentList) {
                    MdmOrgParentChildrenRedisVo redisVo = new MdmOrgParentChildrenRedisVo();
                    redisVo.setOrgCode(item.getOrgCode());
                    redisVo.setList(parentRedisVoList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                    result.add(redisVo);
                }
                return result;
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgParentChildrenRedisVo> findRedisOrgChildrenListByCodeList(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> orgList = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList)) {
                List<String> likeRightRuleCodeList = orgList.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                List<MdmOrgRespVo> childrenList = mdmOrgMapper.findOrgChildrenByRuleCodeList(likeRightRuleCodeList);
                if (CollectionUtil.listNotEmptyNotSizeZero(childrenList)) {
                    List<MdmOrgRedisVo> redisList = CrmBeanUtil.copyList(childrenList, MdmOrgRedisVo.class);
                    List<MdmOrgParentChildrenRedisVo> resultList = new ArrayList<>();
                    for (MdmOrgEntity item :
                            orgList) {
                        MdmOrgParentChildrenRedisVo redisVo = new MdmOrgParentChildrenRedisVo();
                        redisVo.setOrgCode(item.getOrgCode());
                        redisVo.setList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                        resultList.add(redisVo);
                    }
                    return resultList;
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgReloadRedisVo> findRedisReloadList() {
        final List<MdmOrgEntity> all = this.lambdaQuery().list();
        if (CollectionUtil.listNotEmptyNotSizeZero(all)) {
            final Map<String, String> orgCodeNameMap = all.stream().collect(Collectors.toMap(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName));
            final List<MdmOrgRedisVo> redisList = all.stream().map(item -> {
                MdmOrgRedisVo copy = CrmBeanUtil.copy(item, MdmOrgRedisVo.class);
                if (StringUtils.isNotEmpty(copy.getParentCode())) {
                    copy.setParentName(orgCodeNameMap.get(copy.getParentCode()));
                }
                return copy;
            }).collect(Collectors.toList());
            return redisList.stream().map(item -> {
                MdmOrgReloadRedisVo redisVo = new MdmOrgReloadRedisVo();
                redisVo.setOrgCode(item.getOrgCode());
                redisVo.setSelf(item);
                redisVo.setParentList(redisList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                redisVo.setChildrenList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                return redisVo;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmOrgReqVo reqVo) {
        reqVo.setId(null);
        Assert.hasText(reqVo.getOrgName(), "组织名称不能为空");
        MdmOrgEntity entity = CrmBeanUtil.copy(reqVo, MdmOrgEntity.class);
        int levelNum = 1;
        if (StringUtils.isNotEmpty(reqVo.getParentCode())) {
            MdmOrgEntity parent = this.lambdaQuery().eq(MdmOrgEntity::getOrgCode, reqVo.getParentCode()).one();
            Assert.notNull(parent, "上级组织不存在");
            Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(parent.getEnableStatus()), "上级组织被禁用");
            levelNum = parent.getLevelNum() + 1;
        }
        //生成降维编码
        entity.setRuleCode(generateRuleCodeByParentCode(entity.getParentCode()));
        entity.setLevelNum(levelNum);
        if (StringUtils.isEmpty(entity.getOrgCode())) {
            entity.setOrgCode(CodeUtil.generateCode(CodeRuleEnum.MDM_ORG_CODE.getCode()));
        } else {
            List<MdmOrgEntity> list = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, reqVo.getOrgCode())
                    .select(MdmOrgEntity::getOrgCode)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "组织编码已经存在");
        }
        this.save(entity);
        OrgUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(((String) menuObject), entity.getId(), entity.getOrgName(), CrmBeanUtil.copy(reqVo, MdmOrgRespVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmOrgReqVo reqVo) {
        String id = reqVo.getId();
        Assert.hasText(id, "主键不能为空");
        Assert.hasText(reqVo.getOrgCode(), "组织编码不能为空");
        Assert.hasText(reqVo.getOrgName(), "组织名称不能为空");
        MdmOrgEntity entity = this.getById(id);
        MdmOrgEntity old = new MdmOrgEntity();
        CrmBeanUtil.copyProperties(entity,old);

        Assert.notNull(entity, "组织不存在");
        Assert.isTrue(entity.getOrgCode().equals(reqVo.getOrgCode()), "组织编码不能修改");
        String parentCode = reqVo.getParentCode();
        //是否重置降维编码 取决于上级有无变化
        boolean reloadRuleCode = false;
        int levelNum = 1;
        //启用状态有无变化
        String enableStatusChangeTo = "";
        if (StringUtils.isNotEmpty(parentCode)) {
            Assert.isTrue(!entity.getOrgCode().equals(parentCode), "上级组织不能为当前组织");
            MdmOrgEntity parent = this.lambdaQuery().eq(MdmOrgEntity::getOrgCode, parentCode).one();
            Assert.notNull(parent, "上级组织不存在");
            Assert.isTrue(!parent.getRuleCode().startsWith(entity.getRuleCode()), "上级组织不能是当前组织的下级组织");
            levelNum = parent.getLevelNum() + 1;
        }

        //判断上级是否发生变化
        if (!(entity.getParentCode() == null ? "" : entity.getParentCode()).equals(reqVo.getParentCode() == null ? "" : reqVo.getParentCode())) {
            reloadRuleCode = true;
        }
        //判断启用状态是否发生变化
        if (!entity.getEnableStatus().equals(reqVo.getEnableStatus())) {
            enableStatusChangeTo = reqVo.getEnableStatus();
        }

        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        if (reloadRuleCode) {
            //更新当前及下级降维编码
            String ruleCode = generateRuleCodeByParentCode(reqVo.getParentCode());
            updateCurAndChildren(entity.getOrgCode(), ruleCode, levelNum);
        }
        if (StringUtils.isNotEmpty(enableStatusChangeTo)) {
            if (CrmEnableStatusEnum.ENABLE.getCode().equals(enableStatusChangeTo)) {
                this.enableBatch(Collections.singletonList(entity.getId()));
            } else if (CrmEnableStatusEnum.DISABLE.getCode().equals(enableStatusChangeTo)) {
                this.disableBatch(Collections.singletonList(entity.getId()));
            } else {
                throw new BusinessException("启用状态有误");
            }
        }
        OrgUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForUpdate(((String) menuObject), entity.getId(), entity.getOrgName(), CrmBeanUtil.copy(old, MdmOrgRespVo.class), CrmBeanUtil.copy(entity, MdmOrgRespVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmOrgEntity> list = this.lambdaQuery()
                .in(MdmOrgEntity::getId, ids)
                .select(MdmOrgEntity::getId, MdmOrgEntity::getOrgCode, MdmOrgEntity::getRuleCode)
                .list();
        Assert.notEmpty(list, "无效的参数");
        List<String> orgCodeList = list.stream().map(MdmOrgEntity::getOrgCode).collect(Collectors.toList());

        List<MdmOrgEntity> children = this.lambdaQuery()
                .in(MdmOrgEntity::getParentCode, orgCodeList)
                .notIn(MdmOrgEntity::getId, ids)
                .select(MdmOrgEntity::getId)
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(children), "存在下级组织，不能删除");

        List<MdmPositionRespVo> positionList = mdmPositionService.queryBatchByOrgCodeList(orgCodeList);
        Assert.isTrue(CollectionUtils.isEmpty(positionList), "存在关联职位无法删除");

        List<String> terminalRelList = mdmTerminalROrgService.findTerminalCodeByOrg(orgCodeList);
        Assert.isTrue(CollectionUtils.isEmpty(terminalRelList), "存在关联终端无法删除");

        List<String> customerRelList = mdmCustomerROrgService.findCustomerByOrgCodeList(orgCodeList);
        Assert.isTrue(CollectionUtils.isEmpty(customerRelList), "存在关联客户无法删除");

        List<MdmOrgRegionRespVo> orgRegionList = mdmOrgRegionService.findOrgRegionByOrgCodeList(orgCodeList);
        Assert.isTrue(CollectionUtils.isEmpty(orgRegionList), "存在关联行政区域无法删除");

        this.removeByIds(ids);
        OrgUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            for (String id : ids) {
                MdmOrgRespVo detail = detail(id, null);
                if (detail != null) {
                    crmLogSendUtil.sendForDel(((String) menuObject), detail.getId(), detail.getOrgName(), CrmBeanUtil.copy(detail, MdmOrgRespVo.class));
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmOrgReqVo reqVo) {
        Assert.notEmpty(reqVo.getIds(), "缺失参数");
        this.deleteBatch(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmOrgEntity> list = this.lambdaQuery()
                .in(MdmOrgEntity::getId, ids)
                .select(MdmOrgEntity::getRuleCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodesExcludeAnySelf(list.stream().map(MdmOrgEntity::getRuleCode).collect(Collectors.toList()));
            if (!parentRuleCodes.isEmpty()) {
                List<MdmOrgEntity> parentDisableList = this.lambdaQuery()
                        .eq(MdmOrgEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                        .in(MdmOrgEntity::getRuleCode, parentRuleCodes)
                        .select(MdmOrgEntity::getOrgName)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(parentDisableList)) {
                    throw new BusinessException("存在未启用的组织[" + parentDisableList.get(0).getOrgName() + "],不能启用当前组织");
                }
            }
        }
        this.lambdaUpdate()
                .in(MdmOrgEntity::getId, ids)
                .set(MdmOrgEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
        OrgUtil.deleteAllCache();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmOrgReqVo reqVo) {
        Assert.notEmpty(reqVo.getIds(), "缺失参数");
        this.disableBatch(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<String> childrenCodeList = this.findAllChildrenOrgCodeListByList(ids, null);
        if (CollectionUtil.listNotEmptyNotSizeZero(childrenCodeList)) {
            this.lambdaUpdate()
                    .in(MdmOrgEntity::getOrgCode, childrenCodeList)
                    .set(MdmOrgEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                    .update();
            OrgUtil.deleteAllCache();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmOrgReqVo reqVo) {
        Assert.notEmpty(reqVo.getIds(), "缺失参数");
        this.disableBatch(reqVo.getIds());
    }

    @Override
    public List<MdmOrgSelectRespVo> findOrgSelectList(MdmOrgSelectReqVo reqVo) {
        List<MdmOrgSelectRespVo> list = new ArrayList<>();
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        String underThisRuleCode = null;
        String notUnderThisRuleCode = null;
        if (StringUtils.isNotEmpty(reqVo.getUnderThisOrgCode())) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, reqVo.getUnderThisOrgCode())
                    .select(MdmOrgEntity::getRuleCode)
                    .one();
            if (one != null) {
                underThisRuleCode = one.getRuleCode();
            }
        }
        if (StringUtils.isNotEmpty(reqVo.getNotUnderThisOrgCode())) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(MdmOrgEntity::getOrgCode, reqVo.getNotUnderThisOrgCode())
                    .select(MdmOrgEntity::getRuleCode)
                    .one();
            if (one != null) {
                notUnderThisRuleCode = one.getRuleCode();
            }
        }
        Set<String> codeList = new HashSet<>(16);
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmOrgMapper.findOrgSelectList(new Page<>(1L, pageSize, false), reqVo, underThisRuleCode, notUnderThisRuleCode));
        }
        return list;
    }

    @Override
    public List<MdmOrgRespVo> findAllChildrenOrgList(String orgId, String orgCode) {
        List<MdmOrgRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgId) || StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity org = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(orgId), MdmOrgEntity::getId, orgId)
                    .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                    .select(MdmOrgEntity::getRuleCode)
                    .one();
            if (org != null) {
                List<MdmOrgEntity> entityList = this.lambdaQuery()
                        .likeRight(MdmOrgEntity::getRuleCode, org.getRuleCode())
                        .list();
                list.addAll(CrmBeanUtil.copyList(entityList, MdmOrgRespVo.class));
            }
        }
        return list;
    }

    @Override
    public List<MdmOrgRespVo> findAllChildrenOrgListByList(List<String> ids, List<String> orgCodeList) {
        List<MdmOrgRespVo> list = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> orgEntityList = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmOrgEntity::getId, ids)
                    .in(CollectionUtil.listNotEmptyNotSizeZero(orgCodeList), MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgEntityList)) {
                List<String> collect = orgEntityList.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                return mdmOrgMapper.findOrgChildrenByRuleCodeList(collect);
            }
        }
        return list;
    }

    @Override
    public List<String> findAllChildrenOrgCodeList(String orgId, String orgCode) {
        List<String> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgId) || StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity org = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(orgId), MdmOrgEntity::getId, orgId)
                    .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                    .select(MdmOrgEntity::getRuleCode)
                    .one();
            if (org != null) {
                List<MdmOrgEntity> entityList = this.lambdaQuery()
                        .likeRight(MdmOrgEntity::getRuleCode, org.getRuleCode())
                        .select(MdmOrgEntity::getOrgCode)
                        .list();
                list.addAll(entityList.stream().map(MdmOrgEntity::getOrgCode).collect(Collectors.toList()));
            }
        }
        return list;
    }

    @Override
    public List<String> findAllChildrenOrgCodeListByList(List<String> ids, List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> list = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmOrgEntity::getId, ids)
                    .in(CollectionUtil.listNotEmptyNotSizeZero(orgCodeList), MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                List<String> collect = list.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                List<MdmOrgRespVo> orgChildrenList = mdmOrgMapper.findOrgChildrenByRuleCodeList(collect);
                if (CollectionUtil.listNotEmptyNotSizeZero(orgChildrenList)) {
                    return orgChildrenList.stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgRespVo> findSingleChildrenOrgList(String orgId, String orgCode) {
        List<MdmOrgRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgId) || StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity org = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(orgId), MdmOrgEntity::getId, orgId)
                    .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                    .one();
            if (org != null) {
                List<MdmOrgEntity> entityList = this.lambdaQuery()
                        .eq(MdmOrgEntity::getParentCode, org.getOrgCode())
                        .list();
                list.addAll(CrmBeanUtil.copyList(entityList, MdmOrgRespVo.class));
            }
        }
        return list;
    }

    @Override
    public List<MdmOrgRespVo> findAllParentOrgList(String orgId, String orgCode) {
        List<MdmOrgRespVo> list = new ArrayList<>();
        MdmOrgEntity one = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(orgId), MdmOrgEntity::getId, orgId)
                .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                .one();
        if (one != null) {
            Set<String> ruleCodes = TreeRuleCodeUtil.splitParentRuleCodes(one.getRuleCode());
            List<MdmOrgEntity> entityList = this.lambdaQuery()
                    .in(MdmOrgEntity::getRuleCode, ruleCodes)
                    .list();
            list.addAll(CrmBeanUtil.copyList(entityList, MdmOrgRespVo.class));
        }
        return list;
    }

    @Override
    public List<String> findAllParentOrgCodeList(String orgId, String orgCode) {
        List<MdmOrgRespVo> allParentOrgList = findAllParentOrgList(orgId, orgCode);
        if (CollectionUtils.isEmpty(allParentOrgList)) {
            return new ArrayList<>();
        }
        return allParentOrgList.stream()
                .filter(x -> !StringUtils.isEmpty(x.getOrgCode()))
                .map(MdmOrgRespVo::getOrgCode)
                .collect(Collectors.toList());
    }

    @Override
    public MdmOrgRespVo findSingleParentOrg(String orgId, String orgCode) {
        if (StringUtils.isNotEmpty(orgId) || StringUtils.isNotEmpty(orgCode)) {
            MdmOrgEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(orgId), MdmOrgEntity::getId, orgId)
                    .eq(StringUtils.isNotEmpty(orgCode), MdmOrgEntity::getOrgCode, orgCode)
                    .one();
            if (one != null && StringUtils.isNotEmpty(one.getParentCode())) {
                MdmOrgEntity parent = this.lambdaQuery()
                        .eq(MdmOrgEntity::getOrgCode, one.getParentCode())
                        .one();
                if (parent != null) {
                    return CrmBeanUtil.copy(parent, MdmOrgRespVo.class);
                }
            }
        }
        return null;
    }

    @Transactional(rollbackFor = Exception.class)
    @Klock
    @Override
    public void resetRuleCode2() {
        long l = System.currentTimeMillis();
        log.info("-----------重置企业组织降维编码_" + l + "_开始-----------");
        mdmOrgMapper.copyIdToRuleCode();
        log.info("-----------重置企业组织降维编码_" + l + "_编码重置成id成功-----------");
        mdmOrgMapper.setNullNotExistParentCode();
        log.info("-----------重置企业组织降维编码_" + l + "_清除无效上级编码成功-----------");
        List<MdmOrgEntity> list = this.lambdaQuery()
                .isNull(MdmOrgEntity::getParentCode)
                .or()
                .eq(MdmOrgEntity::getParentCode, "")
                .list();
        for (int i = 0; i < list.size(); i++) {
            updateCurAndChildren(list.get(i).getOrgCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
        }
        log.info("-----------重置企业组织降维编码_" + l + "_编码重新生成成功-----------");
        OrgUtil.deleteAllCache();
        log.info("-----------重置企业组织降维编码_" + l + "_缓存清除成功-----------");
        log.info("-----------重置企业组织降维编码_" + l + "_结束-----------");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock
    public void resetRuleCode() {
        long l = System.currentTimeMillis();
        log.info("-----------重置企业组织降维编码_" + l + "_开始-----------");
        mdmOrgMapper.copyIdToRuleCode();
        log.info("-----------重置企业组织降维编码_" + l + "_编码重置成id成功-----------");
        mdmOrgMapper.setNullNotExistParentCode();
        log.info("-----------重置企业组织降维编码_" + l + "_清除无效上级编码成功-----------");
        List<MdmOrgTreeRespVo> orgTree = getOrgTreeForResetRuleCode();
        log.info("-----------重置企业组织降维编码_" + l + "_树形结构构建成功-----------");
        updateCurAndChildren2(orgTree, "", 1);
        log.info("-----------重置企业组织降维编码_" + l + "_编码更新成功-----------");
        OrgUtil.deleteAllCache();
        log.info("-----------重置企业组织降维编码_" + l + "_缓存清除成功-----------");
        log.info("-----------重置企业组织降维编码_" + l + "_结束-----------");
    }

    @Override
    public List<MdmOrgWithPositionRespVo> findOrgWithAllPositionList(List<String> orgCodeList) {
        List<MdmOrgWithPositionRespVo> list = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> orgList = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList)) {
                list = CrmBeanUtil.copyList(orgList, MdmOrgWithPositionRespVo.class);
                list.forEach(item -> {
                    MdmPositionUserOrgReqVo reqVo = new MdmPositionUserOrgReqVo();
                    reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                    reqVo.setAllUnderThisOrgCode(item.getOrgCode());
                    item.setPositionList(mdmPositionService.findPositionUserOrgList(reqVo));
                });
            }
        }
        return list;
    }

    @Override
    public List<MdmOrgWithPositionRespVo> findOrgWithSinglePositionList(List<String> orgCodeList) {
        List<MdmOrgWithPositionRespVo> list = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> orgList = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList)) {
                MdmPositionUserOrgReqVo reqVo = new MdmPositionUserOrgReqVo();
                reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                reqVo.setOrgCodeList(orgCodeList);
                List<MdmPositionUserOrgRespVo> positionUserOrgList = mdmPositionService.findPositionUserOrgList(reqVo);
                list = CrmBeanUtil.copyList(orgList, MdmOrgWithPositionRespVo.class);
                if (CollectionUtil.listNotEmptyNotSizeZero(positionUserOrgList)) {
                    final Map<String, List<MdmPositionUserOrgRespVo>> collect = positionUserOrgList.stream().collect(Collectors.groupingBy(MdmPositionUserOrgRespVo::getOrgCode));
                    list.forEach(item -> {
                        item.setPositionList(collect.get(item.getOrgCode()));
                    });
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmOrgRespVo> findOrgAndChildrenList(MdmOrgSearchReqVo reqVo) {
        List<String> rightLikeOrgRuleCodeList = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getUnderOrgCodeList())) {
            List<MdmOrgEntity> orgList = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, reqVo.getUnderOrgCodeList())
                    .select(MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList)) {
                rightLikeOrgRuleCodeList.addAll(orgList.stream().map(k -> {
                    return k.getRuleCode() + "%";
                }).collect(Collectors.toList()));
            }
        }
        if (reqVo.getPageSize() == null || reqVo.getPageSize() == 0) {
            reqVo.setPageSize(Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE)));
        }
        return mdmOrgMapper.findOrgAndChildrenList(new Page<>(1L, reqVo.getPageSize(), false), reqVo, rightLikeOrgRuleCodeList);
    }

    /**
     * 根据parentCode生成一个降维编码，新增组织的时候调用
     *
     * @param parentCode
     * @return
     */
    protected String generateRuleCodeByParentCode(String parentCode) {
        String ruleCode = "";
        boolean top = true;
        String prefix = "";
        if (StringUtils.isNotEmpty(parentCode)) {
            MdmOrgEntity parent = this.lambdaQuery().eq(MdmOrgEntity::getOrgCode, parentCode).one();
            if (parent != null) {
                top = false;
                prefix = parent.getRuleCode();
            }
        }
        List<MdmOrgEntity> list = null;
        if (top) {
            list = this.lambdaQuery().isNull(MdmOrgEntity::getParentCode).or().eq(MdmOrgEntity::getParentCode, "").list();
        } else {
            list = this.lambdaQuery().eq(MdmOrgEntity::getParentCode, parentCode).list();
        }
        if (CollectionUtil.listEmpty(list)) {
            ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(1);
        } else {
            Set<Integer> set = new LinkedHashSet<>();
            for (MdmOrgEntity item :
                    list) {
                if (StringUtils.isNotEmpty(item.getRuleCode())) {
                    try {
                        Integer integer = TreeRuleCodeUtil.getCurLevelCodeValue(item.getRuleCode());
                        set.add(integer);
                    } catch (BusinessException e) {
                        throw new BusinessException("企业组织" + e.getMsg());
                    }
                }
            }
            int maxRuleCodeValueInLevel = TreeRuleCodeUtil.getMaxRuleCodeValueInLevel();
            for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
                if (i == maxRuleCodeValueInLevel) {
                    throw new BusinessException("降维编码越界，请联系管理员处理");
                }
                if (!set.contains(i)) {
                    ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(i);
                    break;
                }
            }
        }
        return ruleCode;
    }

    /**
     * 更新组织及组织下级降维编码和层级
     *
     * @param orgCode  当前组织编码
     * @param curCode  当前组织降维编码
     * @param levelNum 当前层级
     */
    protected void updateCurAndChildren(String orgCode, String curCode, int levelNum) {

        //更新当前
        this.lambdaUpdate()
                .eq(MdmOrgEntity::getOrgCode, orgCode)
                .set(MdmOrgEntity::getRuleCode, curCode)
                .set(MdmOrgEntity::getLevelNum, levelNum)
                .update();

        //查询下一层
        List<MdmOrgEntity> list = this.lambdaQuery()
                .eq(MdmOrgEntity::getParentCode, orgCode)
                .select(MdmOrgEntity::getOrgCode)
                .list();

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                //递归调用
                updateCurAndChildren(list.get(i).getOrgCode(), curCode + TreeRuleCodeUtil.numToSingleCode(i + 1), (levelNum + 1));
            }
        }
    }

    protected void updateCurAndChildren2(List<MdmOrgTreeRespVo> list, String parentRuleCode, int curLevelNum) {

        //查询下一层
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                MdmOrgTreeRespVo item = list.get(i);
                item.setRuleCode(parentRuleCode + TreeRuleCodeUtil.numToSingleCode(i + 1));
                item.setLevelNum(curLevelNum);
                this.lambdaUpdate()
                        .eq(MdmOrgEntity::getOrgCode, item.getOrgCode())
                        .set(MdmOrgEntity::getRuleCode, item.getRuleCode())
                        .set(MdmOrgEntity::getLevelNum, item.getLevelNum())
                        .update();
                if (CollectionUtil.listNotEmptyNotSizeZero(item.getChildren())) {
                    //递归调用
                    updateCurAndChildren2(item.getChildren(), item.getRuleCode(), (item.getLevelNum() + 1));
                }
            }
        }
    }

    @Override
    public List<MdmOrgRespVo> listCondition(MdmOrgReqVo reqVo) {
        if (CollectionUtil.collectionNotEmpty(reqVo.getOrgCodeList())) {
            List<MdmOrgEntity> mdmOrgEntities = this.lambdaQuery().in(MdmOrgEntity::getOrgCode, reqVo.getOrgCodeList()).list();
            return CrmBeanUtil.copyList(mdmOrgEntities, MdmOrgRespVo.class);
        } else {
            return Lists.emptyList();
        }
    }

    @Override
    public List<MdmOrgRespVo> findOrgList(MdmOrgReqVo reqVo) {
        List<MdmOrgEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()), MdmOrgEntity::getEnableStatus, reqVo.getEnableStatus())
                .eq(StringUtils.isNotEmpty(reqVo.getOrgType()), MdmOrgEntity::getOrgType, reqVo.getOrgType())
                .eq(StringUtils.isNotEmpty(reqVo.getParentCode()), MdmOrgEntity::getParentCode, reqVo.getParentCode())
                .like(StringUtils.isNotEmpty(reqVo.getOrgCode()), MdmOrgEntity::getOrgCode, reqVo.getOrgCode())
                .like(StringUtils.isNotEmpty(reqVo.getOrgName()), MdmOrgEntity::getOrgName, reqVo.getOrgName())
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return CrmBeanUtil.copyList(list, MdmOrgRespVo.class);
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgTreeRespVo> getOrgTree() {
        List<MdmOrgEntity> list = this.lambdaQuery().list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<MdmOrgTreeRespVo> totalList = CrmBeanUtil.copyList(list, MdmOrgTreeRespVo.class);

            //构建树list
            List<MdmOrgTreeRespVo> treeList = new ArrayList<>();
            //当前操作层级数据
            List<MdmOrgTreeRespVo> curLevelList = new ArrayList<>();
            //未操作数据
            List<MdmOrgTreeRespVo> restList = new ArrayList<>();

            //key:id
            Map<String, MdmOrgTreeRespVo> totalMap = totalList.stream().collect(Collectors.toMap(MdmOrgTreeRespVo::getOrgCode, v -> v));

            //查找第一层
            for (MdmOrgTreeRespVo item :
                    totalList) {
                if (StringUtils.isEmpty(item.getParentCode()) || !totalMap.containsKey(item.getParentCode())) {
                    treeList.add(item);
                    curLevelList.add(item);
                } else {
                    restList.add(item);
                }
            }

            //构建数据，从第二层开始
            while (curLevelList.size() > 0 && restList.size() > 0) {
                List<MdmOrgTreeRespVo> restTempList = new ArrayList<>();
                List<MdmOrgTreeRespVo> curLevelTempList = new ArrayList<>();
//                Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(MdmOrgTreeRespVo::getOrgCode, MdmOrgTreeRespVo::getOrgCode));
                Set<String> curLevelSet = curLevelList.stream().map(MdmOrgTreeRespVo::getOrgCode).collect(Collectors.toSet());
                Map<String, List<MdmOrgTreeRespVo>> curLevelChildrenMap = new HashMap<>();

                for (MdmOrgTreeRespVo item :
                        restList) {
                    if (curLevelSet.contains(item.getParentCode())) {
                        curLevelTempList.add(item);

                        List<MdmOrgTreeRespVo> childrenList = new ArrayList<>();
                        if (curLevelChildrenMap.containsKey(item.getParentCode())) {
                            childrenList.addAll(curLevelChildrenMap.get(item.getParentCode()));
                        }
                        childrenList.add(item);
                        curLevelChildrenMap.put(item.getParentCode(), childrenList);
                    } else {
                        restTempList.add(item);
                    }
                }
                for (MdmOrgTreeRespVo item :
                        curLevelList) {
                    if (curLevelChildrenMap.containsKey(item.getOrgCode())) {
                        item.setChildren(curLevelChildrenMap.get(item.getOrgCode()));
                    }
                }
                curLevelList.clear();
                curLevelList.addAll(curLevelTempList);
                restList.clear();
                restList.addAll(restTempList);
            }
            return treeList;

        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findAllParentOrgCodeExcludeSelf(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmOrgEntity> orgList = this.lambdaQuery()
                    .in(MdmOrgEntity::getOrgCode, orgCodeList)
                    .select(MdmOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList)) {
                Set<String> strings = TreeRuleCodeUtil.splitParentRuleCodesExcludeSelf(orgList.stream().map(MdmOrgEntity::getRuleCode).collect(Collectors.toList()));
                if (!strings.isEmpty()) {
                    return this.lambdaQuery()
                            .in(MdmOrgEntity::getRuleCode, strings)
                            .select(MdmOrgEntity::getOrgCode)
                            .list()
                            .stream().map(MdmOrgEntity::getOrgCode).collect(Collectors.toList());
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> findOrgNameMap(List<String> orgCodeList) {
        if (CollectionUtils.isEmpty(orgCodeList)) {
            return Collections.emptyMap();
        }
        return this.lambdaQuery()
                .in(MdmOrgEntity::getOrgCode, orgCodeList)
                .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName)
                .list()
                .stream()
                .filter(x -> !StringUtils.isEmpty(x.getOrgCode()))
                .collect(Collectors.toMap(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName));
    }

    @Override
    public List<String> findAllChildrenOrgCodeListOfPositionChildrenList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionRespVo> allChildrenPositionList = PositionUtil.getChildrenPositionListExcludeSelf(positionCodeList);
            if (CollectionUtil.listNotEmptyNotSizeZero(allChildrenPositionList)) {
                Set<String> orgCodeSet = allChildrenPositionList.stream().filter(item -> StringUtils.isNotEmpty(item.getOrgCode())).map(MdmPositionRespVo::getOrgCode).collect(Collectors.toSet());
                if (!orgCodeSet.isEmpty()) {
                    return OrgUtil.getChildrenOrgCodeListExcludeSelf(new ArrayList<>(orgCodeSet));
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findAllParentOrgCodeListOfPositionParentList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionRespVo> positionList = PositionUtil.getPositionByCodeList(positionCodeList);
            if (CollectionUtil.listNotEmptyNotSizeZero(positionList)) {
                List<MdmPositionRespVo> parentPositionListExcludeSelf = PositionUtil.getParentPositionListExcludeSelf(positionList.stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList()));
                if (CollectionUtil.listNotEmptyNotSizeZero(parentPositionListExcludeSelf)) {
                    Set<String> parentPositionOrgCodeSet = parentPositionListExcludeSelf.stream()
                            .filter(item -> StringUtils.isNotEmpty(item.getOrgCode()))
                            .map(MdmPositionRespVo::getOrgCode)
                            .collect(Collectors.toSet());
                    if (!parentPositionOrgCodeSet.isEmpty()) {
                        return OrgUtil.getParentOrgCodeListExcludeSelf(new ArrayList<>(parentPositionOrgCodeSet));
                    }
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmOrgTreeRespVo> orgSearchTree(MdmOrgReqVo mdmOrgReqVo) {
        List<MdmOrgTreeRespVo> tree = new ArrayList<>();


        List<MdmOrgEntity> list = this.lambdaQuery()
                .like(!StringUtils.isEmpty(mdmOrgReqVo.getOrgName()), MdmOrgEntity::getOrgName, mdmOrgReqVo.getOrgName())
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            if (StringUtils.isNotEmpty(mdmOrgReqVo.getOrgName())) {
                Set<String> set = TreeRuleCodeUtil.splitParentRuleCodes(list.stream().map(MdmOrgEntity::getRuleCode).collect(Collectors.toList()));
                if (!set.isEmpty()) {
                    list = this.lambdaQuery()
                            .in(MdmOrgEntity::getRuleCode, set)
                            .list();
                }
            }
            List<MdmOrgTreeRespVo> collect = CrmBeanUtil.copyList(list, MdmOrgTreeRespVo.class);
            Map<String, String> orgTypeMap = DictUtil.dictMap(DictConstant.MDM_ORG_TYPE);
            if (orgTypeMap != null && orgTypeMap.size() > 0) {
                for (MdmOrgTreeRespVo mdmOrgTreeRespVo : collect) {
                    String orgType = mdmOrgTreeRespVo.getOrgType();
                    if (!StringUtils.isEmpty(orgType)) {
                        mdmOrgTreeRespVo.setOrgTypeName(orgTypeMap.get(orgType));
                    }
                }
            }
            for (MdmOrgTreeRespVo mdmOrgTreeRespVo : collect) {
                if (StringUtils.isEmpty(mdmOrgTreeRespVo.getParentCode())) {
                    tree.add(mdmOrgTreeRespVo);
                }
                List<MdmOrgTreeRespVo> children = mdmOrgTreeRespVo.getChildren();
                if (children == null) {
                    children = new ArrayList<>();
                }
                for (MdmOrgTreeRespVo orgTreeRespVo : collect) {
                    if (mdmOrgTreeRespVo.getOrgCode().equals(orgTreeRespVo.getParentCode())) {
                        children.add(orgTreeRespVo);
                    }
                }
                mdmOrgTreeRespVo.setChildren(children);
            }

        }
        return tree;
    }

    /**
     * 构建树形结构用来重置降维编码
     *
     * @return
     */
    protected List<MdmOrgTreeRespVo> getOrgTreeForResetRuleCode() {
        List<MdmOrgEntity> list = this.lambdaQuery()
                .select(MdmOrgEntity::getId, MdmOrgEntity::getOrgCode, MdmOrgEntity::getParentCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<MdmOrgTreeRespVo> totalList = CrmBeanUtil.copyList(list, MdmOrgTreeRespVo.class);

            //构建树list
            List<MdmOrgTreeRespVo> treeList = new ArrayList<>();
            //当前操作层级数据
            List<MdmOrgTreeRespVo> curLevelList = new ArrayList<>();
            //未操作数据
            List<MdmOrgTreeRespVo> restList = new ArrayList<>();

            final Set<String> totalCodeSet = totalList.stream().map(MdmOrgTreeRespVo::getOrgCode).collect(Collectors.toSet());

            //查找第一层
            for (MdmOrgTreeRespVo item :
                    totalList) {
                if (StringUtils.isEmpty(item.getParentCode()) || !totalCodeSet.contains(item.getParentCode())) {
                    treeList.add(item);
                    curLevelList.add(item);
                } else {
                    restList.add(item);
                }
            }

            //构建数据，从第二层开始
            while (curLevelList.size() > 0 && restList.size() > 0) {
                List<MdmOrgTreeRespVo> restTempList = new ArrayList<>();
                List<MdmOrgTreeRespVo> curLevelTempList = new ArrayList<>();
                final Set<String> curLevelCodeSet = curLevelList.stream().map(MdmOrgTreeRespVo::getOrgCode).collect(Collectors.toSet());
                Map<String, List<MdmOrgTreeRespVo>> curLevelChildrenMap = new HashMap<>();

                for (MdmOrgTreeRespVo item :
                        restList) {
                    if (curLevelCodeSet.contains(item.getParentCode())) {
                        curLevelTempList.add(item);

                        List<MdmOrgTreeRespVo> childrenList = new ArrayList<>();
                        if (curLevelChildrenMap.containsKey(item.getParentCode())) {
                            childrenList.addAll(curLevelChildrenMap.get(item.getParentCode()));
                        }
                        childrenList.add(item);
                        curLevelChildrenMap.put(item.getParentCode(), childrenList);
                    } else {
                        restTempList.add(item);
                    }
                }
                for (MdmOrgTreeRespVo item :
                        curLevelList) {
                    if (curLevelChildrenMap.containsKey(item.getOrgCode())) {
                        item.setChildren(curLevelChildrenMap.get(item.getOrgCode()));
                    }
                }
                curLevelList.clear();
                curLevelList.addAll(curLevelTempList);
                restList.clear();
                restList.addAll(restTempList);
            }
            return treeList;
        }
        return new ArrayList<>();
    }

    /**
     * 查询全部组织
     * @return
     */
    @Override
    public List<MdmOrgRespVo> getAll(){
        List<MdmOrgRespVo> orgList =   mdmOrgMapper.getAll();
        orgList = orgList==null?new ArrayList<>():orgList;
        return orgList;
    }


}
