package com.biz.crm.permission.service.impl;


import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.*;
import com.biz.crm.function.model.MdmFunctionEntity;
import com.biz.crm.function.service.MdmFunctionService;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionReqVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionRespVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubReqVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubRespVo;
import com.biz.crm.nebular.mdm.permission.*;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.org.model.MdmOrgEntity;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.permission.manager.MdmDataPermissionContext;
import com.biz.crm.permission.mapper.MdmDataPermissionMapper;
import com.biz.crm.permission.model.MdmDataPermissionEntity;
import com.biz.crm.permission.model.MdmDataPermissionExtendEntity;
import com.biz.crm.permission.service.IMdmDataPermissionExtendService;
import com.biz.crm.permission.service.IMdmDataPermissionService;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.tableconfig.model.MdmFunctionSubEntity;
import com.biz.crm.tableconfig.service.IMdmFunctionSubService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 数据权限接口实现
 *
 * @author zxw
 * @date 2020-10-26 12:09:31
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmDataPermissionServiceExpandImpl")
public class MdmDataPermissionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmDataPermissionMapper, MdmDataPermissionEntity> implements IMdmDataPermissionService {

    @Autowired
    private MdmDataPermissionMapper mdmDataPermissionMapper;
    @Autowired
    private MdmFunctionService mdmFunctionService;
    @Autowired
    private IMdmFunctionSubService mdmFunctionSubService;
    @Autowired
    private IMdmDataPermissionExtendService mdmDataPermissionExtendService;
    @Autowired
    private MdmOrgService mdmOrgService;
    @Autowired
    private MdmPositionService mdmPositionService;
    @Autowired
    private MdmPositionRoleService mdmPositionRoleService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Autowired
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Autowired
    private MdmDataPermissionContext mdmDataPermissionContext;
    @Autowired
    private MdmRoleService mdmRoleService;

    @Override
    @CrmDictMethod
    public PageResult<MdmDataPermissionRespVo> findList(MdmDataPermissionReqVo reqVo) {
        Page<MdmDataPermissionRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmDataPermissionRespVo> list = mdmDataPermissionMapper.findList(page, reqVo);
        return PageResult.<MdmDataPermissionRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmDataPermissionRespVo query(MdmDataPermissionReqVo reqVo) {
        MdmDataPermissionEntity one = this.lambdaQuery()
                .eq(MdmDataPermissionEntity::getId, reqVo.getId())
                .last(DatabaseTypeUtil.SEGMENT)
                .one();
        return CrmBeanUtil.copy(one, MdmDataPermissionRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmDataPermissionReqVo reqVo) {
        String listConfigCode = reqVo.getListConfigCode();
        Assert.hasText(listConfigCode, "列表编码不能为空");
        String functionCode = reqVo.getFunctionCode();
        Assert.hasText(functionCode, "菜单编码不能为空");
        String roleCode = reqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        this.lambdaUpdate()
                .eq(MdmDataPermissionEntity::getRoleCode, roleCode)
                .eq(MdmDataPermissionEntity::getFunctionCode, functionCode)
                .eq(MdmDataPermissionEntity::getListConfigCode, listConfigCode)
                .remove();

        mdmDataPermissionExtendService.lambdaUpdate()
                .eq(MdmDataPermissionExtendEntity::getRoleCode, roleCode)
                .eq(MdmDataPermissionExtendEntity::getFunctionCode, functionCode)
                .eq(MdmDataPermissionExtendEntity::getListConfigCode, listConfigCode)
                .remove();

        String permissionResource = reqVo.getPermissionResource();
        String permissionObjCode = reqVo.getPermissionObjCode();
        if (StringUtils.isEmpty(permissionResource) || StringUtils.isEmpty(permissionObjCode)) {
            return;
        }
        Assert.isTrue(MdmPermissionObjEnum.contain(permissionObjCode), "权限对象传参错误");
        Assert.isTrue(PermissionResourceEnum.contain(permissionResource), "权限数据来源错误");
        String id = "";
        switch (MdmPermissionObjEnum.convertEnum(permissionObjCode)) {
            case POSITION:
                id = storePosition(reqVo);
                break;
            case ORG:
                id = storeOrg(reqVo);
                break;
            case CUSTOMER:
                id = storeCustomer(reqVo);
                break;
            case CUSTOMER_ORG:
                id = storeCustomerOrg(reqVo);
                break;
            case CHANNEL:
                id = storeChannel(reqVo);
                break;
            case TERMINAL:
                id = storeTerminal(reqVo);
                break;
            default:
        }
        Object obj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (obj != null) {
            crmLogSendUtil.sendForAdd(obj.toString(), id, null, reqVo);
        }
    }

    protected String storeTerminal(MdmDataPermissionReqVo reqVo) {
        return null;
    }

    protected String storeChannel(MdmDataPermissionReqVo reqVo) {
        return null;
    }

    protected String storeCustomerOrg(MdmDataPermissionReqVo reqVo) {
        return null;
    }

    protected String storeCustomer(MdmDataPermissionReqVo reqVo) {
        log.info("数据权限：保存客户:{}", reqVo);
        String permissionResource = reqVo.getPermissionResource();
        String permissionObjCode = reqVo.getPermissionObjCode();
        String roleCode = reqVo.getRoleCode();
        String functionCode = reqVo.getFunctionCode();
        String listConfigCode = reqVo.getListConfigCode();

        List<MdmDataPermissionCustomerVo> dataPermissionCustomerVos = reqVo.getDataPermissionCustomerVos();
        switch (PermissionResourceEnum.convertEnum(permissionResource)) {
            case PRESUPPOSITION:
                String permissionSearchType = reqVo.getPermissionSearchType();
                Assert.hasText(permissionSearchType, "查询方式不能为空");
                MdmDataPermissionExtendEntity mdmDataPermissionExtendEntity = new MdmDataPermissionExtendEntity();
                mdmDataPermissionExtendEntity.setRoleCode(roleCode);
                mdmDataPermissionExtendEntity.setFunctionCode(functionCode);
                mdmDataPermissionExtendEntity.setListConfigCode(listConfigCode);
                mdmDataPermissionExtendEntity.setPermissionObjCode(permissionObjCode);
                mdmDataPermissionExtendEntity.setPermissionResource(permissionResource);
                mdmDataPermissionExtendEntity.setPermissionSearchType(permissionSearchType);
                mdmDataPermissionExtendService.save(mdmDataPermissionExtendEntity);
                break;
            case FIXED:
                Assert.isTrue(!CollectionUtils.isEmpty(dataPermissionCustomerVos), "客户集合不能为空");
                List<MdmDataPermissionExtendEntity> collect = dataPermissionCustomerVos.stream().map(x -> {
                    MdmDataPermissionExtendEntity mdmDataPermissionFixedExtendEntity = new MdmDataPermissionExtendEntity();
                    mdmDataPermissionFixedExtendEntity.setRoleCode(roleCode);
                    mdmDataPermissionFixedExtendEntity.setFunctionCode(functionCode);
                    mdmDataPermissionFixedExtendEntity.setListConfigCode(listConfigCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionObjCode(permissionObjCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionResource(permissionResource);
                    mdmDataPermissionFixedExtendEntity.setPermissionSearchType(MdmPermissionLevelEnum.CURRENT.getCode());
                    mdmDataPermissionFixedExtendEntity.setCode(x.getCustomerCode());
                    return mdmDataPermissionFixedExtendEntity;
                }).collect(Collectors.toList());
                mdmDataPermissionExtendService.saveBatch(collect);
                break;
            default:
        }
        MdmDataPermissionEntity entity = CrmBeanUtil.copy(reqVo, MdmDataPermissionEntity.class);
        this.saveOrUpdate(entity);
        return entity.getId();
    }

    protected String storeOrg(MdmDataPermissionReqVo reqVo) {
        log.info("数据权限：保存组织:{}", reqVo);
        String permissionResource = reqVo.getPermissionResource();
        String permissionObjCode = reqVo.getPermissionObjCode();
        String roleCode = reqVo.getRoleCode();
        String functionCode = reqVo.getFunctionCode();
        String listConfigCode = reqVo.getListConfigCode();
        List<MdmDataPermissionOrgVo> dataPermissionOrgVos = reqVo.getDataPermissionOrgVos();

        switch (PermissionResourceEnum.convertEnum(permissionResource)) {
            case PRESUPPOSITION:
                String permissionSearchType = reqVo.getPermissionSearchType();
                Assert.hasText(permissionSearchType, "查询方式不能为空");
                MdmDataPermissionExtendEntity mdmDataPermissionExtendEntity = new MdmDataPermissionExtendEntity();
                mdmDataPermissionExtendEntity.setRoleCode(roleCode);
                mdmDataPermissionExtendEntity.setFunctionCode(functionCode);
                mdmDataPermissionExtendEntity.setListConfigCode(listConfigCode);
                mdmDataPermissionExtendEntity.setPermissionObjCode(permissionObjCode);
                mdmDataPermissionExtendEntity.setPermissionResource(permissionResource);
                mdmDataPermissionExtendEntity.setPermissionSearchType(permissionSearchType);
                mdmDataPermissionExtendService.save(mdmDataPermissionExtendEntity);
                break;
            case FIXED:
                Assert.isTrue(!CollectionUtils.isEmpty(dataPermissionOrgVos), "职位集合不能为空");
                List<MdmDataPermissionExtendEntity> collect = dataPermissionOrgVos.stream().map(x -> {
                    MdmDataPermissionExtendEntity mdmDataPermissionFixedExtendEntity = new MdmDataPermissionExtendEntity();
                    mdmDataPermissionFixedExtendEntity.setRoleCode(roleCode);
                    mdmDataPermissionFixedExtendEntity.setFunctionCode(functionCode);
                    mdmDataPermissionFixedExtendEntity.setListConfigCode(listConfigCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionObjCode(permissionObjCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionResource(permissionResource);
                    mdmDataPermissionFixedExtendEntity.setPermissionSearchType(x.getPermissionSearchType());
                    mdmDataPermissionFixedExtendEntity.setCode(x.getOrgCode());
                    return mdmDataPermissionFixedExtendEntity;
                }).collect(Collectors.toList());
                mdmDataPermissionExtendService.saveBatch(collect);
                break;
            default:
        }
        MdmDataPermissionEntity entity = CrmBeanUtil.copy(reqVo, MdmDataPermissionEntity.class);
        this.saveOrUpdate(entity);
        return entity.getId();
    }

    protected String storePosition(MdmDataPermissionReqVo reqVo) {
        log.info("数据权限：保存职位:{}", reqVo);
        String permissionResource = reqVo.getPermissionResource();
        String permissionObjCode = reqVo.getPermissionObjCode();
        String roleCode = reqVo.getRoleCode();
        String functionCode = reqVo.getFunctionCode();
        String listConfigCode = reqVo.getListConfigCode();

        List<MdmDataPermissionPositionVo> dataPermissionPositionVos = reqVo.getDataPermissionPositionVos();
        switch (PermissionResourceEnum.convertEnum(permissionResource)) {
            case PRESUPPOSITION:
                String permissionSearchType = reqVo.getPermissionSearchType();
                Assert.hasText(permissionSearchType, "查询方式不能为空");
                MdmDataPermissionExtendEntity mdmDataPermissionExtendEntity = new MdmDataPermissionExtendEntity();
                mdmDataPermissionExtendEntity.setRoleCode(roleCode);
                mdmDataPermissionExtendEntity.setFunctionCode(functionCode);
                mdmDataPermissionExtendEntity.setListConfigCode(listConfigCode);
                mdmDataPermissionExtendEntity.setPermissionObjCode(permissionObjCode);
                mdmDataPermissionExtendEntity.setPermissionResource(permissionResource);
                mdmDataPermissionExtendEntity.setPermissionSearchType(permissionSearchType);
                mdmDataPermissionExtendService.save(mdmDataPermissionExtendEntity);
                break;
            case FIXED:
                Assert.isTrue(!CollectionUtils.isEmpty(dataPermissionPositionVos), "职位集合不能为空");
                List<MdmDataPermissionExtendEntity> collect = dataPermissionPositionVos.stream().map(x -> {
                    MdmDataPermissionExtendEntity mdmDataPermissionFixedExtendEntity = new MdmDataPermissionExtendEntity();
                    mdmDataPermissionFixedExtendEntity.setRoleCode(roleCode);
                    mdmDataPermissionFixedExtendEntity.setFunctionCode(functionCode);
                    mdmDataPermissionFixedExtendEntity.setListConfigCode(listConfigCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionObjCode(permissionObjCode);
                    mdmDataPermissionFixedExtendEntity.setPermissionResource(permissionResource);
                    mdmDataPermissionFixedExtendEntity.setPermissionSearchType(MdmPermissionLevelEnum.CURRENT.getCode());
                    mdmDataPermissionFixedExtendEntity.setCode(x.getPositionCode());
                    return mdmDataPermissionFixedExtendEntity;
                }).collect(Collectors.toList());
                mdmDataPermissionExtendService.saveBatch(collect);
                break;
            default:
        }
        MdmDataPermissionEntity entity = CrmBeanUtil.copy(reqVo, MdmDataPermissionEntity.class);
        this.saveOrUpdate(entity);
        return entity.getId();
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmDataPermissionReqVo reqVo) {
        MdmDataPermissionEntity mdmDataPermissionEntity = this.getById(reqVo.getId());
        MdmDataPermissionReqVo oldObject = CrmBeanUtil.copy(mdmDataPermissionEntity, MdmDataPermissionReqVo.class);
        save(reqVo);
        MdmDataPermissionEntity mdmDataPermissionEntityNew = this.getById(reqVo.getId());
        MdmDataPermissionReqVo newObject = CrmBeanUtil.copy(mdmDataPermissionEntityNew, MdmDataPermissionReqVo.class);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if(menuObject!=null){
            crmLogSendUtil.sendForUpdate(menuObject.toString(),
                reqVo.getId(), null, oldObject, newObject);
        }

    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmDataPermissionReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmDataPermissionEntity> list = this.lambdaQuery()
                .in(MdmDataPermissionEntity::getId, ids)
                .select(MdmDataPermissionEntity::getRoleCode, MdmDataPermissionEntity::getFunctionCode
                        , MdmDataPermissionEntity::getListConfigCode)
                .list();

        this.removeByIds(ids);
        for (MdmDataPermissionEntity entity : list) {
            crmLogSendUtil.sendForDel(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(),
                    entity.getId(), null, entity);
        }
        if (!CollectionUtils.isEmpty(list)) {
            List<String> roleCodeList = list.stream().map(MdmDataPermissionEntity::getRoleCode).filter(Objects::nonNull).collect(Collectors.toList());
            List<String> functionCodeList = list.stream().map(MdmDataPermissionEntity::getFunctionCode).filter(Objects::nonNull).collect(Collectors.toList());
            List<String> listConfigCodeList = list.stream().map(MdmDataPermissionEntity::getListConfigCode).filter(Objects::nonNull).collect(Collectors.toList());
            mdmDataPermissionExtendService.lambdaUpdate()
                    .in(!CollectionUtils.isEmpty(roleCodeList), MdmDataPermissionExtendEntity::getRoleCode, roleCodeList)
                    .in(!CollectionUtils.isEmpty(functionCodeList), MdmDataPermissionExtendEntity::getFunctionCode, functionCodeList)
                    .in(!CollectionUtils.isEmpty(listConfigCodeList), MdmDataPermissionExtendEntity::getListConfigCode, listConfigCodeList)
                    .remove();
        }
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmDataPermissionReqVo reqVo) {
        List<String> ids = valid(reqVo);
        Assert.notEmpty(ids, "缺失参数");
        this.lambdaUpdate()
                .in(MdmDataPermissionEntity::getId, ids)
                .set(MdmDataPermissionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    private List<String> valid(MdmDataPermissionReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        String id = ids.get(0);
        Assert.hasText(id, "id不能为空");
        return ids;
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmDataPermissionReqVo reqVo) {
        List<String> ids = valid(reqVo);
        Assert.notEmpty(ids, "缺失参数");
        this.lambdaUpdate()
                .in(MdmDataPermissionEntity::getId, ids)
                .set(MdmDataPermissionEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    public MdmDataPermissionRespVo queryCondition(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        String functionCode = mdmDataPermissionReqVo.getFunctionCode();
        Assert.hasText(functionCode, "菜单编码不能为空");
        String listConfigCode = mdmDataPermissionReqVo.getListConfigCode();
        Assert.hasText(listConfigCode, "列表配置编码不能为空");
        String roleCode = mdmDataPermissionReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");

        MdmDataPermissionRespVo mdmDataPermissionRespVo = mdmDataPermissionMapper.queryCondition(mdmDataPermissionReqVo);
        if (mdmDataPermissionRespVo == null) {
            return null;
        }
        List<MdmDataPermissionExtendEntity> list = mdmDataPermissionExtendService.lambdaQuery()
                .eq(MdmDataPermissionExtendEntity::getRoleCode, roleCode)
                .eq(MdmDataPermissionExtendEntity::getFunctionCode, functionCode)
                .eq(MdmDataPermissionExtendEntity::getListConfigCode, listConfigCode)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return mdmDataPermissionRespVo;
        }
        MdmDataPermissionExtendEntity mdmDataPermissionExtendEntity = list.get(0);
        String permissionResource = mdmDataPermissionExtendEntity.getPermissionResource();
        if (PermissionResourceEnum.PRESUPPOSITION.getCode().equals(permissionResource)) {
            String permissionObjCode = mdmDataPermissionExtendEntity.getPermissionObjCode();
            mdmDataPermissionRespVo.setPermissionObjCode(permissionObjCode);
            mdmDataPermissionRespVo.setPermissionResource(mdmDataPermissionExtendEntity.getPermissionResource());
            String permissionSearchType = mdmDataPermissionRespVo.getPermissionSearchType();
            mdmDataPermissionRespVo.setPermissionSearchType(permissionSearchType);
            if (!StringUtils.isEmpty(permissionObjCode) && !StringUtils.isEmpty(permissionSearchType)) {
                String subDictTypeCode = MdmPermissionObjEnum.getSubDictTypeCode(permissionObjCode);
                if (!StringUtils.isEmpty(subDictTypeCode)) {
                    mdmDataPermissionRespVo.setPermissionSearchTypeName(DictUtil.dictValue(subDictTypeCode, permissionSearchType));
                }
            }
            return mdmDataPermissionRespVo;
        } else if (PermissionResourceEnum.FIXED.getCode().equals(permissionResource)) {
            String permissionObjCode = mdmDataPermissionExtendEntity.getPermissionObjCode();
            if (!StringUtils.isEmpty(permissionObjCode)) {
                switch (MdmPermissionObjEnum.convertEnum(permissionObjCode)) {
                    case POSITION:
                        mdmDataPermissionRespVo.setDataPermissionPositionVos(setUpPosition(list));
                        break;
                    case ORG:
                        mdmDataPermissionRespVo.setDataPermissionOrgVos(setUpOrg(list));
                        break;
                    default:
                }
            }
        }
        return mdmDataPermissionRespVo;
    }

    protected List<MdmDataPermissionOrgVo> setUpOrg(List<MdmDataPermissionExtendEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }

        List<String> collect = list.stream()
                .filter(x -> !StringUtils.isEmpty(x.getCode()))
                .map(MdmDataPermissionExtendEntity::getCode)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(collect)) {
            return new ArrayList<>();
        }

        Map<String, MdmOrgEntity> mdmOrgEntityMap = mdmOrgService.lambdaQuery()
                .in(MdmOrgEntity::getOrgCode, collect)
                .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getOrgName, MdmOrgEntity::getOrgType)
                .list()
                .stream()
                .collect(Collectors.toMap(MdmOrgEntity::getOrgCode, Function.identity(), (x1, x2) -> x1));
        if (CollectionUtils.isEmpty(mdmOrgEntityMap)) {
            return new ArrayList<>();
        }
        return list.stream()
                .filter(x -> !StringUtils.isEmpty(x.getCode()))
                .map(x -> {
                    MdmDataPermissionOrgVo mdmDataPermissionOrgVo = new MdmDataPermissionOrgVo();
                    MdmOrgEntity mdmOrgEntity = mdmOrgEntityMap.get(x.getCode());
                    if (mdmOrgEntity == null) {
                        return null;
                    }
                    mdmDataPermissionOrgVo.setOrgCode(mdmOrgEntity.getOrgCode());
                    mdmDataPermissionOrgVo.setOrgName(mdmOrgEntity.getOrgName());
                    mdmDataPermissionOrgVo.setOrgType(mdmOrgEntity.getOrgType());
                    mdmDataPermissionOrgVo.setOrgTypeName(DictUtil.dictValue(DictConstant.MDM_ORG_TYPE, mdmOrgEntity.getOrgType()));
                    String permissionSearchType = x.getPermissionSearchType();
                    mdmDataPermissionOrgVo.setPermissionSearchType(permissionSearchType);
                    String permissionObjCode = x.getPermissionObjCode();
                    if (!StringUtils.isEmpty(permissionSearchType) && !StringUtils.isEmpty(permissionObjCode)) {
                        String subDictTypeCode = MdmPermissionObjEnum.getSubDictTypeCode(permissionObjCode);
                        if (!StringUtils.isEmpty(subDictTypeCode)) {
                            mdmDataPermissionOrgVo.setPermissionSearchTypeName(DictUtil.dictValue(subDictTypeCode, permissionSearchType));
                        }
                    }
                    return mdmDataPermissionOrgVo;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    protected List<MdmDataPermissionPositionVo> setUpPosition(List<MdmDataPermissionExtendEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<String> collect = list.stream()
                .filter(x -> !StringUtils.isEmpty(x.getCode()))
                .map(MdmDataPermissionExtendEntity::getCode)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(collect)) {
            return new ArrayList<>();
        }
        return mdmPositionService.findPositionAndOrgList(collect);
    }


    @Override
    public MdmCurrentPermissionRespVo currentUserDataPermission(String competenceCode, String listConfigCode) {
        MdmCurrentPermissionRespVo mdmCurrentPermissionRespVo = new MdmCurrentPermissionRespVo();
        mdmCurrentPermissionRespVo.setUserPermissionList(new ArrayList<>());
        mdmCurrentPermissionRespVo.setPermissionObjCodeList(new ArrayList<>());

        if (StringUtils.isEmpty(competenceCode) || StringUtils.isEmpty(listConfigCode)) {
            return mdmCurrentPermissionRespVo;
        }
        UserRedis user = UserUtils.getUser();
        Assert.notNull(user, "当前用户不存在");
        log.info("[数据权限]获取当前用户权限,用户信息:{}", user);
        String username = user.getUsername();
        Assert.hasText(username, "用户帐号不能为空");
        if (AdminEnum.ADMIN.name().equalsIgnoreCase(username)) {
            return mdmCurrentPermissionRespVo;
        }
        String usertype = user.getUsertype();
        Assert.hasText(usertype, "用户类型不能为空");
        String positionCode = user.getPoscode();
        List<String> roleCodeList = mdmRoleService.getRoleByUserName(username, positionCode);

        Assert.notEmpty(roleCodeList, "当前用户:"+username+"未分配角色");
        if (roleCodeList.stream().anyMatch(x -> AdminEnum.ADMIN.name().equalsIgnoreCase(x))) {
            return mdmCurrentPermissionRespVo;
        }

        MdmFunctionSubEntity mdmFunctionSubEntity = mdmFunctionSubService.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, competenceCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, listConfigCode)
                .select(MdmFunctionSubEntity::getPermissionObj)
                .one();
        log.info("[数据权限][菜单权限]{}", mdmFunctionSubEntity);
        if (mdmFunctionSubEntity == null || StringUtils.isEmpty(mdmFunctionSubEntity.getPermissionObj())) {
            return mdmCurrentPermissionRespVo;
        }
        String permissionObj = mdmFunctionSubEntity.getPermissionObj();
        mdmCurrentPermissionRespVo.setPermissionObjCodeList(Arrays.asList(permissionObj.split(",")));
        List<MdmDataPermissionExtendEntity> permissionExtendEntities = mdmDataPermissionExtendService.lambdaQuery()
                .eq(MdmDataPermissionExtendEntity::getFunctionCode, competenceCode)
                .eq(MdmDataPermissionExtendEntity::getListConfigCode, listConfigCode)
                .eq(MdmDataPermissionExtendEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .list();
        log.info("用户角色编码:{}，职位:{}", roleCodeList, positionCode);
        List<MdmDataPermissionExtendEntity> userRolePermissions = permissionExtendEntities.stream()
                .filter(x -> !StringUtils.isEmpty(x.getRoleCode()))
                .filter(x -> roleCodeList.contains(x.getRoleCode()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userRolePermissions)) {
            return mdmCurrentPermissionRespVo;
        }
        handlePosition(userRolePermissions, mdmCurrentPermissionRespVo, positionCode);
        handleOrg(userRolePermissions, mdmCurrentPermissionRespVo, user);
        handleCustomer(userRolePermissions, mdmCurrentPermissionRespVo, user);
        return mdmCurrentPermissionRespVo;
    }

    protected void handleCustomer(List<MdmDataPermissionExtendEntity> userRolePermissions, MdmCurrentPermissionRespVo mdmCurrentPermissionRespVo, UserRedis user) {
        String orgCode = user.getOrgcode();
        String positionCode = user.getPoscode();
        List<String> customerCodeList = new ArrayList<>();
        List<MdmDataPermissionExtendEntity> extendEntities = userRolePermissions.stream()
                .filter(x -> MdmPermissionObjEnum.CUSTOMER.getCode().equals(x.getPermissionObjCode()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(extendEntities)) {
            return;
        }
        Map<String, List<MdmDataPermissionExtendEntity>> resourceMap = extendEntities.stream()
                .collect(Collectors.groupingBy(MdmDataPermissionExtendEntity::getPermissionResource));
        List<MdmDataPermissionExtendEntity> presuppositionList = resourceMap.get(PermissionResourceEnum.PRESUPPOSITION.getCode());
        if (!CollectionUtils.isEmpty(presuppositionList)) {
            for (MdmDataPermissionExtendEntity presuppositionPermission : presuppositionList) {
                switch (MdmPermissionExtensionEnum.convertEnum(presuppositionPermission.getPermissionSearchType())) {
                    case ALL:
                        mdmCurrentPermissionRespVo.setUpExclude(MdmPermissionObjEnum.CUSTOMER.getCode());
                        break;
                    case CURRENT_ORG:
                        if (!StringUtils.isEmpty(orgCode)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByOrgCodeList(Collections.singletonList(orgCode));
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case CURRENT_SUB_ORG:
                        List<String> allSub = OrgUtil.getChildrenOrgCodeListIncludeSelfEnable(orgCode);
                        if (!CollectionUtils.isEmpty(allSub)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByOrgCodeList(allSub);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case CURRENT_UP_ORG:
                        List<String> allParentOrgCodeList = OrgUtil.getParentOrgCodeListIncludeSelf(orgCode);
                        if (!CollectionUtils.isEmpty(allParentOrgCodeList)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByOrgCodeList(allParentOrgCodeList);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case SUB_ORG:
                        List<String> subOrgCodeList = OrgUtil.getChildrenOrgCodeListExcludeSelfEnable(orgCode);
                        if (!CollectionUtils.isEmpty(subOrgCodeList)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByOrgCodeList(subOrgCodeList);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case CURRENT_POSITION:
                        if (!StringUtils.isEmpty(positionCode)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByPositionCodeList(Collections.singletonList(positionCode));
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case CURRENT_SUB_POSITION:
                        List<String> currentSubPositionCodeList = PositionUtil.getChildrenPositionCodeListIncludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(currentSubPositionCodeList)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByPositionCodeList(currentSubPositionCodeList);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                    case CURRENT_UP_POSITION:
                        List<String> currentUpPositionCodeList = PositionUtil.getParentPositionCodeListIncludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(currentUpPositionCodeList)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByPositionCodeList(currentUpPositionCodeList);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case SUB_POSITION:
                        List<String> subPositionCodeList = PositionUtil.getChildrenPositionCodeListExcludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(subPositionCodeList)) {
                            List<String> currentCustomerCodeList = mdmCustomerMsgService.findByPositionCodeList(subPositionCodeList);
                            if (!CollectionUtils.isEmpty(currentCustomerCodeList)) {
                                customerCodeList.addAll(currentCustomerCodeList);
                            }
                        }
                        break;
                    case CURRENT_CUSTOMER:
                        String usertype = user.getUsertype();
                        if (UserTypeEnum.CUSTOMER.getCode().equals(usertype) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(usertype)) {
                            String customerCode = user.getCustcode();
                            if (!StringUtils.isEmpty(customerCode)) {
                                customerCodeList.add(customerCode);
                            }
                        }
                    default:
                }
            }
        }
        List<MdmDataPermissionExtendEntity> fixedList = resourceMap.get(PermissionResourceEnum.FIXED.getCode());
        if (!CollectionUtils.isEmpty(fixedList)) {
            List<String> fixedCustomerCodeList = userRolePermissions.stream()
                    .filter(x -> MdmPermissionObjEnum.CUSTOMER.getCode().equals(x.getPermissionObjCode()))
                    .filter(x -> PermissionResourceEnum.FIXED.getCode().equals(x.getPermissionResource()))
                    .filter(x -> !StringUtils.isEmpty(x.getCode()))
                    .map(MdmDataPermissionExtendEntity::getCode)
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(fixedCustomerCodeList)) {
                customerCodeList.addAll(fixedCustomerCodeList);
            }
        }

        if (!CollectionUtils.isEmpty(customerCodeList)) {
            MdmCurrentUserPermissionRespVo mdmCurrentUserPermissionRespVo = new MdmCurrentUserPermissionRespVo();
            mdmCurrentUserPermissionRespVo.setPermissionObjCode(MdmPermissionObjEnum.CUSTOMER.getCode());
            mdmCurrentUserPermissionRespVo.setPermissionValueList(customerCodeList);
            mdmCurrentPermissionRespVo.getUserPermissionList().add(mdmCurrentUserPermissionRespVo);
        }
    }

    /**
     * 设置当前权限配置的组织
     *
     * @param userRolePermissions
     * @param mdmCurrentPermissionRespVo
     * @param user
     */
    protected void handleOrg(List<MdmDataPermissionExtendEntity> userRolePermissions, MdmCurrentPermissionRespVo mdmCurrentPermissionRespVo, UserRedis user) {
        String orgCode = user.getOrgcode();
        if (StringUtils.isEmpty(orgCode)) {
            return;
        }
        List<String> orgCodeList = new ArrayList<>();
        List<MdmDataPermissionExtendEntity> presuppositionPermissions = userRolePermissions.stream()
                .filter(x -> MdmPermissionObjEnum.ORG.getCode().equals(x.getPermissionObjCode()))
                .filter(x -> PermissionResourceEnum.PRESUPPOSITION.getCode().equals(x.getPermissionResource()))
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(presuppositionPermissions)) {
            for (MdmDataPermissionExtendEntity presuppositionPermission : presuppositionPermissions) {
                switch (MdmPermissionLevelEnum.convertEnum(presuppositionPermission.getPermissionSearchType())) {
                    case ALL:
                        mdmCurrentPermissionRespVo.setUpExclude(MdmPermissionObjEnum.ORG.getCode());
                        orgCodeList.add(orgCode);
                        break;
                    case CURRENT:
                        orgCodeList.add(orgCode);
                        break;
                    case CURRENT_SUB:
                        List<String> allSub = OrgUtil.getChildrenOrgCodeListIncludeSelfEnable(orgCode);
                        if (!CollectionUtils.isEmpty(allSub)) {
                            orgCodeList.addAll(allSub);
                        }
                        break;
                    case CURRENT_UP:
                        List<String> allParentOrgCodeList = OrgUtil.getParentOrgCodeListIncludeSelf(orgCode);
                        if (!CollectionUtils.isEmpty(allParentOrgCodeList)) {
                            orgCodeList.addAll(allParentOrgCodeList);
                        }
                        break;
                    default:
                }
            }
        }
        List<MdmDataPermissionExtendEntity> fixedPermissions = userRolePermissions.stream()
                .filter(x -> MdmPermissionObjEnum.ORG.getCode().equals(x.getPermissionObjCode()))
                .filter(x -> PermissionResourceEnum.FIXED.getCode().equals(x.getPermissionResource()))
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(fixedPermissions)) {
            for (MdmDataPermissionExtendEntity fixedPermission : fixedPermissions) {
                switch (MdmPermissionLevelEnum.convertEnum(fixedPermission.getPermissionSearchType())) {
                    case CURRENT:
                        orgCodeList.add(fixedPermission.getCode());
                        break;
                    case CURRENT_SUB:
                        List<String> allChildrenOrgCodeList = mdmOrgService.findAllChildrenOrgCodeList(null, fixedPermission.getCode());
                        if (!CollectionUtils.isEmpty(allChildrenOrgCodeList)) {
                            orgCodeList.addAll(allChildrenOrgCodeList);
                        }
                        break;
                    default:
                }
            }
        }
        if (!CollectionUtils.isEmpty(orgCodeList)) {
            MdmCurrentUserPermissionRespVo mdmCurrentUserPermissionRespVo = new MdmCurrentUserPermissionRespVo();
            mdmCurrentUserPermissionRespVo.setPermissionObjCode(MdmPermissionObjEnum.ORG.getCode());
            mdmCurrentUserPermissionRespVo.setPermissionValueList(orgCodeList);
            mdmCurrentPermissionRespVo.getUserPermissionList().add(mdmCurrentUserPermissionRespVo);
        }
    }

    /**
     * 设置当前权限配置的职位
     *
     * @param userRolePermissions
     * @param mdmCurrentPermissionRespVo
     * @param positionCode
     */
    protected void handlePosition(List<MdmDataPermissionExtendEntity> userRolePermissions, MdmCurrentPermissionRespVo mdmCurrentPermissionRespVo, String positionCode) {
        List<String> positionCodeList = new ArrayList<>();
        List<MdmDataPermissionExtendEntity> presuppositionPermissions = userRolePermissions.stream()
                .filter(x -> MdmPermissionObjEnum.POSITION.getCode().equals(x.getPermissionObjCode()))
                .filter(x -> PermissionResourceEnum.PRESUPPOSITION.getCode().equals(x.getPermissionResource()))
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(presuppositionPermissions)) {
            for (MdmDataPermissionExtendEntity presuppositionPermission : presuppositionPermissions) {
                switch (MdmPermissionLevelEnum.convertEnum(presuppositionPermission.getPermissionSearchType())) {
                    case ALL:
                        mdmCurrentPermissionRespVo.setUpExclude(MdmPermissionObjEnum.POSITION.getCode());
                        positionCodeList.add(positionCode);
                        break;
                    case CURRENT:
                        positionCodeList.add(positionCode);
                        break;
                    case CURRENT_SUB:
                        List<String> allSub = PositionUtil.getChildrenPositionCodeListIncludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(allSub)) {
                            positionCodeList.addAll(allSub);
                        }
                        break;
                    case CURRENT_UP:
                        List<String> allParent = PositionUtil.getParentPositionCodeListIncludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(allParent)) {
                            positionCodeList.addAll(allParent);
                        }
                        break;
                    case SUB_POSITION:
                        List<String> allSubExceptCurrent = PositionUtil.getChildrenPositionCodeListExcludeSelfEnable(positionCode);
                        if (!CollectionUtils.isEmpty(allSubExceptCurrent)) {
                            positionCodeList.addAll(allSubExceptCurrent);
                        }
                        break;
                    case USER_POSITION:
                        String username = Optional.ofNullable(UserUtils.getUser())
                                .map(UserRedis::getUsername)
                                .orElse(null);
                        List<MdmPositionRespVo> userAllPositionList = mdmPositionService.getUserAllPositionList(username);
                        if (!CollectionUtils.isEmpty(userAllPositionList)) {
                            List<String> collect = userAllPositionList.stream()
                                    .map(MdmPositionRespVo::getPositionCode)
                                    .filter(x -> !StringUtils.isEmpty(x))
                                    .collect(Collectors.toList());
                            if (!CollectionUtils.isEmpty(collect)) {
                                positionCodeList.addAll(collect);
                            }
                        }
                    default:
                }
            }
        }
        List<String> fixedPositionCodeList = userRolePermissions.stream()
                .filter(x -> MdmPermissionObjEnum.POSITION.getCode().equals(x.getPermissionObjCode()))
                .filter(x -> PermissionResourceEnum.FIXED.getCode().equals(x.getPermissionResource()))
                .filter(x -> !StringUtils.isEmpty(x.getCode()))
                .map(MdmDataPermissionExtendEntity::getCode)
                .distinct()
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(fixedPositionCodeList)) {
            positionCodeList.addAll(fixedPositionCodeList);
        }
        if (!CollectionUtils.isEmpty(positionCodeList)) {
            MdmCurrentUserPermissionRespVo mdmCurrentUserPermissionRespVo = new MdmCurrentUserPermissionRespVo();
            mdmCurrentUserPermissionRespVo.setPermissionObjCode(MdmPermissionObjEnum.POSITION.getCode());
            mdmCurrentUserPermissionRespVo.setPermissionValueList(positionCodeList);
            mdmCurrentPermissionRespVo.getUserPermissionList().add(mdmCurrentUserPermissionRespVo);
        }
    }

    @Override
    public List<MdmPermissionObjRespVo> objList(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        List<DictDataVo> permissionTree = DictUtil.tree(DictConstant.PERMISSION_OBJ);
        if (CollectionUtils.isEmpty(permissionTree)) {
            return new ArrayList<>();
        }
        String functionCode = mdmDataPermissionReqVo.getFunctionCode();
        String listConfigCode = mdmDataPermissionReqVo.getListConfigCode();

        Assert.hasText(functionCode, "菜单编码不能为空");
        Assert.hasText(listConfigCode, "列表配置编码不能为空");
        MdmFunctionSubEntity mdmFunctionSubEntity = mdmFunctionSubService.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, functionCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, listConfigCode)
                .select(MdmFunctionSubEntity::getPermissionObj)
                .one();
        log.info("[功能列表][已配置的权限对象]:{}", mdmFunctionSubEntity);
        if (mdmFunctionSubEntity == null) {
            return new ArrayList<>();
        }
        String permissionObj = mdmFunctionSubEntity.getPermissionObj();
        if (com.biz.crm.util.StringUtils.isEmpty(permissionObj)) {
            return new ArrayList<>();
        }
        List<String> permissionObjList = Arrays.asList(permissionObj.split(","));
        return permissionTree
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getDictCode()) && permissionObjList.contains(x.getDictCode()))
                .map(x -> {
                    MdmPermissionObjRespVo mdmPermissionObjRespVo = new MdmPermissionObjRespVo();
                    mdmPermissionObjRespVo.setPermissionObjCode(x.getDictCode());
                    mdmPermissionObjRespVo.setPermissionObjName(x.getDictValue());
                    return mdmPermissionObjRespVo;
                }).collect(Collectors.toList());
    }


    @Override
    public Map<String, String> searchTypeList(String permissionObjCode) {
        return DictUtil.dictMap(MdmPermissionObjEnum.getSubDictTypeCode(permissionObjCode));
    }

    @Override
    public List<MdmFunctionRespVo> functionSelect(MdmFunctionReqVo mdmFunctionReqVo) {
        return mdmDataPermissionMapper.functionSelect(mdmFunctionReqVo);
    }

    @Override
    public List<MdmRoleRespVo> roleSelectList(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        return mdmDataPermissionMapper.roleFunctionSelectList(mdmDataPermissionReqVo);
    }

    @Override
    public List<MdmListConfigRespVo> configList(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        String functionCode = mdmDataPermissionReqVo.getFunctionCode();
        Assert.hasText(functionCode, "菜单编码不能为空");
        MdmFunctionSubReqVo mdmFunctionSubReqVo = new MdmFunctionSubReqVo();
        mdmFunctionSubReqVo.setParentCode(functionCode);
        mdmFunctionSubReqVo.setFunctionType(YesNoEnum.yesNoEnum.ONE.getValue());
        List<MdmFunctionSubRespVo> subList = mdmFunctionSubService.findSubList(mdmFunctionSubReqVo);
        if (CollectionUtils.isEmpty(subList)) {
            return new ArrayList<>();
        }
        return subList.stream().map(x -> {
            MdmListConfigRespVo mdmListConfigRespVo = new MdmListConfigRespVo();
            mdmListConfigRespVo.setListConfigCode(x.getFunctionCode());
            mdmListConfigRespVo.setListConfigName(x.getFunctionName());
            return mdmListConfigRespVo;
        }).collect(Collectors.toList());
    }

    @Override
    public List<MdmFunctionRespVo> functionTree(MdmFunctionReqVo mdmFunctionReqVo) {
        List<MdmFunctionEntity> list = mdmFunctionService.lambdaQuery().list();
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<MdmFunctionRespVo> mdmFunctionRespVos = CrmBeanUtil.copyList(list, MdmFunctionRespVo.class);
        List<String> relationFunctionCodeList = new ArrayList<>();
        if (!StringUtils.isEmpty(mdmFunctionReqVo.getRoleCode())) {
            relationFunctionCodeList = mdmDataPermissionMapper.functionCodeSelect(mdmFunctionReqVo);
        }
        List<String> permissionFunctionCodeList = mdmFunctionSubService.lambdaQuery()
                .isNotNull(MdmFunctionSubEntity::getPermissionObj)
                .select(MdmFunctionSubEntity::getParentCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getParentCode()))
                .map(MdmFunctionSubEntity::getParentCode)
                .collect(Collectors.toList());
        log.info("[功能列表][已配置的权限对象]:{}", permissionFunctionCodeList);

        for (MdmFunctionRespVo mdmFunctionRespVo : mdmFunctionRespVos) {
            boolean permission = permissionFunctionCodeList.contains(mdmFunctionRespVo.getFunctionCode());
            mdmFunctionRespVo.setFunctionPermission(permission ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        return mdmFunctionService.constructFunctionTree(mdmFunctionReqVo.getFunctionName(), mdmFunctionRespVos, relationFunctionCodeList);
    }

    @Override
    public List<MdmPermissionSetVo> currentUserPermissionSql() {
        String menuCode = (String) ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        String functionCode = (String) ThreadLocalUtil.getObj(GlobalParam.FUNCTION_CODE);
        log.info("[数据权限]menuCode:{},functionCode:{}", menuCode, functionCode);
        if (StringUtils.isEmpty(menuCode) || StringUtils.isEmpty(functionCode)) {
            return new ArrayList<>();
        }
        MdmCurrentPermissionRespVo mdmCurrentPermissionRespVo = currentUserDataPermission(menuCode, functionCode);
        log.info("[数据权限] vo:{}", mdmCurrentPermissionRespVo);
        List<MdmCurrentUserPermissionRespVo> userPermissionList = mdmCurrentPermissionRespVo.getUserPermissionList();
        if (CollectionUtils.isEmpty(userPermissionList)) {
            return new ArrayList<>();
        }
        List<String> excludePermissionObjCodeList = mdmCurrentPermissionRespVo.getExcludePermissionObjCodeList();
        return userPermissionList.stream()
                .filter(x -> !CollectionUtils.isEmpty(x.getPermissionValueList()))
                .filter(x -> {
                    String permissionObjCode = x.getPermissionObjCode();
                    if (CollectionUtils.isEmpty(excludePermissionObjCodeList)) {
                        return true;
                    }
                    return excludePermissionObjCodeList.contains(permissionObjCode);
                })
                .map(x -> {
                    MdmPermissionSetVo mdmPermissionSetVo = new MdmPermissionSetVo();
                    mdmPermissionSetVo.setPermissionObjCode(x.getPermissionObjCode());
                    mdmPermissionSetVo.setStatement("in " + SqlUtil.bracketsSetToReplaceStr(new HashSet<>(x.getPermissionValueList())));
                    return mdmPermissionSetVo;
                }).collect(Collectors.toList());
    }

    @Override
    public void updateAllSub(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        String roleCode = mdmDataPermissionReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        mdmDataPermissionReqVo.setId(null);
        List<String> functionCodeList = mdmDataPermissionReqVo.getFunctionCodeList();
        if (!CollectionUtils.isEmpty(functionCodeList)) {
            MdmFunctionSubReqVo mdmFunctionSubReqVo = new MdmFunctionSubReqVo();
            mdmFunctionSubReqVo.setParentCodeList(functionCodeList);
            mdmFunctionSubReqVo.setFunctionType(MdmTableConfigFunctionTypeEnum.LIST_CONFIG.getCode());
            List<MdmFunctionSubRespVo> subList = mdmFunctionSubService.findPermissionList(mdmFunctionSubReqVo);
            if (!CollectionUtils.isEmpty(subList)) {
                //原使用的并行流处理，为了保存日志更改为串行流处理
                subList.stream()
                        .map(x -> {
                            MdmDataPermissionReqVo copy = CrmBeanUtil.copy(mdmDataPermissionReqVo, MdmDataPermissionReqVo.class);
                            copy.setFunctionCode(x.getParentCode());
                            copy.setListConfigCode(x.getFunctionCode());
                            return copy;
                        }).forEach(this::save);
            }
        }
    }

    @Override
    public Map<String, List<String>> findDataPermissionMap(MdmDataPermissionReqVo mdmDataPermissionReqVo) {
        return this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmDataPermissionReqVo.getFunctionCode()), MdmDataPermissionEntity::getFunctionCode, mdmDataPermissionReqVo.getFunctionCode())
                .in(!CollectionUtils.isEmpty(mdmDataPermissionReqVo.getListConfigCodeList()), MdmDataPermissionEntity::getListConfigCode, mdmDataPermissionReqVo.getListConfigCodeList())
                .select(MdmDataPermissionEntity::getFunctionCode, MdmDataPermissionEntity::getListConfigCode)
                .list()
                .stream()
                .collect(Collectors.groupingBy(MdmDataPermissionEntity::getFunctionCode, Collectors.mapping(MdmDataPermissionEntity::getListConfigCode, Collectors.toList())));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmDataPermissionVo reqVo) {
        String listConfigCode = reqVo.getListConfigCode();
        Assert.hasText(listConfigCode, "列表编码不能为空");
        String functionCode = reqVo.getFunctionCode();
        Assert.hasText(functionCode, "菜单编码不能为空");
        String roleCode = reqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        this.lambdaUpdate()
                .eq(MdmDataPermissionEntity::getRoleCode, roleCode)
                .eq(MdmDataPermissionEntity::getFunctionCode, functionCode)
                .eq(MdmDataPermissionEntity::getListConfigCode, listConfigCode)
                .remove();

        mdmDataPermissionExtendService.lambdaUpdate()
                .eq(MdmDataPermissionExtendEntity::getRoleCode, roleCode)
                .eq(MdmDataPermissionExtendEntity::getFunctionCode, functionCode)
                .eq(MdmDataPermissionExtendEntity::getListConfigCode, listConfigCode)
                .remove();
        mdmDataPermissionContext.save(reqVo);
        this.save(CrmBeanUtil.copy(reqVo, MdmDataPermissionEntity.class));
    }

    @Override
    public MdmDataPermissionVo queryDetail(MdmDataPermissionVo mdmDataPermissionVo) {
        Assert.hasText(mdmDataPermissionVo.getFunctionCode(), "菜单编码不能为空");
        Assert.hasText(mdmDataPermissionVo.getListConfigCode(), "列表编码不能为空");
        Assert.hasText(mdmDataPermissionVo.getRoleCode(), "角色编码不能为空");
        List<MdmDataPermissionDetailVo> list = mdmDataPermissionExtendService.listCondition(mdmDataPermissionVo);
        mdmDataPermissionVo.setMdmDataPermissionDetailVos(list);
        mdmDataPermissionContext.populate(mdmDataPermissionVo);
        return mdmDataPermissionVo;
    }

}
