package com.biz.crm.poi.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapDistrictRespVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapDistrictStatisticianRespVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmTerminalStatisticianTypeTotalVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmTerminalStatisticianTypeVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmAmapAreaReqVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmAmapDistrictStatisticianReqVo;
import com.biz.crm.poi.service.MdmAmapStatisticianService;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.util.List;

/**
 * @author zxw
 * @date 2021-05-14 14:58
 * 客户与终端分布统计
 **/
@RestController
@RequestMapping("/mdmAmapStatisticsController")
@Api(tags = "mdm-客户与终端分布统计")
@CrmGlobalLog
@Slf4j
public class MdmAmapStatisticianController {

    @Autowired
    private MdmAmapStatisticianService mdmAmapStatisticianService;

    @ApiOperation("查询查询所有类型统计")
    @GetMapping("/statisticalTotal")
    public Result<MdmTerminalStatisticianTypeTotalVo> statisticalTotal() {
        return Result.ok(mdmAmapStatisticianService.statisticalTotal());
    }

    @ApiOperation("查询已合作客户或终端类型统计")
    @GetMapping("/statisticalType")
    public Result<List<MdmTerminalStatisticianTypeVo>> statisticalType() {
        MdmTerminalStatisticianTypeTotalVo result = new MdmTerminalStatisticianTypeTotalVo();
        mdmAmapStatisticianService.statisticalType(result);
        return Result.ok(result.getCooperationTypeList());
    }

    @ApiOperation("查询已合作客户或终端区域统计")
    @PostMapping("/statisticalDistrict")
    public Result<List<MdmAmapDistrictStatisticianRespVo>> statisticalDistrict(@RequestBody MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        return Result.ok(mdmAmapStatisticianService.statisticalDistrict(mdmAmapDistrictStatisticianReqVo));
    }

    @ApiOperation("查询未合作客户或终端类型统计")
    @GetMapping("/unCooperationStatisticalType")
    public Result<List<MdmTerminalStatisticianTypeVo>> unCooperationStatisticalType() {
        MdmTerminalStatisticianTypeTotalVo result = new MdmTerminalStatisticianTypeTotalVo();
        mdmAmapStatisticianService.unCooperationStatisticalType(result);
        return Result.ok(result.getUnUooperationTypeList());
    }

    @ApiOperation("查询未合作区域统计")
    @PostMapping("/unCooperationStatisticalDistrict")
    public Result<List<MdmAmapDistrictStatisticianRespVo>> unCooperationStatisticalDistrict(@RequestBody MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        return Result.ok(mdmAmapStatisticianService.unCooperationStatisticalDistrict(mdmAmapDistrictStatisticianReqVo));
    }

    @ApiOperation("查询已合作的打点统计")
    @PostMapping("/district")
    public Result<List<MdmAmapDistrictRespVo>> district(@RequestBody MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        return Result.ok(mdmAmapStatisticianService.district(mdmAmapDistrictStatisticianReqVo));
    }

    @ApiOperation("查询未合作的打点统计")
    @PostMapping("/unCooperationDistrict")
    public Result<List<MdmAmapDistrictRespVo>> unCooperationDistrict(@RequestBody MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        return Result.ok(mdmAmapStatisticianService.unCooperationDistrict(mdmAmapDistrictStatisticianReqVo));
    }

    @ApiOperation("获取地图静态资源")
    @PostMapping("/obtainMapData")
    public ResponseEntity<byte[]> obtainMapData(@RequestBody MdmAmapAreaReqVo mdmAmapAreaReqVo) throws IOException {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Cache-Control", "no-cache, no-store, must-revalidate");
        headers.add("Content-Disposition", "attachment; filename=map.json");
        headers.add("Pragma", "no-cache");
        headers.add("Expires", "0");
        headers.add("Last-Modified", LocalDateTime.now().toString());
        headers.add("ETag", String.valueOf(System.currentTimeMillis()));
        String amapDistrictCode = mdmAmapAreaReqVo.getAmapDistrictCode();
        String province = mdmAmapAreaReqVo.getProvince();
        StringBuilder sb = new StringBuilder();
        sb.append(File.separator);
        sb.append("map");
        sb.append(File.separator);
        if (!StringUtils.isEmpty(amapDistrictCode)) {
            sb.append("citys").append(File.separator).append(".json");
        } else if (!StringUtils.isEmpty(province)) {
            sb.append("province").append(File.separator).append(".json");
        } else {
            sb.append("china.json");
        }
        InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(sb.toString());
        Assert.notNull(inputStream, sb.toString() + "资源不存在");
        byte[] body = null;
        try {
            body = new byte[inputStream.available()];
            inputStream.read(body);
            return ResponseEntity
                    .ok()
                    .headers(headers)
                    .contentType(MediaType.parseMediaType("application/octet-stream"))
                    .body(body);
        } catch (IOException e) {
            log.error("获取地图资源失败:{}",e);
            return null;
        } finally {
            inputStream.close();
        }
    }
}
