package com.biz.crm.position.controller;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.position.req.*;
import com.biz.crm.nebular.mdm.position.resp.*;
import com.biz.crm.nebular.mdm.user.req.MdmUserReqVo;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.util.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

/**
 * 职位表
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Slf4j
@RestController
@RequestMapping("/mdmPositionController")
@Api(tags = "MDM-职位管理")
public class MdmPositionController {

    @Autowired
    private MdmPositionService mdmPositionService;

    @ApiOperation(value = "分页列表查询")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmPositionPageRespVo>> pageList(@RequestBody MdmPositionPageReqVo reqVo) {
        PageResult<MdmPositionPageRespVo> result = mdmPositionService.findList(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "详情(页面编辑回显数据使用)", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "positionCode", value = "编码编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmDictMethod
    public Result<MdmPositionRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "positionCode", required = false) String positionCode) {
        MdmPositionRespVo respVo = mdmPositionService.detail(id, positionCode);
        return Result.ok(respVo);
    }

    @ApiOperation(value = "根据职位id或者职位编码查询职位主表，以及职位关联用户、组织、角色、流程角色，查询数据较多，请谨慎使用", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "positionCode", value = "编码编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/getPositionDetail")
    @CrmDictMethod
    public Result<MdmPositionDetailRespVo> getPositionDetail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "positionCode", required = false) String positionCode) {
        return Result.ok(mdmPositionService.getPositionDetail(id, positionCode));
    }

    @ApiOperation(value = "通过id集合批量查询职位全表及关联用户、组织、上级的主要字段", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/detailBatchByIds")
    public Result<List<MdmPositionUserOrgRespVo>> detailBatchByIds(@RequestBody List<String> ids) {
        return Result.ok(mdmPositionService.detailBatch(ids, null));
    }

    @ApiOperation(value = "通过职位编码集合批量查询职位全表及关联用户、组织、上级的主要字段", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "positionCodeList", value = "职位编码集合", required = true, paramType = "body")
    })
    @PostMapping("/detailBatchByPositionCodeList")
    public Result<List<MdmPositionUserOrgRespVo>> detailBatchByPositionCodeList(@RequestBody List<String> positionCodeList) {
        return Result.ok(mdmPositionService.detailBatch(null, positionCodeList));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result save(@RequestBody MdmPositionReqVo mdmPositionReqVo) {
        mdmPositionService.save(mdmPositionReqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result update(@RequestBody MdmPositionReqVo mdmPositionReqVo) {
        mdmPositionService.update(mdmPositionReqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result delete(@RequestBody List<String> ids) {
        mdmPositionService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    public Result enable(@RequestBody List<String> ids) {
        mdmPositionService.enableBatch(ids);
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    public Result disable(@RequestBody List<String> ids) {
        mdmPositionService.disableBatch(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation(value = "查询职位下拉框列表")
    @PostMapping("/positionSelectList")
    public Result<List<MdmPositionSelectRespVo>> positionSelectList(@RequestBody MdmPositionSelectReqVo reqVo) {
        return Result.ok(mdmPositionService.findPositionSelectList(reqVo));
    }

    @ApiOperation(value = "根据userName获取用户主职位", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName", value = "用户登录名", dataType = "String", paramType = "query"),
    })
    @GetMapping("/getPrimaryPositionByUserName")
    public Result<MdmPositionRespVo> getPrimaryPositionByUserName(@RequestParam("userName") String userName) {
        return Result.ok(mdmPositionService.getUserPrimaryPosition(new MdmUserReqVo().setUserName(userName)));
    }

    @ApiOperation(value = "（不推荐使用）根据userName获取用户当前职位", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName", value = "用户登录名", dataType = "String", paramType = "query"),
    })
    @GetMapping("/getCurrentPositionByUserName")
    public Result<MdmPositionUserOrgRespVo> getCurrentPositionByUserName(@RequestParam("userName") String userName) {
        return Result.ok(mdmPositionService.getCurrentPositionByUserName(userName));
    }

    @ApiOperation(value = "根据userName获取用户全部职位")
    @PostMapping("/getUserAllPositionList")
    public Result<List<MdmPositionRespVo>> getUserAllPositionList(@RequestBody MdmUserReqVo reqVo) {
        return Result.ok(mdmPositionService.getUserAllPositionList(reqVo.getUserName()));
    }

    @ApiOperation(value = "（已废弃）根据用户编码（userCode）获取用户全部职位", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userCode", value = "用户编码", dataType = "String", paramType = "query"),
    })
    @GetMapping("/getUserPositionByUserCode")
    public Result<List<MdmPositionRespVo>> getUserPositionByUserCode(@RequestParam("userCode") String userCode) {
        return Result.ok(mdmPositionService.getUserPositionByUserCode(userCode));
    }

    @ApiOperation(value = "切换用户主职位")
    @PostMapping("/changeUserPrimaryPosition")
    public Result changeUserPrimaryPosition(@RequestBody MdmPositionUserOrgOptReqVo reqVo) {
        mdmPositionService.changeUserPrimaryPosition(reqVo.getUserName(), reqVo.getPositionCode());
        return Result.ok();
    }

    @ApiOperation(value = "切换用户当前职位")
    @PostMapping("/changeUserCurrentPosition")
    public Result changeUserCurrentPosition(@RequestBody MdmPositionUserOrgOptReqVo reqVo) {
        mdmPositionService.changeUserCurrentPosition(reqVo.getUserName(), reqVo.getPositionCode());
        return Result.ok();
    }

    @ApiOperation(value = "根据职位编码查询职位主表及关联用户、组织、上级的主要字段", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "positionCode", value = "职位编码", required = true, dataType = "String", paramType = "query"),
    })
    @GetMapping("/getPositionUserOrgByPositionCode")
    public Result<MdmPositionUserOrgRespVo> getPositionUserOrgByPositionCode(@RequestParam("positionCode") String positionCode) {
        return Result.ok(mdmPositionService.getPositionUserOrgByPositionCode(positionCode));
    }

    @ApiOperation(value = "查询以职位维度的职位全表及关联用户、组织、上级的主要字段")
    @PostMapping("/findPositionUserOrgList")
    public Result<List<MdmPositionUserOrgRespVo>> findPositionUserOrgList(@RequestBody MdmPositionUserOrgReqVo reqVo) {
        return Result.ok(mdmPositionService.findPositionUserOrgList(reqVo));
    }

    @ApiOperation(value = "查询以职位维度的职位全表及关联用户、组织、上级的主要字段（分页），部分字段为模糊查询，一般用于以职位为主体的弹框")
    @PostMapping("/findPositionUserOrgPageList")
    @CrmDictMethod
    public Result<PageResult<MdmPositionUserOrgPageRespVo>> findPositionUserOrgPageList(@RequestBody MdmPositionUserOrgPageReqVo reqVo) {
        return Result.ok(mdmPositionService.findPositionUserOrgPageList(reqVo));
    }

    @ApiOperation(value = "（SFA专用）以当前登录人查询本人及同组织的人及下属（分页）")
    @PostMapping("/findCurrentPositionUserOrgPageList")
    public Result<PageResult<MdmPositionUserOrgPageRespVo>> findCurrentPositionUserOrgPageList(@RequestBody MdmPositionCurrentUserReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        if (null == user) {
            throw new BusinessException("无法获取登陆信息，请重新登陆");
        }
        String orgCode = user.getOrgcode();
        if (StringUtils.isBlank(orgCode)) {
            throw new BusinessException("无法获取登陆者当前组织信息，请重新登陆");
        }
        MdmPositionUserOrgPageReqVo orgReqVo = CrmBeanUtil.copy(reqVo, MdmPositionUserOrgPageReqVo.class);
        if (null == reqVo.getCurrentLoginOrg() || reqVo.getCurrentLoginOrg()) {
            orgReqVo.setAllUnderThisOrgCode(orgCode);
        } else {
            orgReqVo.setAllUnderThisOrgCodeExcludeSelf(orgCode);
        }
        orgReqVo.setPrimaryFlag(YesNoEnum.yesNoEnum.ONE.getValue());
        orgReqVo.setUserNameOrFullName(reqVo.getUserNameOrFullName());
        orgReqVo.setIncludeEmptyUserPosition(false);
        orgReqVo.setNotInUserNameList(reqVo.getNotInUserNameList());
        return Result.ok(mdmPositionService.findPositionUserOrgPageList(orgReqVo));
    }

    @ApiOperation("批量修改与职位关联的组织编码")
    @PostMapping("/batchUpdatePositionOrg")
    public Result batchUpdatePositionOrg(@RequestBody MdmPositionUserOrgReqVo reqVo) {
        mdmPositionService.batchUpdatePositionOrg(reqVo);
        return Result.ok();
    }

    @ApiOperation("重置降维编码")
    @PostMapping("/resetPositionRuleCode")
    public Result resetPositionRuleCode() {
        mdmPositionService.resetPositionRuleCode();
        PositionUtil.reloadAllCache();
        return Result.ok();
    }

    @ApiOperation("删除全部缓存")
    @PostMapping("/deleteAllCache")
    public Result deleteAllCache() {
        PositionUtil.deleteAllCache();
        return Result.ok();
    }

    @ApiOperation("重新加载全部缓存")
    @PostMapping("/reloadAllCache")
    public Result reloadAllCache() {
        PositionUtil.reloadAllCache();
        return Result.ok();
    }

    @ApiOperation(value = "根据流程角色编码查询职位，按照组织从指定组织向上寻址", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "orgCode", value = "组织编码", required = true, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "bpmRoleCode", value = "流程角色编码", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/findPositionListByBpmRoleStartUpByOrgCode")
    public Result<List<MdmPositionUserOrgRespVo>> findPositionListByBpmRoleStartUpByOrgCode(@RequestParam("orgCode") String orgCode, @RequestParam("bpmRoleCode") String bpmRoleCode) {
        return Result.ok(mdmPositionService.findPositionListByBpmRoleStartUpByOrgCode(orgCode, bpmRoleCode));
    }

    @ApiOperation(value = "根据流程角色编码查询职位编码集合，按照组织及组织上级")
    @PostMapping("/findPositionCodeSetByOrgParentAndBpmRoleCodeList")
    public Result<Set<String>> findPositionCodeSetByOrgParentAndBpmRoleCodeList(@RequestParam(value = "orgCode", required = false) String orgCode, @RequestBody List<String> bpmRoleCodeList) {
        return Result.ok(mdmPositionService.findPositionCodeSetByOrgParentAndBpmRoleCodeList(orgCode, bpmRoleCodeList));
    }

    @ApiOperation(value = "根据流程角色编码查询职位编码集合，按照组织")
    @PostMapping("/findPositionCodeSetByOrgAndBpmRoleCodeList")
    public Result<Set<String>> findPositionCodeSetByOrgAndBpmRoleCodeList(@RequestParam(value = "orgCode", required = false) String orgCode, @RequestBody List<String> bpmRoleCodeList) {
        return Result.ok(mdmPositionService.findPositionCodeSetByOrgAndBpmRoleCodeList(orgCode, bpmRoleCodeList));
    }

    @ApiOperation(value = "职位与用户分页列表(SFA)")
    @PostMapping("/positionAndUserPage")
    public Result<PageResult<MdmPositionPageRespVo>> positionAndUserPage(@RequestBody MdmPositionCurrentUserReqVo reqVo) {
        PageResult<MdmPositionPageRespVo> result = mdmPositionService.positionAndUserPage(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "根据当前用户类型查询职位与用户分页列表(SFA)")
    @PostMapping("/findPositionAndUserByType")
    public Result<PageResult<MdmPositionPageRespVo>> findPositionAndUserByType(@RequestBody MdmPositionCurrentUserReqVo reqVo) {
        PageResult<MdmPositionPageRespVo> result = mdmPositionService.findPositionAndUserByType(reqVo);
        return Result.ok(result);
    }

}
