package com.biz.crm.position.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.bpmrole.service.MdmBpmRoleService;
import com.biz.crm.common.PageResult;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.bpmrole.req.MdmBpmRoleRelationPositionPageReqVo;
import com.biz.crm.nebular.mdm.bpmrole.req.MdmBpmRoleRelationPositionReqVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRelationPositionPageRespVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionBpmRoleRespVo;
import com.biz.crm.position.mapper.MdmPositionBpmRoleMapper;
import com.biz.crm.position.model.MdmPositionBpmRoleEntity;
import com.biz.crm.position.service.MdmPositionBpmRoleService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位与流程角色关联接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-26 20:07:10
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionBpmRoleServiceExpandImpl")
public class MdmPositionBpmRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPositionBpmRoleMapper, MdmPositionBpmRoleEntity> implements MdmPositionBpmRoleService {

    @Resource
    private MdmPositionBpmRoleMapper mdmPositionBpmRoleMapper;
    @Resource
    private MdmBpmRoleService mdmBpmRoleService;
    @Resource
    private RocketMQProducer rocketMQProducer;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addPositionBpmRole(String positionCode, String bpmRoleCode) {
        if (StringUtils.isNotEmpty(positionCode) && StringUtils.isNotEmpty(bpmRoleCode)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .eq(MdmPositionBpmRoleEntity::getPositionCode, positionCode)
                    .eq(MdmPositionBpmRoleEntity::getBpmRoleCode, bpmRoleCode)
                    .select(MdmPositionBpmRoleEntity::getId)
                    .list();
            if (CollectionUtil.listEmpty(list)) {
                MdmPositionBpmRoleEntity entity = new MdmPositionBpmRoleEntity();
                entity.setPositionCode(positionCode);
                entity.setBpmRoleCode(bpmRoleCode);
                this.save(entity);
                this.sendPositionRelBpmRoleAdd(Collections.singletonList(CrmBeanUtil.copy(entity, MdmPositionBpmRoleRespVo.class)));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addPositionBpmRole(String positionCode, List<String> bpmRoleCodeList) {
        if (StringUtils.isNotEmpty(positionCode) && CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList)) {
            bpmRoleCodeList.forEach(bpmRoleCode -> {
                this.addPositionBpmRole(positionCode, bpmRoleCode);
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addPositionBpmRole(List<String> positionCodeList, String bpmRoleCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList) && StringUtils.isNotEmpty(bpmRoleCode)) {
            positionCodeList.forEach(positionCode -> {
                this.addPositionBpmRole(positionCode, bpmRoleCode);
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPosition(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            this.removeByPositions(Collections.singletonList(positionCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPositions(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionBpmRoleEntity::getPositionCode, positionCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmPositionBpmRoleEntity::getId).collect(Collectors.toList()));
                this.sendPositionRelBpmRoleDelete(CrmBeanUtil.copyList(list, MdmPositionBpmRoleRespVo.class));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByBpmRole(String bpmRoleCode) {
        if (StringUtils.isNotEmpty(bpmRoleCode)) {
            this.removeByBpmRoles(Collections.singletonList(bpmRoleCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByBpmRoles(List<String> bpmRoleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionBpmRoleEntity::getBpmRoleCode, bpmRoleCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmPositionBpmRoleEntity::getId).collect(Collectors.toList()));
                this.sendPositionRelBpmRoleDelete(CrmBeanUtil.copyList(list, MdmPositionBpmRoleRespVo.class));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPositionBpmRole(String positionCode, String bpmRoleCode) {
        if (StringUtils.isNotEmpty(positionCode) && StringUtils.isNotEmpty(bpmRoleCode)) {
            this.removeByPositionBpmRole(positionCode, Collections.singletonList(bpmRoleCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPositionBpmRole(String positionCode, List<String> bpmRoleCodeList) {
        if (StringUtils.isNotEmpty(positionCode) && CollectionUtil.listNotEmptyNotSizeZero(bpmRoleCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .eq(MdmPositionBpmRoleEntity::getPositionCode, positionCode)
                    .in(MdmPositionBpmRoleEntity::getBpmRoleCode, bpmRoleCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmPositionBpmRoleEntity::getId).collect(Collectors.toList()));
                this.sendPositionRelBpmRoleDelete(CrmBeanUtil.copyList(list, MdmPositionBpmRoleRespVo.class));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void removeByPositionBpmRole(List<String> positionCodeList, String bpmRoleCode) {
        if (StringUtils.isNotEmpty(bpmRoleCode) && CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .eq(MdmPositionBpmRoleEntity::getBpmRoleCode, bpmRoleCode)
                    .in(MdmPositionBpmRoleEntity::getPositionCode, positionCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmPositionBpmRoleEntity::getId).collect(Collectors.toList()));
                this.sendPositionRelBpmRoleDelete(CrmBeanUtil.copyList(list, MdmPositionBpmRoleRespVo.class));
            }
        }
    }

    @Override
    public Set<String> findBpmRoleCodeSetByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            Set<String> bpmRoleCodeSet = new HashSet<>(16);
            List<List<String>> positionCodeListGroup = Lists.partition(positionCodeList, 500);
            for (List<String> group :
                    positionCodeListGroup) {
                bpmRoleCodeSet.addAll(this.lambdaQuery()
                        .in(MdmPositionBpmRoleEntity::getPositionCode, group)
                        .select(MdmPositionBpmRoleEntity::getBpmRoleCode)
                        .list()
                        .stream().map(MdmPositionBpmRoleEntity::getBpmRoleCode).collect(Collectors.toSet()));
            }
            return bpmRoleCodeSet;
        }
        return Collections.emptySet();
    }

    @Override
    public List<MdmBpmRoleRespVo> getPositionBpmRoleList(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            List<MdmPositionBpmRoleEntity> positionRoleEntityList = this.lambdaQuery()
                    .eq(MdmPositionBpmRoleEntity::getPositionCode, positionCode)
                    .select(MdmPositionBpmRoleEntity::getBpmRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionRoleEntityList)) {
                return mdmBpmRoleService.queryBatchByBpmRoleCodeList(positionRoleEntityList.stream().map(MdmPositionBpmRoleEntity::getBpmRoleCode).collect(Collectors.toList()));
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, List<String>> getPositionBpmRoleCodeGroupByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionBpmRoleEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionBpmRoleEntity::getPositionCode, MdmPositionBpmRoleEntity::getBpmRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.groupingBy(MdmPositionBpmRoleEntity::getPositionCode, Collectors.mapping(MdmPositionBpmRoleEntity::getBpmRoleCode, Collectors.toList())));
            }
        }
        return new HashMap<>(16);
    }

    @Override
    public Map<String, List<MdmBpmRoleRespVo>> getPositionBpmRoleGroupByPositionCodeList(List<String> positionCodeList) {
        Map<String, List<MdmBpmRoleRespVo>> map = new HashMap<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionBpmRoleEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionBpmRoleEntity::getPositionCode, MdmPositionBpmRoleEntity::getBpmRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                List<String> bpmRoleCodeList = list.stream().map(MdmPositionBpmRoleEntity::getBpmRoleCode).collect(Collectors.toList());
                final Map<String, MdmBpmRoleRespVo> bpmRoleMap = mdmBpmRoleService.queryBatchByBpmRoleCodeList(bpmRoleCodeList)
                        .stream().collect(Collectors.toMap(MdmBpmRoleRespVo::getBpmRoleCode, v -> v));
                Map<String, List<String>> groupMap = list.stream().collect(Collectors.groupingBy(MdmPositionBpmRoleEntity::getPositionCode, Collectors.mapping(MdmPositionBpmRoleEntity::getBpmRoleCode, Collectors.toList())));
                for (Map.Entry<String, List<String>> entry :
                        groupMap.entrySet()) {
                    List<MdmBpmRoleRespVo> roleList = new ArrayList<>();
                    for (String item :
                            entry.getValue()) {
                        if (bpmRoleMap.containsKey(item)) {
                            roleList.add(bpmRoleMap.get(item));
                        }
                    }
                    map.put(entry.getKey(), roleList);
                }
            }
        }
        return map;
    }

    @Override
    public PageResult<MdmBpmRoleRelationPositionPageRespVo> findPositionNotRelateAnyBpmRoleList(MdmBpmRoleRelationPositionPageReqVo reqVo) {
        Page<MdmBpmRoleRelationPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmBpmRoleRelationPositionPageRespVo> list = mdmPositionBpmRoleMapper.findPositionNotRelateAnyBpmRoleList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> positionCodeList = list.stream().map(MdmBpmRoleRelationPositionPageRespVo::getPositionCode).collect(Collectors.toList());
            Map<String, List<MdmBpmRoleRespVo>> map = this.getPositionBpmRoleGroupByPositionCodeList(positionCodeList);
            list.forEach(item -> {
                if (map.containsKey(item.getPositionCode())) {
                    List<MdmBpmRoleRespVo> bpmRoleList = map.get(item.getPositionCode());
                    item.setBpmRoleCode(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleCode).collect(Collectors.joining(",")));
                    item.setBpmRoleName(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleName).collect(Collectors.joining(",")));
                }
            });
        }
        return PageResult.<MdmBpmRoleRelationPositionPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmBpmRoleRelationPositionPageRespVo> findPositionNotRelateCurBpmRoleList(MdmBpmRoleRelationPositionPageReqVo reqVo) {
        Assert.hasText(reqVo.getBpmRoleCode(), "缺失流程角色编码");
        Page<MdmBpmRoleRelationPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmBpmRoleRelationPositionPageRespVo> list = mdmPositionBpmRoleMapper.findPositionNotRelateCurBpmRoleList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> positionCodeList = list.stream().map(MdmBpmRoleRelationPositionPageRespVo::getPositionCode).collect(Collectors.toList());
            Map<String, List<MdmBpmRoleRespVo>> map = this.getPositionBpmRoleGroupByPositionCodeList(positionCodeList);
            list.forEach(item -> {
                if (map.containsKey(item.getPositionCode())) {
                    List<MdmBpmRoleRespVo> bpmRoleList = map.get(item.getPositionCode());
                    item.setBpmRoleCode(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleCode).collect(Collectors.joining(",")));
                    item.setBpmRoleName(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleName).collect(Collectors.joining(",")));
                }
            });
        }
        return PageResult.<MdmBpmRoleRelationPositionPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmBpmRoleRelationPositionPageRespVo> findPositionHasRelateCurBpmRoleList(MdmBpmRoleRelationPositionPageReqVo reqVo) {
        Assert.hasText(reqVo.getBpmRoleCode(), "缺失流程角色编码");
        Page<MdmBpmRoleRelationPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmBpmRoleRelationPositionPageRespVo> list = mdmPositionBpmRoleMapper.findPositionHasRelateCurBpmRoleList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> positionCodeList = list.stream().map(MdmBpmRoleRelationPositionPageRespVo::getPositionCode).collect(Collectors.toList());
            Map<String, List<MdmBpmRoleRespVo>> map = this.getPositionBpmRoleGroupByPositionCodeList(positionCodeList);
            list.forEach(item -> {
                if (map.containsKey(item.getPositionCode())) {
                    List<MdmBpmRoleRespVo> bpmRoleList = map.get(item.getPositionCode());
                    item.setBpmRoleCode(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleCode).collect(Collectors.joining(",")));
                    item.setBpmRoleName(bpmRoleList.stream().map(MdmBpmRoleRespVo::getBpmRoleName).collect(Collectors.joining(",")));
                }
            });
        }
        return PageResult.<MdmBpmRoleRelationPositionPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void positionRelationBpmRole(MdmBpmRoleRelationPositionReqVo reqVo) {
        Assert.hasText(reqVo.getBpmRoleCode(), "缺失流程角色编码");
        Assert.notEmpty(reqVo.getPositionCodeList(), "缺失职位编码");
        MdmBpmRoleRespVo one = mdmBpmRoleService.detail(null, reqVo.getBpmRoleCode());
        Assert.notNull(one, "无效的流程角色编码");
        this.addPositionBpmRole(reqVo.getPositionCodeList(), reqVo.getBpmRoleCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceBindBpmRole(MdmBpmRoleRelationPositionReqVo reqVo) {
        Assert.notEmpty(reqVo.getIds(), "缺失关联id");
        Assert.hasText(reqVo.getBpmRoleCode(), "缺失流程角色编码");
        MdmBpmRoleRespVo one = mdmBpmRoleService.detail(null, reqVo.getBpmRoleCode());
        Assert.notNull(one, "无效的流程角色编码");
        List<MdmPositionBpmRoleEntity> list = this.lambdaQuery()
                .in(MdmPositionBpmRoleEntity::getId, reqVo.getIds())
                .select(MdmPositionBpmRoleEntity::getPositionCode)
                .list();
        Assert.notEmpty(list, "无效的关联id");
        this.removeByIds(reqVo.getIds());
        this.sendPositionRelBpmRoleDelete(CrmBeanUtil.copyList(list, MdmPositionBpmRoleRespVo.class));
        List<String> positionCodeList = list.stream().map(MdmPositionBpmRoleEntity::getPositionCode).collect(Collectors.toList());
        this.addPositionBpmRole(positionCodeList, one.getBpmRoleCode());
    }

    protected void sendPositionRelBpmRoleAdd(List<MdmPositionBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位关联流程角色发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_REL_BPM_ROLE_ADD_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位关联流程角色发送MQ消息成功：{}", list);
    }

    protected void sendPositionRelBpmRoleDelete(List<MdmPositionBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("解除职位关联流程角色发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_REL_BPM_ROLE_DELETE_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("解除职位关联流程角色发送MQ消息成功：{}", list);
    }
}
