package com.biz.crm.position.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.bpmrole.service.MdmBpmRoleService;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.constant.PositionOperationEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.permission.MdmDataPermissionPositionVo;
import com.biz.crm.nebular.mdm.position.req.*;
import com.biz.crm.nebular.mdm.position.resp.*;
import com.biz.crm.nebular.mdm.user.req.MdmUserReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.position.mapper.MdmPositionMapper;
import com.biz.crm.position.model.MdmPositionEntity;
import com.biz.crm.position.model.MdmPositionRoleEntity;
import com.biz.crm.position.model.MdmPositionUserEntity;
import com.biz.crm.position.service.MdmPositionBpmRoleService;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.position.service.MdmPositionUserService;
import com.biz.crm.positionlevel.service.MdmPositionLevelService;
import com.biz.crm.role.mode.MdmRoleEntity;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionServiceExpandImpl")
public class MdmPositionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPositionMapper, MdmPositionEntity> implements MdmPositionService {

    @Resource
    private MdmPositionMapper mdmPositionMapper;
    @Resource
    private MdmOrgService mdmOrgService;
    @Resource
    private MdmUserService mdmUserService;
    @Resource
    private MdmPositionLevelService mdmPositionLevelService;
    @Resource
    private MdmPositionUserService mdmPositionUserService;
    @Resource
    private MdmPositionRoleService mdmPositionRoleService;
    @Resource
    private MdmRoleService mdmRoleService;
    @Resource
    private MdmPositionBpmRoleService mdmPositionBpmRoleService;
    @Resource
    private MdmBpmRoleService mdmBpmRoleService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Resource
    private RocketMQProducer rocketMQProducer;

    @Override
    public PageResult<MdmPositionPageRespVo> findList(MdmPositionPageReqVo reqVo) {
        Page<MdmPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setOrgRuleCode(mdmOrgService.getOrgRuleCode(reqVo.getCode()));
        List<MdmPositionPageRespVo> list = mdmPositionMapper.findList(page, reqVo);
        this.setRoleCodeName(list);
        return PageResult.<MdmPositionPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmPositionRespVo detail(String id, String positionCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionRespVo resp = mdmPositionMapper.detail(id, positionCode);
            if (resp != null) {
                List<MdmPositionUserOrgRespVo> positionUserOrgList = this.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCode(resp.getPositionCode()));
                if (CollectionUtil.listNotEmptyNotSizeZero(positionUserOrgList)) {
                    resp.setUnionName(generateUnionName(positionUserOrgList.get(0).getOrgName(), positionUserOrgList.get(0).getPositionName(), positionUserOrgList.get(0).getFullName()));
                }
                resp.setRoleList(mdmPositionRoleService.getPositionRoleList(resp.getPositionCode()));
                resp.setBpmRoleList(mdmPositionBpmRoleService.getPositionBpmRoleList(resp.getPositionCode()));
                return resp;
            }
        }
        return null;
    }

    @Override
    public MdmPositionRespVo queryByPositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity one = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, positionCode)
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmPositionRespVo.class);
            }
        }
        return null;
    }

    @Override
    public List<MdmPositionRespVo> queryBatchByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionEntity> list = this.lambdaQuery()
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(MdmPositionEntity::getPositionCode, positionCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmPositionRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionRespVo> queryBatchByOrgCodeList(List<String> orgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
            List<MdmPositionEntity> list = this.lambdaQuery()
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(MdmPositionEntity::getOrgCode, orgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmPositionRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public MdmPositionDetailRespVo getPositionDetail(String id, String positionCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity one = this.lambdaQuery()
                    .eq(!StringUtils.isEmpty(id), MdmPositionEntity::getId, id)
                    .eq(!StringUtils.isEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .one();
            if (one != null) {
                MdmPositionDetailRespVo resp = CrmBeanUtil.copy(one, MdmPositionDetailRespVo.class);
                if (StringUtils.isNotEmpty(resp.getOrgCode())) {
                    resp.setOrg(mdmOrgService.detail(null, resp.getOrgCode()));
                }
                String userName = mdmPositionUserService.getUserNameByPositionCode(resp.getPositionCode());
                if (StringUtils.isNotEmpty(userName)) {
                    resp.setUser(mdmUserService.queryByUserName(userName));
                }
                resp.setRoleList(mdmPositionRoleService.getPositionRoleList(one.getPositionCode()));
                resp.setBpmRoleList(mdmPositionBpmRoleService.getPositionBpmRoleList(one.getPositionCode()));
                return resp;
            }
        }
        return null;
    }

    @Override
    public List<MdmPositionUserOrgRespVo> detailBatch(List<String> ids, List<String> positionCodeList) {
        List<MdmPositionUserOrgRespVo> list = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionEntity> positionList = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmPositionEntity::getId, ids)
                    .in(CollectionUtil.listNotEmptyNotSizeZero(positionCodeList), MdmPositionEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionEntity::getPositionCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionList)) {
                list.addAll(this.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCodeList(positionList.stream().map(MdmPositionEntity::getPositionCode).collect(Collectors.toList()))));
            }
        }
        return list;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String save(MdmPositionReqVo reqVo) {
        reqVo.setId(null);
        MdmPositionEntity entity = CrmBeanUtil.copy(reqVo, MdmPositionEntity.class);
        if (StringUtils.isEmpty(entity.getPositionName())) {
            Assert.hasText(reqVo.getPositionLevelCode(), "缺失职位级别编码");
            entity.setPositionName(mdmPositionLevelService.generatePositionNameByPositionLevelCode(reqVo.getPositionLevelCode()));
        }
        if (StringUtils.isEmpty(entity.getPositionCode())) {
            entity.setPositionCode(CodeUtil.generateCode(CodeRuleEnum.MDM_POSITION_CODE.getCode()));
        } else {
            List<MdmPositionEntity> list = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, reqVo.getPositionCode())
                    .select(MdmPositionEntity::getPositionCode)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "职位编码已经存在");
        }
        int levelNum = 1;
        if (StringUtils.isNotEmpty(reqVo.getParentCode())) {
            MdmPositionEntity parent = this.lambdaQuery().eq(MdmPositionEntity::getPositionCode, reqVo.getParentCode()).one();
            Assert.notNull(parent, "上级职位不存在");
            Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(parent.getEnableStatus()), "上级职位被禁用");
            levelNum = parent.getLevelNum() + 1;
        }
        entity.setRuleCode(generateRuleCodeByParentCode(entity.getParentCode()));
        entity.setLevelNum(levelNum);
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRoleCodeList())) {
            mdmPositionRoleService.addRole(reqVo.getRoleCodeList(), entity.getPositionCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getBpmRoleCodeList())) {
            mdmPositionBpmRoleService.addPositionBpmRole(entity.getPositionCode(), reqVo.getBpmRoleCodeList());
        }
        //发送MQ消息
        this.sendPositionAdd(Collections.singletonList(CrmBeanUtil.copy(entity, MdmPositionRespVo.class)));
        PositionUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            //记录日志
            crmLogSendUtil.sendForAdd(menuObject.toString(), entity.getId(), entity.getPositionCode(), CrmBeanUtil.copy(entity, MdmPositionReqVo.class));
        }
        return entity.getPositionCode();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmPositionReqVo reqVo) {
        String id = reqVo.getId();
        Assert.hasText(id, "主键不能为空");
        Assert.hasText(reqVo.getPositionCode(), "职位编码不能为空");
        Assert.hasText(reqVo.getPositionName(), "职位名称不能为空");
        MdmPositionEntity entity = this.getById(id);
        MdmPositionReqVo oldObject = CrmBeanUtil.copy(entity, MdmPositionReqVo.class);
        Assert.notNull(entity, "职位不存在");
        Assert.isTrue(entity.getPositionCode().equals(reqVo.getPositionCode()), "职位编码不能修改");
        String entityRuleCode = entity.getRuleCode();
        String parentCode = reqVo.getParentCode();
        //是否重置降维编码 取决于上级有无变化
        boolean reloadRuleCode = false;
        int levelNum = 1;
        if (StringUtils.isNotEmpty(parentCode)) {
            Assert.isTrue(!entity.getPositionCode().equals(parentCode), "上级职位不能为当前职位");
            if (!parentCode.equals(entity.getParentCode())) {
                //更改上级职位
                reloadRuleCode = true;
                MdmPositionEntity parent = this.lambdaQuery().eq(MdmPositionEntity::getPositionCode, parentCode).one();
                Assert.notNull(parent, "上级职位不存在");
                Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(parent.getEnableStatus()), "上级职位被禁用");
                String parentRuleCode = parent.getRuleCode();
                Assert.hasText(parentRuleCode, "上级职位ruleCode不能为空");
                Assert.isTrue(!parentRuleCode.startsWith(entityRuleCode), "上级职位不能是当前职位的下级职位");
                levelNum = parent.getLevelNum() + 1;
            }
        } else {
            if (!StringUtils.isEmpty(entity.getParentCode())) {
                reloadRuleCode = true;
                reqVo.setParentCode("");
            }
        }

        mdmPositionRoleService.removePositionCode(reqVo.getPositionCode());
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRoleCodeList())) {
            mdmPositionRoleService.addRole(reqVo.getRoleCodeList(), entity.getPositionCode());
        }
        mdmPositionBpmRoleService.removeByPosition(reqVo.getPositionCode());
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getBpmRoleCodeList())) {
            mdmPositionBpmRoleService.addPositionBpmRole(entity.getPositionCode(), reqVo.getBpmRoleCodeList());
        }

        //判断启用状态
        if (!entity.getEnableStatus().equals(reqVo.getEnableStatus())) {
            //如果是禁用，先禁用全部下级
            if (CrmEnableStatusEnum.DISABLE.getCode().equals(reqVo.getEnableStatus())) {
                this.disableBatch(Collections.singletonList(entity.getId()));
            }
        }
        MdmPositionEntity updateEntity = CrmBeanUtil.copy(reqVo, MdmPositionEntity.class);
        this.updateById(updateEntity);
        if (reloadRuleCode) {
            //更新当前及下级降维编码
            String ruleCode = generateRuleCodeByParentCode(reqVo.getParentCode());
            updateCurAndChildren(entity.getPositionCode(), ruleCode, levelNum);
        }
        PositionUtil.deleteAllCache();

        //发送MQ消息
        this.sendPositionUpdate(Collections.singletonList(CrmBeanUtil.copy(updateEntity, MdmPositionRespVo.class)));

        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            //记录日志
            crmLogSendUtil.sendForUpdate(menuObject.toString(), entity.getId(), entity.getPositionCode(), oldObject, CrmBeanUtil.copy(updateEntity, MdmPositionReqVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changePositionParent(String positionCode, String parentPositionCode) {
        MdmPositionEntity position = this.lambdaQuery()
                .eq(MdmPositionEntity::getPositionCode, positionCode)
                .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .one();
        Assert.notNull(position, "编码" + positionCode + "不存在");
        int levelNum = 1;
        if (StringUtils.isNotEmpty(parentPositionCode)) {
            MdmPositionEntity parent = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, parentPositionCode)
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .select(MdmPositionEntity::getId, MdmPositionEntity::getLevelNum)
                    .one();
            Assert.notNull(parent, "编码" + parentPositionCode + "无效");
            levelNum = parent.getLevelNum() + 1;
        }
        String ruleCode = generateRuleCodeByParentCode(parentPositionCode);
        position.setRuleCode(ruleCode);
        position.setLevelNum(levelNum);
        position.setParentCode(parentPositionCode);
        this.updateById(position);
        updateCurAndChildren(positionCode, ruleCode, levelNum);

        //发送MQ消息
        this.sendPositionUpdate(Collections.singletonList(CrmBeanUtil.copy(position, MdmPositionRespVo.class)));

        PositionUtil.deleteAllCache();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByPositionCode(MdmPositionReqVo reqVo) {
        Assert.hasText(reqVo.getPositionCode(), "职位编码不能为空");
        MdmPositionEntity entity = this.lambdaQuery().eq(MdmPositionEntity::getPositionCode, reqVo.getPositionCode()).one();
        Assert.notNull(entity, "职位不存在");

        String entityRuleCode = entity.getRuleCode();
        String parentCode = reqVo.getParentCode();
        //是否重置降维编码 取决于上级有无变化
        boolean reloadRuleCode = false;
        int levelNum = 1;
        if (StringUtils.isNotEmpty(parentCode)) {
            Assert.isTrue(!entity.getPositionCode().equals(parentCode), "上级职位不能为当前职位");
            if (!parentCode.equals(entity.getParentCode())) {
                //更改上级职位
                reloadRuleCode = true;
                MdmPositionEntity parent = this.lambdaQuery().eq(MdmPositionEntity::getPositionCode, parentCode).one();
                Assert.notNull(parent, "上级职位不存在");
                Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(parent.getEnableStatus()), "上级职位被禁用");
                String parentRuleCode = parent.getRuleCode();
                Assert.hasText(parentRuleCode, "上级职位ruleCode不能为空");
                Assert.isTrue(!parentRuleCode.startsWith(entityRuleCode), "上级职位不能是当前职位的下级职位");
                levelNum = parent.getLevelNum() + 1;
            }
        } else {
            if (!StringUtils.isEmpty(entity.getParentCode())) {
                reloadRuleCode = true;
                entity.setParentCode("");
            }
        }

//        entity.setPrimaryFlag(reqVo.getPrimaryFlag());
        entity.setOrgCode(reqVo.getOrgCode());
        entity.setParentCode(reqVo.getParentCode());

        this.updateById(entity);
        if (reloadRuleCode) {
            //更新当前及下级降维编码
            String ruleCode = generateRuleCodeByParentCode(reqVo.getParentCode());
            updateCurAndChildren(entity.getPositionCode(), ruleCode, levelNum);
        }

        //发送MQ消息
        this.sendPositionUpdate(Collections.singletonList(CrmBeanUtil.copy(entity, MdmPositionRespVo.class)));

        PositionUtil.deleteAllCache();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        List<MdmPositionEntity> mdmPositionEntities = mdmPositionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(mdmPositionEntities)) {
            Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            mdmPositionEntities.forEach(o -> {
                List<MdmPositionEntity> children = this.lambdaQuery()
                        .eq(MdmPositionEntity::getParentCode, o.getPositionCode())
                        .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                        .select(MdmPositionEntity::getPositionCode)
                        .list();
                Assert.isTrue(CollectionUtil.listEmpty(children), "职位[" + o.getPositionName() + "]存在下级，不能删除");
                String userNameByPrimaryPositionCode = mdmPositionUserService.getUserNameByPrimaryPositionCode(o.getPositionCode());
                if (StringUtils.isNotEmpty(userNameByPrimaryPositionCode)) {
                    String fullNameByUserName = mdmUserService.getFullNameByUserName(userNameByPrimaryPositionCode);
                    throw new BusinessException("职位[" + o.getPositionName() + "]是用户[" + fullNameByUserName + "]的主职位，不能删除");
                }
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
                o.setRuleCode(o.getId());
                if (menuObject != null) {
                    //记录日志
                    crmLogSendUtil.sendForDel(menuObject.toString(), o.getId(), o.getPositionCode(), o);
                }
            });
            List<String> positionCodeList = mdmPositionEntities.stream().map(MdmPositionEntity::getPositionCode).collect(Collectors.toList());
            mdmPositionUserService.removeByPositionCodes(positionCodeList);
            mdmPositionRoleService.removePositionCodes(positionCodeList);
            mdmPositionBpmRoleService.removeByPositions(positionCodeList);
            this.updateBatchById(mdmPositionEntities);

            //发送MQ消息
            this.sendPositionDelete(CrmBeanUtil.copyList(mdmPositionEntities, MdmPositionRespVo.class));

            PositionUtil.deleteAllCache();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        List<MdmPositionEntity> mdmPositionEntities = mdmPositionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(mdmPositionEntities)) {
            mdmPositionEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                //TODO 校验上级
            });
        }
        this.updateBatchById(mdmPositionEntities);
        //发送MQ消息
        this.sendPositionUpdate(CrmBeanUtil.copyList(mdmPositionEntities, MdmPositionRespVo.class));
        PositionUtil.deleteAllCache();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        List<MdmPositionEntity> mdmPositionEntities = mdmPositionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(mdmPositionEntities)) {
            mdmPositionEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                /*
                 * 2021-11-17 产品需求修改：禁用该职位时，判断该职位是否被用户关联
                 * 提出人：智亚楠
                 * 修改人：包宏彬
                 * */
                List<MdmPositionRespVo> positionRelations =
                        mdmPositionMapper.findPositionRelation(
                                new MdmPositionReqVo() {{
                                    setPositionCode(o.getPositionCode());
                                    setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                                }}
                        );
                if (CollectionUtils.isNotEmpty(positionRelations)) {
                    StringJoiner stringJoiner = new StringJoiner("、");
                    positionRelations.forEach(positionRelation -> stringJoiner.add(positionRelation.getUserName()));
                    throw new IllegalArgumentException("职位\"" + o.getPositionName()
                            + "\"已被用户\"" + stringJoiner.toString() + "\"使用，禁用前请先解除与用户的关联！");
                }
                //TODO 禁用全部下级
            });
        }
        this.updateBatchById(mdmPositionEntities);
        //发送MQ消息
        this.sendPositionUpdate(CrmBeanUtil.copyList(mdmPositionEntities, MdmPositionRespVo.class));
        PositionUtil.deleteAllCache();
    }

    @Override
    public List<MdmPositionRespVo> findAllParentPositionList(String id, String positionCode) {
        List<MdmPositionRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .one();
            if (position != null) {
                Set<String> ruleCodeSet = TreeRuleCodeUtil.splitParentRuleCodes(position.getRuleCode());
                List<MdmPositionEntity> parentList = this.lambdaQuery()
                        .in(MdmPositionEntity::getRuleCode, ruleCodeSet)
                        .list();
                list.addAll(CrmBeanUtil.copyList(parentList, MdmPositionRespVo.class));
            }
        }
        return list;
    }

    @Override
    public List<String> findAllParentPositionCodeList(String id, String positionCode) {
        List<MdmPositionRespVo> allParentPositionList = findAllParentPositionList(id, positionCode);
        if (CollectionUtils.isEmpty(allParentPositionList)) {
            return new ArrayList<>();
        }
        return allParentPositionList.stream()
                .filter(x -> !StringUtils.isEmpty(x.getPositionCode()))
                .map(MdmPositionRespVo::getPositionCode)
                .collect(Collectors.toList());
    }

    @Override
    public List<MdmPositionRespVo> findAllChildrenPositionList(String id, String positionCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                List<MdmPositionEntity> list = this.lambdaQuery()
                        .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                        .likeRight(MdmPositionEntity::getRuleCode, position.getRuleCode())
                        .list();
                return CrmBeanUtil.copyList(list, MdmPositionRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findAllChildrenPositionCodeList(String id, String positionCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                List<MdmPositionEntity> list = this.lambdaQuery()
                        .likeRight(MdmPositionEntity::getRuleCode, position.getRuleCode())
                        .select(MdmPositionEntity::getPositionCode)
                        .list();
                return list.stream().map(MdmPositionEntity::getPositionCode).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findAllChildrenPositionCodeListExcludeSelf(String id, String positionCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                List<MdmPositionEntity> list = this.lambdaQuery()
                        .likeRight(MdmPositionEntity::getRuleCode, position.getRuleCode())
                        .select(MdmPositionEntity::getId, MdmPositionEntity::getPositionCode)
                        .list();
                return list.stream().filter(x -> (StringUtils.isEmpty(id) || !id.equals(x.getId())) && (StringUtils.isEmpty(positionCode) || !positionCode.equals(x.getPositionCode()))).map(MdmPositionEntity::getPositionCode).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionRespVo> findAllChildrenPositionListByList(List<String> ids, List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionEntity> list = this.lambdaQuery()
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmPositionEntity::getId, ids)
                    .in(CollectionUtil.listNotEmptyNotSizeZero(positionCodeList), MdmPositionEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                List<String> likeRightRuleCodeList = list.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                return mdmPositionMapper.findPositionChildrenByRuleCodeList(new MdmPositionReqVo(), likeRightRuleCodeList);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionRespVo> findSingleChildrenPositionList(String id, String positionCode) {
        List<MdmPositionRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionCode)) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionCode), MdmPositionEntity::getPositionCode, positionCode)
                    .one();
            if (position != null) {
                List<MdmPositionEntity> children = this.lambdaQuery()
                        .eq(MdmPositionEntity::getParentCode, position.getPositionCode())
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(children)) {
                    list.addAll(CrmBeanUtil.copyList(children, MdmPositionRespVo.class));
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmPositionSelectRespVo> findPositionSelectList(MdmPositionSelectReqVo reqVo) {
        MdmPositionSelectTransportReqVo transportReqVo = new MdmPositionSelectTransportReqVo();
        Set<String> excludeOrgCodeList = new HashSet<>(16);
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisOrgCode())) {
            transportReqVo.setAllUnderOrgRuleCode(mdmOrgService.getOrgRuleCode(reqVo.getAllUnderThisOrgCode()));
        }
        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCode())) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, reqVo.getAllUnderThisPositionCode())
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                transportReqVo.setAllUnderPositionRuleCode(position.getRuleCode());
            }
        }
        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCodeExcludeSelf())) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, reqVo.getAllUnderThisPositionCodeExcludeSelf())
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                transportReqVo.setAllUnderPositionRuleCodeExcludeSelf(position.getRuleCode());
            }
        }
        if (StringUtils.isNotEmpty(reqVo.getNotUnderThisPositionCode())) {
            MdmPositionEntity position = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, reqVo.getNotUnderThisPositionCode())
                    .select(MdmPositionEntity::getRuleCode)
                    .one();
            if (position != null) {
                transportReqVo.setNotUnderPositionRuleCode(position.getRuleCode());
            }
        }
        if (StringUtils.isNotEmpty(reqVo.getAllAboveThisOrgCode())) {
            List<MdmOrgRespVo> allParentOrgList = mdmOrgService.findAllParentOrgList(null, reqVo.getAllAboveThisOrgCode());
            if (CollectionUtil.listEmpty(allParentOrgList)) {
                return new ArrayList<>();
            }
            transportReqVo.setOrgCodeList(allParentOrgList.stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList()));
        }
        if (StringUtils.isNotEmpty(reqVo.getNotUnderThisOrgCodeExcludeSelf())) {
            List<String> childrenOrgCodeList = mdmOrgService.findAllChildrenOrgCodeList(null, reqVo.getNotUnderThisOrgCodeExcludeSelf());
            if (CollectionUtil.listNotEmptyNotSizeZero(childrenOrgCodeList)) {
                excludeOrgCodeList.addAll(childrenOrgCodeList.stream().filter(item -> !item.equals(reqVo.getNotUnderThisOrgCodeExcludeSelf())).collect(Collectors.toSet()));
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getExcludeChildOrgOfThisPositionCode())) {
            if (StringUtils.isNotEmpty(reqVo.getExcludeChildOrgOfThisPositionCode())) {
                List<String> excludeChildrenOrgCodeList = mdmOrgService.findAllChildrenOrgCodeListOfPositionChildrenList(Collections.singletonList(reqVo.getExcludeChildOrgOfThisPositionCode()));
                if (CollectionUtil.listNotEmptyNotSizeZero(excludeChildrenOrgCodeList)) {
                    excludeOrgCodeList.addAll(excludeChildrenOrgCodeList);
                }
            }
        }

        transportReqVo.setExcludeOrgCodeList(new ArrayList<>(excludeOrgCodeList));

        List<MdmPositionSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmPositionMapper.findPositionSelectList(new Page<>(1L, pageSize, false), reqVo, transportReqVo));
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setUnionName(generateUnionName(item.getOrgName(), item.getPositionName(), item.getFullName()));
            });
        }
        return list;
    }

    @Override
    public MdmPositionRespVo getUserPrimaryPosition(MdmUserReqVo reqVo) {
        List<MdmPositionRespVo> positionRelation = mdmPositionMapper.findPositionRelation(new MdmPositionReqVo().setUserName(reqVo.getUserName()).setUserId(reqVo.getId()));
        if (CollectionUtil.listNotEmptyNotSizeZero(positionRelation)) {
            for (MdmPositionRespVo positionItem :
                    positionRelation) {
                if (YesNoEnum.yesNoEnum.ONE.getValue().equals(positionItem.getPrimaryFlag())) {
                    return positionItem;
                }
            }
        }
        return null;
    }

    @Override
    public MdmPositionUserOrgRespVo getPrimaryPositionByUserName(String userName) {
        String primaryPositionCodeByUserName = mdmPositionUserService.getPrimaryPositionCodeByUserName(userName);
        if (StringUtils.isNotEmpty(primaryPositionCodeByUserName)) {
            List<MdmPositionUserOrgRespVo> positionUserOrgList = this.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCode(primaryPositionCodeByUserName));
            if (CollectionUtil.listNotEmptyNotSizeZero(positionUserOrgList)) {
                return positionUserOrgList.get(0);
            }
        }
        return null;
    }

    @Override
    public MdmPositionRespVo getUserCurrentPosition(MdmUserReqVo reqVo) {
        MdmUserRespVo userRespVo = mdmUserService.detail(null, reqVo.getUserName());
        if (userRespVo != null) {
            if (UserTypeEnum.USER.getCode().equals(userRespVo.getUserType()) && CollectionUtil.listNotEmptyNotSizeZero(userRespVo.getPositionList())) {
                MdmPositionRespVo primaryPosition = null;
                for (MdmPositionRespVo positionItem :
                        userRespVo.getPositionList()) {
                    if (YesNoEnum.yesNoEnum.ONE.getValue().equals(positionItem.getPrimaryFlag())) {
                        primaryPosition = positionItem;
                    }
                    if (YesNoEnum.yesNoEnum.ONE.getValue().equals(positionItem.getCurrentFlag())) {
                        return positionItem;
                    }
                }

                if (primaryPosition != null) {
                    //如果没有当前职位，设置主职位为当前职位
                    changeUserCurrentPosition(userRespVo.getUserName(), userRespVo.getPositionCode());
                    return primaryPosition;
                }
            }
        }
        return null;
    }

    @Override
    public MdmPositionRespVo getUserCurrentPositionForLogin(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            MdmUserRespVo mdmUserRespVo = mdmUserService.queryByUserName(userName);
            Assert.notNull(mdmUserRespVo, "未找到用户");
            String currentPositionCode = getCurrentPositionCodeByUserName(userName);
            Assert.hasText(currentPositionCode, "用户职位异常，请联系管理员");
            MdmPositionEntity one = this.lambdaQuery()
                    .eq(MdmPositionEntity::getPositionCode, currentPositionCode)
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmPositionRespVo.class);
            }
        }
        return null;
    }

    @Override
    public MdmPositionUserOrgRespVo getCurrentPositionByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            MdmUserRespVo mdmUserRespVo = mdmUserService.queryByUserName(userName);
            Assert.notNull(mdmUserRespVo, "未找到用户");
            String currentPositionCode = getCurrentPositionCodeByUserName(userName);
            Assert.hasText(currentPositionCode, "用户职位异常，请联系管理员");
            return this.getPositionUserOrgByPositionCode(currentPositionCode);
        }
        return null;
    }

    @Override
    public List<MdmPositionRespVo> getUserAllPositionList(MdmUserReqVo reqVo) {
        return this.getUserAllPositionList(reqVo.getUserName());
    }

    @Override
    public List<MdmPositionRespVo> getUserAllPositionList(String userName) {
        List<MdmPositionRespVo> list = mdmPositionMapper.findPositionRelation(new MdmPositionReqVo().setUserName(userName));
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setRoleList(mdmPositionRoleService.getPositionRoleList(item.getPositionCode()));
                item.setOperationType(PositionOperationEnum.HISTORY.getCode());
                item.setUnionName(generateUnionName(item.getOrgName(), item.getPositionName(), item.getFullName()));
                item.setBpmRoleList(mdmPositionBpmRoleService.getPositionBpmRoleList(item.getPositionCode()));
            });
        }
        return list;
    }

    @Override
    public List<MdmPositionRespVo> getUserPositionByUserCode(String userCode) {
        if (StringUtils.isNotEmpty(userCode)) {
            MdmUserRespVo mdmUserRespVo = mdmUserService.queryByUserCode(userCode);
            if (mdmUserRespVo != null) {
                return getUserAllPositionList(mdmUserRespVo.getUserName());
            }
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserPrimaryPosition(MdmPositionReqVo reqVo) {
        changeUserCurrentPosition(reqVo.getUserName(), reqVo.getPositionCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserPrimaryPosition(String userName, String positionCode) {
        Assert.hasText(userName, "缺失用户登录名");
        Assert.hasText(positionCode, "缺失职位编码");
        List<String> positionCodeList = mdmPositionUserService.getPositionCodeListByUserName(userName);
        Assert.isTrue(CollectionUtil.listNotEmptyNotSizeZero(positionCodeList) && positionCodeList.contains(positionCode), "用户[" + userName + "]未关联职位[" + positionCode + "]");
        mdmPositionUserService.changeUserPrimaryPosition(userName, positionCode);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserCurrentPosition(MdmPositionReqVo reqVo) {
        changeUserCurrentPosition(reqVo.getUserName(), reqVo.getPositionCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserCurrentPosition(String userName, String positionCode) {
        Assert.hasText(userName, "缺失用户登录名");
        Assert.hasText(positionCode, "缺失职位编码");
        List<String> positionCodeList = mdmPositionUserService.getPositionCodeListByUserName(userName);
        Assert.isTrue(CollectionUtil.listNotEmptyNotSizeZero(positionCodeList) && positionCodeList.contains(positionCode), "用户[" + userName + "]未关联职位[" + positionCode + "]");
        mdmPositionUserService.changeUserCurrentPosition(userName, positionCode);
        //TODO 切换当前职位需更新redis缓存
        UserRedis user = UserUtils.getUser();
        String token = UserUtils.getToken();
        if (user != null && user.getUsername().equals(userName) && StringUtils.isNotEmpty(token)) {
            List<MdmPositionUserOrgRespVo> positionUserOrgList = this.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCode(positionCode));
            if (CollectionUtil.listNotEmptyNotSizeZero(positionUserOrgList)) {
                MdmPositionUserOrgRespVo positionUserOrg = positionUserOrgList.get(0);
                user.setPoscode(positionUserOrg.getPositionCode());
                user.setPosname(positionUserOrg.getPositionName());
                user.setOrgcode(positionUserOrg.getOrgCode());
                user.setOrgname(positionUserOrg.getOrgName());
                UserUtils.setUser(token, user);
            }
        }
    }

    @Override
    public MdmPositionUserOrgRespVo getPositionUserOrgByPositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            List<MdmPositionUserOrgRespVo> list = this.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCode(positionCode));
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.get(0);
            }
        }
        return null;
    }

    @Override
    public List<MdmPositionUserOrgRespVo> findPositionUserOrgList(MdmPositionUserOrgReqVo reqVo) {

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisOrgCode())) {
            List<MdmOrgRespVo> childrenOrgList = mdmOrgService.findAllChildrenOrgList(null, reqVo.getAllUnderThisOrgCode());
            if (CollectionUtil.listEmpty(childrenOrgList)) {
                return new ArrayList<>();
            }
            List<String> childrenOrgCodeList = childrenOrgList.stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getOrgCodeList())) {
                reqVo.setOrgCodeList(childrenOrgCodeList);
            } else {
                reqVo.getOrgCodeList().retainAll(childrenOrgCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisOrgCodeExcludeSelf())) {
            List<MdmOrgRespVo> childrenOrgList = mdmOrgService.findAllChildrenOrgList(null, reqVo.getAllUnderThisOrgCodeExcludeSelf());
            if (CollectionUtil.listEmpty(childrenOrgList) || childrenOrgList.size() == 1) {
                return new ArrayList<>();
            }
            List<String> childrenOrgCodeList = childrenOrgList.stream().filter(item -> !item.getOrgCode().equals(reqVo.getAllUnderThisOrgCodeExcludeSelf())).map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getOrgCodeList())) {
                reqVo.setOrgCodeList(childrenOrgCodeList);
            } else {
                reqVo.getOrgCodeList().retainAll(childrenOrgCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCode())) {
            List<MdmPositionRespVo> childrenPositionList = this.findAllChildrenPositionList(null, reqVo.getAllUnderThisPositionCode());
            if (CollectionUtil.listEmpty(childrenPositionList)) {
                return new ArrayList<>();
            }
            List<String> childrenPositionCodeList = childrenPositionList.stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getPositionCodeList())) {
                reqVo.setPositionCodeList(childrenPositionCodeList);
            } else {
                reqVo.getPositionCodeList().retainAll(childrenPositionCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCodeExcludeSelf())) {
            List<MdmPositionRespVo> childrenPositionList = this.findAllChildrenPositionList(null, reqVo.getAllUnderThisPositionCodeExcludeSelf());
            if (CollectionUtil.listEmpty(childrenPositionList) || childrenPositionList.size() == 1) {
                return new ArrayList<>();
            }
            List<String> childrenPositionCodeList = childrenPositionList.stream().filter(item -> !item.getPositionCode().equals(reqVo.getAllUnderThisPositionCodeExcludeSelf())).map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getPositionCodeList())) {
                reqVo.setPositionCodeList(childrenPositionCodeList);
            } else {
                reqVo.getPositionCodeList().retainAll(childrenPositionCodeList);
            }
        }

        return mdmPositionMapper.findPositionUserOrgList(null, reqVo);
    }

    @Override
    public PageResult<MdmPositionUserOrgPageRespVo> findPositionUserOrgPageList(MdmPositionUserOrgPageReqVo reqVo) {

        Page<MdmPositionUserOrgPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisOrgCode())) {
            List<MdmOrgRespVo> childrenOrgList = mdmOrgService.findAllChildrenOrgList(null, reqVo.getAllUnderThisOrgCode());
            if (CollectionUtil.listEmpty(childrenOrgList)) {
                return PageResult.<MdmPositionUserOrgPageRespVo>builder()
                        .data(new ArrayList<>())
                        .count(0L)
                        .build();
            }
            List<String> childrenOrgCodeList = childrenOrgList.stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getOrgCodeList())) {
                reqVo.setOrgCodeList(childrenOrgCodeList);
            } else {
                reqVo.getOrgCodeList().retainAll(childrenOrgCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisOrgCodeExcludeSelf())) {
            List<MdmOrgRespVo> childrenOrgList = mdmOrgService.findAllChildrenOrgList(null, reqVo.getAllUnderThisOrgCodeExcludeSelf());
            if (CollectionUtil.listEmpty(childrenOrgList) || childrenOrgList.size() == 1) {
                return PageResult.<MdmPositionUserOrgPageRespVo>builder()
                        .data(new ArrayList<>())
                        .count(0L)
                        .build();
            }
            List<String> childrenOrgCodeList = childrenOrgList.stream().filter(item -> !item.getOrgCode().equals(reqVo.getAllUnderThisOrgCodeExcludeSelf())).map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getOrgCodeList())) {
                reqVo.setOrgCodeList(childrenOrgCodeList);
            } else {
                reqVo.getOrgCodeList().retainAll(childrenOrgCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCode())) {
            List<MdmPositionRespVo> childrenPositionList = this.findAllChildrenPositionList(null, reqVo.getAllUnderThisPositionCode());
            if (CollectionUtil.listEmpty(childrenPositionList)) {
                return PageResult.<MdmPositionUserOrgPageRespVo>builder()
                        .data(new ArrayList<>())
                        .count(0L)
                        .build();
            }
            List<String> childrenPositionCodeList = childrenPositionList.stream().map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getPositionCodeList())) {
                reqVo.setPositionCodeList(childrenPositionCodeList);
            } else {
                reqVo.getPositionCodeList().retainAll(childrenPositionCodeList);
            }
        }

        if (StringUtils.isNotEmpty(reqVo.getAllUnderThisPositionCodeExcludeSelf())) {
            List<MdmPositionRespVo> childrenPositionList = this.findAllChildrenPositionList(null, reqVo.getAllUnderThisPositionCodeExcludeSelf());
            if (CollectionUtil.listEmpty(childrenPositionList) || childrenPositionList.size() == 1) {
                return PageResult.<MdmPositionUserOrgPageRespVo>builder()
                        .data(new ArrayList<>())
                        .count(0L)
                        .build();
            }
            List<String> childrenPositionCodeList = childrenPositionList.stream().filter(item -> !item.getPositionCode().equals(reqVo.getAllUnderThisPositionCodeExcludeSelf())).map(MdmPositionRespVo::getPositionCode).collect(Collectors.toList());
            if (CollectionUtil.listEmpty(reqVo.getPositionCodeList())) {
                reqVo.setPositionCodeList(childrenPositionCodeList);
            } else {
                reqVo.getPositionCodeList().retainAll(childrenPositionCodeList);
            }
        }

        List<MdmPositionUserOrgPageRespVo> list = mdmPositionMapper.findPositionUserOrgPageList(page, reqVo);
        return PageResult.<MdmPositionUserOrgPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchUpdatePositionOrg(MdmPositionUserOrgReqVo reqVo) {
        Assert.notEmpty(reqVo.getPositionCodeList(), "缺失职位编码");
        Assert.notNull(reqVo.getOrgCode(), "缺失组织编码");
        MdmOrgRespVo org = mdmOrgService.detail(null, reqVo.getOrgCode());
        Assert.notNull(org, "未找到组织");

        List<MdmPositionEntity> positionList = this.lambdaQuery()
                .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmPositionEntity::getPositionCode, reqVo.getPositionCodeList())
                .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getPositionName, MdmPositionEntity::getOrgCode)
                .list();
        Assert.notEmpty(positionList, "无效的职位编码");
        final Map<String, MdmPositionEntity> positionMap = positionList.stream().collect(Collectors.toMap(MdmPositionEntity::getPositionCode, v -> v));
        reqVo.getPositionCodeList().forEach(positionCode -> {
            Assert.isTrue(positionMap.containsKey(positionCode), "职位编码[" + positionCode + "]无效");
            MdmPositionEntity position = positionMap.get(positionCode);

            //TODO 逐条校验，效率不高，后续优化

            //校验下级
            List<String> excludeOrgCodeList = mdmOrgService.findAllChildrenOrgCodeListOfPositionChildrenList(Collections.singletonList(positionCode));
            if (CollectionUtil.listNotEmptyNotSizeZero(excludeOrgCodeList)) {
                if (excludeOrgCodeList.contains(org.getOrgCode())) {
                    throw new BusinessException("组织[" + org.getOrgName() + "]是职位[" + position.getPositionName() + "]的下级职位的组织的下级组织，不能作为当前职位的组织");
                }
            }

            //校验上级
            List<String> excludeParentOrgCodeList = mdmOrgService.findAllParentOrgCodeListOfPositionParentList(Collections.singletonList(positionCode));
            if (CollectionUtil.listNotEmptyNotSizeZero(excludeParentOrgCodeList)) {
                if (excludeParentOrgCodeList.contains(org.getOrgCode())) {
                    throw new BusinessException("组织[" + org.getOrgName() + "]是职位[" + position.getPositionName() + "]的上级职位的组织的上级组织，不能作为当前职位的组织");
                }
            }
        });
        this.lambdaUpdate()
                .in(MdmPositionEntity::getPositionCode, reqVo.getPositionCodeList())
                .set(MdmPositionEntity::getOrgCode, org.getOrgCode())
                .update();

        //发送MQ消息
        this.sendPositionUpdate(this.queryBatchByPositionCodeList(reqVo.getPositionCodeList()));

        PositionUtil.deleteAllCache();
    }

    @Override
    @Klock
    @Transactional(rollbackFor = Exception.class)
    public void resetPositionRuleCode() {
        long l = System.currentTimeMillis();
        log.info("-----------重置职位降维编码_" + l + "_开始-----------");
        mdmPositionMapper.copyIdToRuleCode();
        log.info("-----------重置职位降维编码_" + l + "_编码重置成id成功-----------");
        mdmPositionMapper.setNullNotExistParentCode();
        log.info("-----------重置职位降维编码_" + l + "_清除无效上级编码成功-----------");
        QueryWrapper<MdmPositionEntity> wrapper = new QueryWrapper<>();
        wrapper.and(x -> x.eq("parent_code", "").or().isNull("parent_code"));
        wrapper.eq("del_flag", CrmDelFlagEnum.NORMAL.getCode());
        List<MdmPositionEntity> list = mdmPositionMapper.selectList(wrapper);

        this.lambdaQuery()
                .eq(MdmPositionEntity::getPositionCode, "")
                .or()
                .eq(MdmPositionEntity::getPositionName, "")
                .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getPositionName)
                .list();


        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            for (int i = 0; i < list.size(); i++) {
                updateCurAndChildren(list.get(i).getPositionCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
            }
        }
        log.info("-----------重置职位降维编码_" + l + "_编码重新生成成功-----------");
        PositionUtil.deleteAllCache();
        log.info("-----------重置职位降维编码_" + l + "_缓存清除成功-----------");
        log.info("-----------重置职位降维编码_" + l + "_结束-----------");
    }

    @Override
    public List<MdmPositionUserOrgRespVo> findPositionListByBpmRoleStartUpByOrgCode(String orgCode, String bpmRoleCode) {
        Assert.hasText(orgCode, "缺失组织编码");
        Assert.hasText(bpmRoleCode, "缺失流程角色编码");
        List<MdmOrgRespVo> parentOrgList = mdmOrgService.findAllParentOrgList(null, orgCode);
        List<MdmOrgRespVo> collectOrderByLevelNumDesc = parentOrgList.stream().sorted((x1, x2) -> new BigDecimal(x2.getLevelNum()).compareTo(new BigDecimal(x1.getLevelNum()))).collect(Collectors.toList());
        List<String> collect = collectOrderByLevelNumDesc.stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
        List<MdmPositionUserOrgRespVo> list = mdmPositionMapper.findPositionListByBpmRole(CrmDelFlagEnum.NORMAL.getCode(), CrmEnableStatusEnum.ENABLE.getCode(), bpmRoleCode, collect);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Map<String, List<MdmPositionUserOrgRespVo>> groupByOrgMap = list.stream().collect(Collectors.groupingBy(MdmPositionUserOrgRespVo::getOrgCode));
            for (MdmOrgRespVo parentOrg
                    : collectOrderByLevelNumDesc) {
                if (groupByOrgMap.containsKey(parentOrg.getOrgCode())) {
                    return groupByOrgMap.get(parentOrg.getOrgCode());
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Set<String> findPositionCodeSetByOrgParentAndBpmRoleCodeList(String orgCode, List<String> bpmRoleCodeList) {
        Assert.notEmpty(bpmRoleCodeList, "流程角色编码不能为空");
        List<String> parentOrgCodeList = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgCode)) {
            parentOrgCodeList.addAll(OrgUtil.getParentOrgCodeListIncludeSelfEnable(orgCode));
            if (parentOrgCodeList.size() == 0) {
                return Collections.emptySet();
            }
        }
        return new HashSet<>(mdmPositionMapper.findPositionCodeListByOrgListAndBpmRoleList(parentOrgCodeList, null, bpmRoleCodeList));
    }

    @Override
    public Set<String> findPositionCodeSetByOrgAndBpmRoleCodeList(String orgCode, List<String> bpmRoleCodeList) {
        Assert.notEmpty(bpmRoleCodeList, "流程角色编码不能为空");
        List<String> parentOrgCodeList = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgCode)) {
            parentOrgCodeList = Collections.singletonList(orgCode);
        }
        return new HashSet<>(mdmPositionMapper.findPositionCodeListByOrgListAndBpmRoleList(parentOrgCodeList, null, bpmRoleCodeList));
    }

    @Override
    public List<MdmDataPermissionPositionVo> findPositionAndOrgList(List<String> positionCodeList) {
        if (CollectionUtils.isEmpty(positionCodeList)) {
            return new ArrayList<>();
        }
        return mdmPositionMapper.findPositionAndOrgList(positionCodeList);
    }

    @Override
    public List<MdmPositionRedisVo> findRedisListByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            return mdmPositionMapper.findRedisListByPositionCodeList(positionCodeList);
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionParentChildrenRedisVo> findRedisPositionParentListByCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionEntity> list = this.lambdaQuery()
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(MdmPositionEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(list.stream().map(MdmPositionEntity::getRuleCode).collect(Collectors.toList()));
                List<MdmPositionEntity> parentList = this.lambdaQuery()
                        .in(MdmPositionEntity::getRuleCode, parentRuleCodes)
                        .list();
                Set<String> collect = parentList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmPositionEntity::getParentCode).collect(Collectors.toSet());
                Map<String, String> parentPositionCodeNameMap = new HashMap<>(16);
                if (!collect.isEmpty()) {
                    parentPositionCodeNameMap.putAll(this.lambdaQuery()
                            .in(MdmPositionEntity::getPositionCode, collect)
                            .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getPositionName)
                            .list()
                            .stream()
                            .collect(Collectors.toMap(MdmPositionEntity::getPositionCode, MdmPositionEntity::getPositionName)));
                }
                List<MdmPositionRedisVo> parentRedisVoList = parentList.stream().map(item -> {
                    MdmPositionRedisVo copy = CrmBeanUtil.copy(item, MdmPositionRedisVo.class);
                    if (StringUtils.isNotEmpty(copy.getParentCode())) {
                        if (parentPositionCodeNameMap.containsKey(copy.getParentCode())) {
//                            copy.setParentName(parentPositionCodeNameMap.get(copy.getParentCode()));
                        }
                    }
                    return copy;
                }).collect(Collectors.toList());
                List<MdmPositionParentChildrenRedisVo> result = new ArrayList<>();
                for (MdmPositionEntity item :
                        parentList) {
                    MdmPositionParentChildrenRedisVo redisVo = new MdmPositionParentChildrenRedisVo();
                    redisVo.setPositionCode(item.getPositionCode());
                    redisVo.setList(parentRedisVoList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                    result.add(redisVo);
                }
                return result;
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionParentChildrenRedisVo> findRedisPositionChildrenListByCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionEntity> positionList = this.lambdaQuery()
                    .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(MdmPositionEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionList)) {
                List<String> likeRightRuleCodeList = positionList.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                List<MdmPositionRespVo> childrenList = mdmPositionMapper.findPositionChildrenByRuleCodeList(new MdmPositionReqVo(), likeRightRuleCodeList);
                if (CollectionUtil.listNotEmptyNotSizeZero(childrenList)) {
                    List<MdmPositionRedisVo> redisList = CrmBeanUtil.copyList(childrenList, MdmPositionRedisVo.class);
                    List<MdmPositionParentChildrenRedisVo> resultList = new ArrayList<>();
                    for (MdmPositionEntity item :
                            positionList) {
                        MdmPositionParentChildrenRedisVo redisVo = new MdmPositionParentChildrenRedisVo();
                        redisVo.setPositionCode(item.getPositionCode());
                        redisVo.setList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                        resultList.add(redisVo);
                    }
                    return resultList;
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmPositionReloadRedisVo> findRedisReloadList() {
        final List<MdmPositionEntity> all = this.lambdaQuery()
                .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(all)) {
            final Map<String, String> positionCodeNameMap = all.stream().collect(Collectors.toMap(MdmPositionEntity::getPositionCode, MdmPositionEntity::getPositionName));
            final List<MdmPositionRedisVo> redisList = all.stream().map(item -> {
                MdmPositionRedisVo copy = CrmBeanUtil.copy(item, MdmPositionRedisVo.class);
                if (StringUtils.isNotEmpty(copy.getParentCode())) {
//                    copy.setParentName(positionCodeNameMap.get(copy.getParentCode()));
                }
                return copy;
            }).collect(Collectors.toList());
            return redisList.stream().map(item -> {
                MdmPositionReloadRedisVo redisVo = new MdmPositionReloadRedisVo();
                redisVo.setPositionCode(item.getPositionCode());
                redisVo.setSelf(item);
                redisVo.setParentList(redisList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                redisVo.setChildrenList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                return redisVo;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public PageResult<MdmPositionPageRespVo> positionAndUserPage(MdmPositionCurrentUserReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        if (user == null) {
            return PageResult.empty();
        }
        Page<MdmPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (UserTypeEnum.CUSTOMER.getCode().equals(user.getUsertype()) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(user.getUsertype())) {
            reqVo.setUnderCustomerUserName(user.getUsername());
        }
        List<MdmPositionPageRespVo> mdmPositionPageRespVos = mdmPositionMapper.positionAndUserPage(page, reqVo);
        return PageResult.<MdmPositionPageRespVo>builder()
                .data(mdmPositionPageRespVos)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmPositionPageRespVo> findPositionAndUserByType(MdmPositionCurrentUserReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        if (user == null) {
            return PageResult.empty();
        }
        String usertype = user.getUsertype();
        if (UserTypeEnum.USER.getCode().equals(usertype)) {
            return positionAndUserPage(reqVo);
        } else if (UserTypeEnum.CUSTOMER.getCode().equals(usertype) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(usertype)) {
            Page<MdmPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
            List<MdmPositionPageRespVo> mdmPositionPageRespVos = mdmUserService.findCustomerEmployeePage(reqVo);
            if (!CollectionUtils.isEmpty(mdmPositionPageRespVos)) {
                //也就sfa会把用户当职位用了
                for (MdmPositionPageRespVo mdmPositionPageRespVo : mdmPositionPageRespVos) {
                    mdmPositionPageRespVo.setPositionCode(mdmPositionPageRespVo.getUserName());
                    mdmPositionPageRespVo.setPositionName(mdmPositionPageRespVo.getFullName());
                }
            }
            return PageResult.<MdmPositionPageRespVo>builder()
                    .data(mdmPositionPageRespVos)
                    .count(page.getTotal())
                    .build();
        }
        return PageResult.empty();
    }

    /**
     * 生成 组织名称/职位名称/用户名称
     *
     * @param orgName      组织名称
     * @param positionName 职位名称
     * @param fullName     用户名称
     * @return
     */
    protected String generateUnionName(String orgName, String positionName, String fullName) {
        return (orgName == null ? "" : orgName) + "/" + (positionName == null ? "" : positionName) + "/" + (fullName == null ? "" : fullName);
    }

    /**
     * 根据parentCode生成一个降维编码
     *
     * @param parentCode
     * @return
     */
    protected String generateRuleCodeByParentCode(String parentCode) {
        String ruleCode = "";
        boolean top = true;
        String prefix = "";
        if (StringUtils.isNotEmpty(parentCode)) {
            MdmPositionEntity parent = this.lambdaQuery().eq(MdmPositionEntity::getPositionCode, parentCode).one();
            if (parent != null) {
                top = false;
                prefix = parent.getRuleCode();
            }
        }
        List<MdmPositionEntity> list = null;
        QueryWrapper<MdmPositionEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("del_flag", CrmDelFlagEnum.NORMAL.getCode());
        if (top) {
            wrapper.and(x -> x.eq("parent_code", "").or().isNull("parent_code"));
        } else {
            wrapper.eq("parent_code", parentCode);
        }
        list = mdmPositionMapper.selectList(wrapper);
        if (CollectionUtil.listEmpty(list)) {
            ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(1);
        } else {
            Set<Integer> set = new LinkedHashSet<>();
            for (MdmPositionEntity item :
                    list) {
                if (StringUtils.isNotEmpty(item.getRuleCode())) {
                    try {
                        Integer integer = TreeRuleCodeUtil.getCurLevelCodeValue(item.getRuleCode());
                        set.add(integer);
                    } catch (BusinessException e) {
                        throw new BusinessException("职位" + e.getMsg());
                    }
                }
            }
            int maxRuleCodeValueInLevel = TreeRuleCodeUtil.getMaxRuleCodeValueInLevel();
            for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
                if (i == maxRuleCodeValueInLevel) {
                    throw new BusinessException("降维编码越界，请联系管理员处理");
                }
                if (!set.contains(i)) {
                    ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(i);
                    break;
                }
            }
        }
        return ruleCode;
    }

    /**
     * 更新职位及职位下级降维编码和层级
     *
     * @param positionCode 当前职位编码
     * @param curCode      当前职位降维编码
     * @param levelNum     当前层级
     */
    protected void updateCurAndChildren(String positionCode, String curCode, int levelNum) {

        //更新当前
        this.lambdaUpdate()
                .eq(MdmPositionEntity::getPositionCode, positionCode)
                .set(MdmPositionEntity::getRuleCode, curCode)
                .set(MdmPositionEntity::getLevelNum, levelNum)
                .update();

        //查询下一层
        List<MdmPositionEntity> list = this.lambdaQuery()
                .eq(MdmPositionEntity::getParentCode, positionCode)
                .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .select(MdmPositionEntity::getPositionCode)
                .list();

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                //递归调用
                updateCurAndChildren(list.get(i).getPositionCode(), curCode + TreeRuleCodeUtil.numToSingleCode(i + 1), (levelNum + 1));
            }
        }
    }

    /**
     * 根据登录名获取当前职位
     *
     * @param userName 用户登录名
     * @return
     */
    protected String getCurrentPositionCodeByUserName(String userName) {
        String thisPositionCode = "";
        if (StringUtils.isNotEmpty(userName)) {
            //TODO 修改到 mdmPositionUserService
            List<MdmPositionUserEntity> list = mdmPositionUserService.lambdaQuery()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                boolean resetCurrent = true;
                final Set<String> positionCodeList = list.stream().map(MdmPositionUserEntity::getPositionCode).collect(Collectors.toSet());
                final Map<String, MdmPositionEntity> positionCodeMap = this.lambdaQuery()
                        .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                        .in(MdmPositionEntity::getPositionCode, positionCodeList)
                        .select(MdmPositionEntity::getPositionCode, MdmPositionEntity::getEnableStatus)
                        .list()
                        .stream().collect(Collectors.toMap(MdmPositionEntity::getPositionCode, v -> v));

                List<String> currentPositionCodeList = list.stream().filter(item -> YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getCurrentFlag())).map(MdmPositionUserEntity::getPositionCode).collect(Collectors.toList());
                if (StringUtils.isEmpty(thisPositionCode)) {
                    if (currentPositionCodeList.size() == 1) {
                        if (positionCodeMap.containsKey(currentPositionCodeList.get(0)) && CrmEnableStatusEnum.ENABLE.getCode().equals(positionCodeMap.get(currentPositionCodeList.get(0)).getEnableStatus())) {
                            thisPositionCode = currentPositionCodeList.get(0);
                            resetCurrent = false;
                        }
                    }
                }
                if (StringUtils.isEmpty(thisPositionCode)) {
                    if (currentPositionCodeList.size() > 0) {
                        for (String positionCode :
                                currentPositionCodeList) {
                            if (positionCodeMap.containsKey(positionCode) && CrmEnableStatusEnum.ENABLE.getCode().equals(positionCodeMap.get(positionCode).getEnableStatus())) {
                                thisPositionCode = positionCode;
                                break;
                            }
                        }
                    }
                }
                if (StringUtils.isEmpty(thisPositionCode)) {
                    final List<String> primaryCodeList = list.stream().filter(item -> YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getPrimaryFlag())).map(MdmPositionUserEntity::getPositionCode).collect(Collectors.toList());
                    if (primaryCodeList.size() > 1) {
                        if (positionCodeMap.containsKey(primaryCodeList.get(0)) && CrmEnableStatusEnum.ENABLE.getCode().equals(positionCodeMap.get(primaryCodeList.get(0)).getEnableStatus())) {
                            thisPositionCode = primaryCodeList.get(0);
                        }
                    }
                }

                if (StringUtils.isEmpty(thisPositionCode)) {
                    for (String positionCode :
                            positionCodeList) {
                        if (positionCodeMap.containsKey(positionCode) && CrmEnableStatusEnum.ENABLE.getCode().equals(positionCodeMap.get(positionCode).getEnableStatus())) {
                            thisPositionCode = positionCode;
                            break;
                        }
                    }
                }
                if (StringUtils.isEmpty(thisPositionCode)) {
                    throw new BusinessException("用户[" + userName + "]没有启用的职位");
                }
                if (resetCurrent) {
                    mdmPositionUserService.changeUserCurrentPosition(userName, thisPositionCode);
                }
            }
        }
        return thisPositionCode;
    }

    /**
     * 查询角色编码名称
     *
     * @param allList
     */
    protected void setRoleCodeName(List<MdmPositionPageRespVo> allList) {
        if (CollectionUtil.listEmpty(allList)) {
            return;
        }
        List<List<MdmPositionPageRespVo>> partition = Lists.partition(allList, 500);
        for (List<MdmPositionPageRespVo> list :
                partition) {
            List<MdmPositionRoleEntity> positionRoleEntityList = mdmPositionRoleService.lambdaQuery()
                    .in(MdmPositionRoleEntity::getPositionCode, list.stream().map(MdmPositionPageRespVo::getPositionCode).collect(Collectors.toList()))
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionRoleEntityList)) {
                final Map<String, String> roleCodeNameMap = mdmRoleService.lambdaQuery()
                        .in(MdmRoleEntity::getRoleCode, positionRoleEntityList.stream().map(MdmPositionRoleEntity::getRoleCode).distinct().collect(Collectors.toList()))
                        .select(MdmRoleEntity::getRoleCode, MdmRoleEntity::getRoleName)
                        .list()
                        .stream()
                        .collect(Collectors.toMap(MdmRoleEntity::getRoleCode, MdmRoleEntity::getRoleName));
                final Map<String, List<MdmPositionRoleEntity>> positionRoleListMap = positionRoleEntityList.stream().collect(Collectors.groupingBy(MdmPositionRoleEntity::getPositionCode));
                list.forEach(item -> {
                    if (positionRoleListMap.containsKey(item.getPositionCode())) {
                        List<MdmPositionRoleEntity> relationList = positionRoleListMap.get(item.getPositionCode());
                        if (CollectionUtil.listNotEmptyNotSizeZero(relationList)) {
                            List<String> codeList = new ArrayList<>();
                            List<String> nameList = new ArrayList<>();
                            for (MdmPositionRoleEntity relation :
                                    relationList) {
                                if (roleCodeNameMap.containsKey(relation.getRoleCode())) {
                                    codeList.add(relation.getRoleCode());
                                    nameList.add(roleCodeNameMap.get(relation.getRoleCode()));
                                }
                            }
                            item.setRoleCode(String.join(",", codeList));
                            item.setRoleName(String.join(",", nameList));
                        }
                    }
                });
            }
        }
    }

    protected void sendPositionAdd(List<MdmPositionRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位新增发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_ADD_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位新增发送MQ消息成功：{}", list);
    }

    protected void sendPositionUpdate(List<MdmPositionRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位编辑发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_UPDATE_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位编辑发送MQ消息成功：{}", list);
    }

    protected void sendPositionDelete(List<MdmPositionRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位删除发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_DELETE_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位删除发送MQ消息成功：{}", list);
    }

}
