package com.biz.crm.position.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.position.mapper.MdmPositionUserMapper;
import com.biz.crm.position.model.MdmPositionEntity;
import com.biz.crm.position.model.MdmPositionUserEntity;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.position.service.MdmPositionUserService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.PositionUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位-用户关联表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionUserServiceExpandImpl")
@Transactional(rollbackFor = Exception.class)
public class MdmPositionUserServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPositionUserMapper, MdmPositionUserEntity> implements MdmPositionUserService {

    @Resource
    private MdmPositionUserMapper mdmPositionUserMapper;

    @Resource
    private MdmPositionService mdmPositionService;

    @Override
    public void addUserPosition(String userName, String positionCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(positionCode)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .eq(MdmPositionUserEntity::getPositionCode, positionCode)
                    .list();
            if (CollectionUtil.listEmpty(list)) {
                MdmPositionUserEntity entity = new MdmPositionUserEntity();
                entity.setPositionCode(positionCode);
                entity.setUserName(userName);
                entity.setPrimaryFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                entity.setCurrentFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                this.save(entity);
            }
        }
    }

    @Override
    public void addUserPosition(String userName, String positionCode, String primaryFlag) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(positionCode)) {
            if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(primaryFlag) && !YesNoEnum.yesNoEnum.ZERO.getValue().equals(primaryFlag)) {
                throw new BusinessException("主职位标记参数错误");
            }
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getPositionCode, positionCode)
                    .list();
            if (CollectionUtil.listEmpty(list)) {
                MdmPositionUserEntity entity = new MdmPositionUserEntity();
                entity.setPositionCode(positionCode);
                entity.setUserName(userName);
                entity.setPrimaryFlag(primaryFlag);
                entity.setCurrentFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                this.save(entity);
            }
        }
    }

    @Override
    public void changeUserPrimaryPosition(String userName, String primaryPositionCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(primaryPositionCode)) {
            this.lambdaUpdate()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .set(MdmPositionUserEntity::getPrimaryFlag, YesNoEnum.yesNoEnum.ZERO.getValue())
                    .update();
            this.lambdaUpdate()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .eq(MdmPositionUserEntity::getPositionCode, primaryPositionCode)
                    .set(MdmPositionUserEntity::getPrimaryFlag, YesNoEnum.yesNoEnum.ONE.getValue())
                    .update();
        }
    }

    @Override
    public void changeUserCurrentPosition(String userName, String currentPositionCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(currentPositionCode)) {
            this.lambdaUpdate()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .set(MdmPositionUserEntity::getCurrentFlag, YesNoEnum.yesNoEnum.ZERO.getValue())
                    .update();
            this.lambdaUpdate()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .eq(MdmPositionUserEntity::getPositionCode, currentPositionCode)
                    .set(MdmPositionUserEntity::getCurrentFlag, YesNoEnum.yesNoEnum.ONE.getValue())
                    .update();
        }
    }

    @Override
    public void addUserPosition(String userName, List<String> positionCodeList) {
        if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            positionCodeList.forEach(positionCode -> {
                this.addUserPosition(userName, positionCode);
            });
        }
    }

    @Override
    public void resetUserPosition(String userName, String primaryPositionCode, List<String> positionCodeList) {
        if (StringUtils.isNotEmpty(userName)) {
            this.removeByUserName(userName);
            if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
                Set<String> positionCodeSet = new HashSet<>(8);
                if (StringUtils.isNotEmpty(primaryPositionCode)) {
                    positionCodeSet.add(primaryPositionCode);
                }
                if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
                    positionCodeSet.addAll(positionCodeList);
                }
                this.removeByPositionCodes(new ArrayList<>(positionCodeSet));
                List<MdmPositionUserEntity> list = new ArrayList<>();
                for (String positionCode :
                        positionCodeSet) {
                    MdmPositionUserEntity entity = new MdmPositionUserEntity();
                    entity.setPositionCode(positionCode);
                    entity.setUserName(userName);
                    entity.setPrimaryFlag(StringUtils.isNotEmpty(primaryPositionCode) && primaryPositionCode.equals(positionCode) ? YesNoEnum.yesNoEnum.ONE.getValue() : YesNoEnum.yesNoEnum.ZERO.getValue());
                    entity.setCurrentFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                    list.add(entity);
                }
                this.saveBatch(list);
            }
        }
    }

    @Override
    public void removeByPositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            this.removeByPositionCodes(Collections.singletonList(positionCode));
        }
    }

    @Override
    public void removeByPositionCodes(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            this.lambdaUpdate()
                    .in(MdmPositionUserEntity::getPositionCode, positionCodeList)
                    .remove();
        }
    }

    @Override
    public void removeByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            this.removeByUserNames(Collections.singletonList(userName));
        }
    }

    @Override
    public void removeByUserNames(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            this.lambdaUpdate()
                    .in(MdmPositionUserEntity::getUserName, userNameList)
                    .remove();
        }
    }

    @Override
    public List<String> getPositionCodeListByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .select(MdmPositionUserEntity::getPositionCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().map(MdmPositionUserEntity::getPositionCode).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> getPositionCodeListByUserNameList(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .in(MdmPositionUserEntity::getUserName, userNameList)
                    .select(MdmPositionUserEntity::getPositionCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().map(MdmPositionUserEntity::getPositionCode).distinct().collect(Collectors.toList());
            }
        }

        return new ArrayList<>();
    }

    @Override
    public Map<String, List<String>> getPositionCodeListGroupByUserNameList(List<String> userNameList) {

        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .in(MdmPositionUserEntity::getUserName, userNameList)
                    .select(MdmPositionUserEntity::getPositionCode, MdmPositionUserEntity::getUserName)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.groupingBy(MdmPositionUserEntity::getUserName, Collectors.collectingAndThen(Collectors.toList(), x -> x.stream().map(MdmPositionUserEntity::getPositionCode).collect(Collectors.toList()))));
            }
        }
        return new HashMap<>();
    }

    @Override
    public String getUserNameByPositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            MdmPositionUserEntity one = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getPositionCode, positionCode)
                    .select(MdmPositionUserEntity::getUserName)
                    .one();
            if (one != null) {
                return one.getUserName();
            }
        }
        return null;
    }

    @Override
    public String getUserNameByPrimaryPositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            MdmPositionUserEntity one = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getPositionCode, positionCode)
                    .eq(MdmPositionUserEntity::getPrimaryFlag, YesNoEnum.yesNoEnum.ONE.getValue())
                    .select(MdmPositionUserEntity::getUserName)
                    .one();
            if (one != null) {
                return one.getUserName();
            }
        }
        return null;
    }

    @Override
    public String getPrimaryPositionCodeByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .eq(MdmPositionUserEntity::getUserName, userName)
                    .eq(MdmPositionUserEntity::getPrimaryFlag, YesNoEnum.yesNoEnum.ONE.getValue())
                    .select(MdmPositionUserEntity::getPositionCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.get(0).getPositionCode();
            }
        }
        return null;
    }

    @Override
    public List<String> getUserNameListByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .in(MdmPositionUserEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionUserEntity::getUserName)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().map(MdmPositionUserEntity::getUserName).distinct().collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> getUserNameGroupByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionUserEntity> list = this.lambdaQuery()
                    .in(MdmPositionUserEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionUserEntity::getPositionCode, MdmPositionUserEntity::getUserName)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.toMap(MdmPositionUserEntity::getPositionCode, MdmPositionUserEntity::getUserName));
            }
        }
        return new HashMap<>();
    }

}
