package com.biz.crm.pricesetting.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.mdm.PriceSettingFieldEnum;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceConditionGroupRelFieldReqVo;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceConditionGroupReqVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceConditionGroupRelFieldRespVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceConditionGroupRespVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceOperationCheckRespVo;
import com.biz.crm.pricesetting.mapper.MdmPriceConditionGroupMapper;
import com.biz.crm.pricesetting.model.MdmPriceConditionGroupEntity;
import com.biz.crm.pricesetting.model.MdmPriceConditionGroupRelFieldEntity;
import com.biz.crm.pricesetting.service.MdmPriceConditionGroupRelFieldService;
import com.biz.crm.pricesetting.service.MdmPriceConditionGroupService;
import com.biz.crm.pricesetting.model.MdmPriceConditionTypeRelGroupEntity;
import com.biz.crm.pricesetting.service.MdmPriceConditionTypeRelGroupService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 条件字段分类主表 接口实现
 *
 * @author huanglong
 * @date 2020-10-14 17:21:40
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPriceConditionGroupServiceExpandImpl")
public class MdmPriceConditionGroupServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPriceConditionGroupMapper, MdmPriceConditionGroupEntity> implements MdmPriceConditionGroupService {

    @Resource
    private MdmPriceConditionGroupMapper mdmPriceConditionGroupMapper;

    @Resource
    private MdmPriceConditionGroupRelFieldService mdmPriceConditionGroupRelFieldService;

    @Resource
    private MdmPriceConditionTypeRelGroupService mdmPriceConditionTypeRelGroupService;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Override
    public PageResult<MdmPriceConditionGroupRespVo> findList(MdmPriceConditionGroupReqVo reqVo) {
        Page<MdmPriceConditionGroupRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmPriceConditionGroupRespVo> list = mdmPriceConditionGroupMapper.findList(page, reqVo);
        return PageResult.<MdmPriceConditionGroupRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmPriceConditionGroupRespVo query(String id, String conditionGroupCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(conditionGroupCode)) {
            MdmPriceConditionGroupEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPriceConditionGroupEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(conditionGroupCode), MdmPriceConditionGroupEntity::getConditionGroupCode, conditionGroupCode)
                    .one();
            if (one != null) {
                MdmPriceConditionGroupRespVo respVo = CrmBeanUtil.copy(one, MdmPriceConditionGroupRespVo.class);
                List<MdmPriceConditionGroupRelFieldEntity> list = mdmPriceConditionGroupRelFieldService.lambdaQuery()
                        .eq(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode, one.getConditionGroupCode())
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    respVo.setConditionFieldList(CrmBeanUtil.copyList(list, MdmPriceConditionGroupRelFieldRespVo.class));
                }
                return respVo;
            }
        }
        return null;
    }

    @Override
    public MdmPriceOperationCheckRespVo editCheck(String id) {
        MdmPriceOperationCheckRespVo respVo = new MdmPriceOperationCheckRespVo();
        respVo.setEditCheck(true);
        Assert.hasText(id, "缺失参数");
        MdmPriceConditionGroupEntity entity = this.getById(id);
        Assert.notNull(entity, "无效的id");
        List<MdmPriceConditionTypeRelGroupEntity> relList = mdmPriceConditionTypeRelGroupService.lambdaQuery()
                .eq(MdmPriceConditionTypeRelGroupEntity::getConditionGroupCode, entity.getConditionGroupCode())
                .select(MdmPriceConditionTypeRelGroupEntity::getId)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(relList)) {
            respVo.setEditCheck(false);
            respVo.setMessage("条件字段分类编码为" + entity.getConditionGroupCode() + "的数据关联了条件类型，不可编辑");
        }
        return respVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmPriceConditionGroupReqVo reqVo) {
        reqVo.setId(null);
        Assert.notEmpty(reqVo.getConditionFieldList(), "请选择条件分类描述");
        if (StringUtils.isNotEmpty(reqVo.getConditionGroupCode())) {
            List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                    .eq(MdmPriceConditionGroupEntity::getConditionGroupCode, reqVo.getConditionGroupCode())
                    .select(MdmPriceConditionGroupEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "条件字段分类编码已经存在");
        }
        final Map<String, String> priceConditionMap = DictUtil.dictMap(DictConstant.PRICE_CONDITION);
        reqVo.getConditionFieldList().forEach(item -> {
            Assert.hasText(item.getFieldCode(), "条件字段编码缺失");
            Assert.isTrue(priceConditionMap.containsKey(item.getFieldCode()), "条件字段编码无效");
            item.setFieldName(priceConditionMap.get(item.getFieldCode()));
        });
        Assert.isTrue(reqVo.getConditionFieldList().stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldCode).collect(Collectors.toSet()).contains(PriceSettingFieldEnum.PRODUCT_CODE.getCode()), "条件字段分类必须选择" + PriceSettingFieldEnum.PRODUCT_CODE.getDes());
        List<MdmPriceConditionGroupRelFieldReqVo> sortedList = reqVo.getConditionFieldList().stream().sorted(Comparator.comparing(MdmPriceConditionGroupRelFieldReqVo::getFieldCode)).collect(Collectors.toList());
        String groupCodesKey = sortedList.stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldCode).collect(Collectors.joining("/"));
        String conditionGroupDesc = sortedList.stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldName).collect(Collectors.joining("/"));
        List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                .eq(MdmPriceConditionGroupEntity::getGroupCodesKey, groupCodesKey)
                .select(MdmPriceConditionGroupEntity::getId)
                .list()
                .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(list), "组合" + conditionGroupDesc + "已经存在");
        reqVo.setGroupCodesKey(groupCodesKey);
        reqVo.setConditionGroupDesc(conditionGroupDesc);

        MdmPriceConditionGroupEntity entity = CrmBeanUtil.copy(reqVo, MdmPriceConditionGroupEntity.class);
        if (StringUtils.isEmpty(entity.getConditionGroupCode())) {
            entity.setConditionGroupCode(CodeUtil.createOneCode(CrmCodeRuleConstants.PRICE_CONDITION_GROUP));
        }
        this.save(entity);
        reqVo.getConditionFieldList().forEach(item -> {
            item.setConditionGroupCode(entity.getConditionGroupCode());
        });
        mdmPriceConditionGroupRelFieldService.saveBatch(CrmBeanUtil.copyList(reqVo.getConditionFieldList(), MdmPriceConditionGroupRelFieldEntity.class));
        //添加日志
        MdmPriceConditionGroupRespVo newVo = CrmBeanUtil.copy(entity, MdmPriceConditionGroupRespVo.class);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if(menuObject!=null){
            crmLogSendUtil.sendForAdd(menuObject.toString(),newVo.getId(),newVo.getId(),newVo);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmPriceConditionGroupReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        MdmPriceConditionGroupEntity entity = this.getById(reqVo.getId());
        MdmPriceConditionGroupRespVo oldVo = CrmBeanUtil.copy(entity, MdmPriceConditionGroupRespVo.class);
        Assert.notNull(entity, "未找到条件分类");
        List<MdmPriceConditionTypeRelGroupEntity> relList = mdmPriceConditionTypeRelGroupService.lambdaQuery()
                .eq(MdmPriceConditionTypeRelGroupEntity::getConditionGroupCode, entity.getConditionGroupCode())
                .select(MdmPriceConditionTypeRelGroupEntity::getId)
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(relList), "该条件分类已经关联了条件类型，不能编辑");
        Assert.hasText(reqVo.getConditionGroupCode(), "缺失条件类型编码");
        Assert.isTrue(entity.getConditionGroupCode().equals(reqVo.getConditionGroupCode()), "条件类型编码不能修改");
        Assert.notEmpty(reqVo.getConditionFieldList(), "请选择条件分类描述");
        final Map<String, String> priceConditionMap = DictUtil.dictMap(DictConstant.PRICE_CONDITION);
        reqVo.getConditionFieldList().forEach(item -> {
            Assert.hasText(item.getFieldCode(), "条件字段编码缺失");
            Assert.isTrue(priceConditionMap.containsKey(item.getFieldCode()), "条件字段编码无效");
            item.setFieldName(priceConditionMap.get(item.getFieldCode()));
        });
        Assert.isTrue(reqVo.getConditionFieldList().stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldCode).collect(Collectors.toSet()).contains(PriceSettingFieldEnum.PRODUCT_CODE.getCode()), "条件字段分类必须选择" + PriceSettingFieldEnum.PRODUCT_CODE.getDes());
        List<MdmPriceConditionGroupRelFieldReqVo> sortedList = reqVo.getConditionFieldList().stream().sorted(Comparator.comparing(MdmPriceConditionGroupRelFieldReqVo::getFieldCode)).collect(Collectors.toList());
        String groupCodesKey = sortedList.stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldCode).collect(Collectors.joining("/"));
        String conditionGroupDesc = sortedList.stream().map(MdmPriceConditionGroupRelFieldReqVo::getFieldName).collect(Collectors.joining("/"));
        List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                .eq(MdmPriceConditionGroupEntity::getGroupCodesKey, groupCodesKey)
                .select(MdmPriceConditionGroupEntity::getId)
                .list()
                .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(list), "组合" + conditionGroupDesc + "已经存在");
        reqVo.setGroupCodesKey(groupCodesKey);
        reqVo.setConditionGroupDesc(conditionGroupDesc);
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        mdmPriceConditionGroupRelFieldService.lambdaUpdate()
                .eq(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode, entity.getConditionGroupCode())
                .remove();
        reqVo.getConditionFieldList().forEach(item -> {
            item.setConditionGroupCode(entity.getConditionGroupCode());
        });
        mdmPriceConditionGroupRelFieldService.saveBatch(CrmBeanUtil.copyList(reqVo.getConditionFieldList(), MdmPriceConditionGroupRelFieldEntity.class));
        //记录更新日志
        MdmPriceConditionGroupRespVo newVo = CrmBeanUtil.copy(entity, MdmPriceConditionGroupRespVo.class);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if(menuObject!=null){
            crmLogSendUtil.sendForUpdate(menuObject.toString(),newVo.getId(),newVo.getId(),oldVo,newVo);
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {

        Assert.notEmpty(ids, "缺失参数");
        List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                .in(MdmPriceConditionGroupEntity::getId, ids)
                .select(MdmPriceConditionGroupEntity::getId, MdmPriceConditionGroupEntity::getConditionGroupCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                List<MdmPriceConditionTypeRelGroupEntity> typeRelGroupList = mdmPriceConditionTypeRelGroupService.lambdaQuery()
                        .eq(MdmPriceConditionTypeRelGroupEntity::getConditionGroupCode, item.getConditionGroupCode())
                        .select(MdmPriceConditionTypeRelGroupEntity::getId)
                        .list();
                Assert.isTrue(CollectionUtil.listEmpty(typeRelGroupList), "条件字段编码为{" + item.getConditionGroupCode() + "}的数据关联了条件类型，不能删除！");
            });
            this.removeByIds(ids);
            mdmPriceConditionGroupRelFieldService.lambdaUpdate()
                    .in(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode, list.stream().map(MdmPriceConditionGroupEntity::getConditionGroupCode).collect(Collectors.toList()))
                    .remove();
        }


    }

    @Override
    public List<MdmPriceConditionGroupRespVo> findCollection(MdmPriceConditionGroupReqVo reqVo) {
        List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                .like(StringUtils.isNotEmpty(reqVo.getConditionGroupCode()), MdmPriceConditionGroupEntity::getConditionGroupCode, reqVo.getConditionGroupCode())
                .like(StringUtils.isNotEmpty(reqVo.getConditionGroupDesc()), MdmPriceConditionGroupEntity::getConditionGroupDesc, reqVo.getConditionGroupDesc())
                .select(MdmPriceConditionGroupEntity::getId, MdmPriceConditionGroupEntity::getConditionGroupCode, MdmPriceConditionGroupEntity::getConditionGroupDesc)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return CrmBeanUtil.copyList(list, MdmPriceConditionGroupRespVo.class);
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> getGroupMap() {
        return getGroupMap(new ArrayList<>());
    }

    @Override
    public Map<String, String> getGroupMap(List<String> conditionGroupCodeList) {
        List<MdmPriceConditionGroupEntity> list = this.lambdaQuery()
                .in(CollectionUtil.listNotEmptyNotSizeZero(conditionGroupCodeList), MdmPriceConditionGroupEntity::getConditionGroupCode, conditionGroupCodeList)
                .select(MdmPriceConditionGroupEntity::getConditionGroupCode, MdmPriceConditionGroupEntity::getConditionGroupDesc)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return list.stream().collect(Collectors.toMap(MdmPriceConditionGroupEntity::getConditionGroupCode, MdmPriceConditionGroupEntity::getConditionGroupDesc));
        }
        return new HashMap<>();
    }

}
