package com.biz.crm.product.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.IsShelfEnum;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceProductUnitReqVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceProductUnitRespVo;
import com.biz.crm.nebular.mdm.product.req.*;
import com.biz.crm.nebular.mdm.product.resp.*;
import com.biz.crm.product.service.MdmProductMediaService;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.util.Result;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * 商品主表
 *
 * @author crm
 * @date 2020-08-27 17:34:16
 */
@Slf4j
@RestController
@RequestMapping("/mdmProductController")
@Api(tags = "MDM-商品管理")
@CrmGlobalLog
public class MdmProductController {

    @Resource
    private MdmProductService mdmProductService;
    @Resource
    private MdmProductMediaService mdmProductMediaService;

    @ApiOperation(value = "分页查询列表")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmProductRespVo>> pageList(@RequestBody MdmProductReqVo mdmProductReqVo) {
        PageResult<MdmProductRespVo> result = mdmProductService.findList(mdmProductReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "公用分页弹框")
    @PostMapping("/commonPageList")
    @CrmDictMethod
    public Result<PageResult<MdmProductRespVo>> commonPageList(@RequestBody MdmProductReqVo mdmProductReqVo) {
        mdmProductReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        mdmProductReqVo.setIsShelf(IsShelfEnum.UP.getCode());
        PageResult<MdmProductRespVo> result = mdmProductService.findList(mdmProductReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "productCode", value = "商品编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmDictMethod
    public Result<MdmProductRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "productCode", required = false) String productCode) {
        MdmProductRespVo respVo = mdmProductService.detail(id, productCode);
        return Result.ok(respVo);
    }

    @ApiOperation(value = "通过id集合批量查询商品主表信息", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/queryBatchByIds")
    public Result<List<MdmProductRespVo>> queryBatchByIds(@RequestBody List<String> ids) {
        return Result.ok(mdmProductService.queryBatch(ids, null));
    }

    @ApiOperation(value = "通过商品编码集合批量查询商品主表信息", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "productCodeList", value = "商品编码集合", required = true, paramType = "body")
    })
    @PostMapping("/queryBatchByProductCodeList")
    public Result<List<MdmProductRespVo>> queryBatchByProductCodeList(@RequestBody List<String> productCodeList) {
        return Result.ok(mdmProductService.queryBatch(null, productCodeList));
    }

    @ApiOperation(value = "通过产品层级编码集合批量查询商品主表信息（产品层级及下级的全部下级层级关联的商品）", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "productLevelCodeList", value = "产品层级编码集合", required = true, paramType = "body")
    })
    @PostMapping("/queryBatchByProductLevelCodeList")
    public Result<List<MdmProductRespVo>> queryBatchByProductLevelCodeList(@RequestBody List<String> productLevelCodeList) {
        return Result.ok(mdmProductService.queryBatchByProductLevelCodeList(productLevelCodeList));
    }

    @ApiOperation(value = "根据条件查询商品")
    @PostMapping("/queryProductList")
    public Result<List<MdmProductQueryRespVo>> queryProductList(@RequestBody MdmProductQueryReqVo reqVo) {
        return Result.ok(mdmProductService.queryProductList(reqVo));
    }

    @ApiOperation(value = "根据商品编码查询图片")
    @PostMapping("/queryBatchProductPictureList")
    public Result<List<MdmProductMediaRespVo>> queryBatchProductPictureList(@RequestBody List<String> productCodeList) {
        return Result.ok(mdmProductMediaService.queryBatchProductPictureList(productCodeList));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result save(@RequestBody MdmProductReqVo mdmProductReqVo) {
        mdmProductService.save(mdmProductReqVo);
        return Result.ok("新增成功");
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result update(@RequestBody MdmProductReqVo mdmProductReqVo) {
        mdmProductService.update(mdmProductReqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result delete(@RequestBody List<String> ids) {
        mdmProductService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    public Result enable(@RequestBody List<String> ids) {
        mdmProductService.enableBatch(ids);
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    public Result disable(@RequestBody List<String> ids) {
        mdmProductService.disableBatch(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation(value = "上架", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/upShelf")
    public Result upShelf(@RequestBody List<String> ids) {
        mdmProductService.upShelf(ids);
        return Result.ok("上架成功");
    }

    @ApiOperation(value = "下架", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/downShelf")
    public Result downShelf(@RequestBody List<String> ids) {
        mdmProductService.downShelf(ids);
        return Result.ok("下架成功");
    }

    @ApiOperation(value = "查询商品和产品层级集合的下拉框（只查启用），商品和产品层级并列返回，谨慎使用")
    @PostMapping("/findProductAndProductLevel")
    public Result<List<MdmProductAndProductLevelRespVo>> findProductAndProductLevelByCodeOrName(@RequestBody MdmProductAndProductLevelReqVo reqVo) {
        List<MdmProductAndProductLevelRespVo> result = mdmProductService.findProductAndProductLevelByCodeOrName(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "商品下拉框选项")
    @PostMapping("/productSelectList")
    @CrmDictMethod
    public Result<List<MdmProductSelectRespVo>> productSelectList(@RequestBody MdmProductSelectReqVo reqVo) {
        return Result.ok(mdmProductService.productSelectList(reqVo));
    }

    @ApiOperation("查询商品单位")
    @PostMapping("/findProductUnit")
    public Result<List<MdmPriceProductUnitRespVo>> findProductUnit(@RequestBody MdmPriceProductUnitReqVo mdmPriceProductUnitReqVo) {
        List<MdmPriceProductUnitRespVo> mdmPriceProductUnitRespVos = mdmProductService.findProductUnit(mdmPriceProductUnitReqVo.getProductCode());
        return Result.ok(mdmPriceProductUnitRespVos);
    }

    @ApiModelProperty("条件查询列表")
    @PostMapping("/listCondition")
    public Result<List<MdmProductRespVo>> listCondition(@RequestBody MdmProductReqVo mdmProductReqVo) {
        List<MdmProductRespVo> list = mdmProductService.listCondition(mdmProductReqVo);
        return Result.ok(list);
    }

    @ApiOperation("查询商品详情及价格(dms专用)")
    @PostMapping("/queryByProductCodeAndCus")
    public Result<Map<String, MdmProductAdviseAndPriceRespVo>> queryByProductCodeAndCus(@RequestBody MdmProductAdviseAndPriceReqVo mdmProductAdviseAndPriceReqVo) {
        return Result.ok(mdmProductService.queryByProductCodeAndCus(mdmProductAdviseAndPriceReqVo));
    }

    @ApiOperation("查询上架商品及价格(dms专用)")
    @PostMapping("/queryShelfProductList")
    public Result<List<CusAvailablelistVo>> queryShelfProductList(@RequestBody MdmProductShelfPriceReqVo mdmProductShelfPriceReqVo) {
        return Result.ok(mdmProductService.queryShelfProductList(mdmProductShelfPriceReqVo));
    }

    @ApiOperation(value = "sfa查询商品专用接口")
    @PostMapping("/findProductListSourceSfa")
    public Result<PageResult<MdmProductBySfaRespVo>> findProductListSourceSfa(@RequestBody MdmProductBySfaReqVo reqVo){
        return Result.ok(mdmProductService.findProductListSourceSfa(reqVo));
    }

    @ApiOperation(value = "sfa tpm活动查询商品基础信息")
    @PostMapping("/findProductBaseSourceSfaTpmAct")
    public Result<List<MdmProductBaseDataRespVo>> findProductBaseSourceSfaTpmAct(@RequestBody List<String> productCodeList){
        return Result.ok(mdmProductService.findProductBaseSourceSfaTpmAct(productCodeList));
    }

    @ApiOperation(value = "商品的分页查询接口和条件接口")
    @PostMapping("/pageListWithCondition")
    @CrmDictMethod
    public Result<PageResult<MdmProductRespVo>> pageListWithCondition(@RequestBody MdmProductReqVo mdmProductReqVo) {
        PageResult<MdmProductRespVo> result = mdmProductService.pageListWithCondition(mdmProductReqVo);
        return Result.ok(result);
    }
}
