package com.biz.crm.product.service.impl;

import com.biz.crm.eunm.FileTypeEnum;
import com.biz.crm.eunm.IsShelfEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.product.req.MdmProductReqVo;
import com.biz.crm.product.model.MdmProductEntity;
import com.biz.crm.product.model.MdmProductIntroductionEntity;
import com.biz.crm.product.model.MdmProductMaterialEntity;
import com.biz.crm.product.model.MdmProductMediaEntity;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.product.service.impl
 * @ClassName MdmProductBuilder
 * @Author HuangLong
 * @Date 2020/9/2 10:03
 * @Description TODO
 */
@Slf4j
public class MdmProductBuilder {

    private MdmProductServiceHelper helper;

    private MdmProductReqVo reqVo;

    private MdmProductEntity productEntity;

    private MdmProductIntroductionEntity introductionEntity;

    private List<MdmProductMediaEntity> mediaEntities = Lists.newArrayList();

    private List<MdmProductMaterialEntity> materialEntities = Lists.newArrayList();


    public MdmProductBuilder(MdmProductServiceHelper helper, MdmProductReqVo reqVo) {
        this.helper = helper;
        this.reqVo = reqVo;
    }

    public static MdmProductBuilder builder(MdmProductServiceHelper helper, MdmProductReqVo reqVo) {
        return new MdmProductBuilder(helper, reqVo);
    }

    /**
     * 组装必要参数
     *
     * @return
     */
    public MdmProductBuilder init() {
        return this;
    }

    /**
     * 参数校验
     *
     * @return
     */
    public MdmProductBuilder check() {
        //数据不为空校验
        AssertUtils.isNotEmpty(reqVo.getProductCode(), "请输入商品编码");
//        AssertUtils.isNotEmpty(reqVo.getProductLevelCode(),"请选择产品层级");
        AssertUtils.isNotEmpty(reqVo.getProductName(), "请输入商品名称");
//        AssertUtils.isNotEmpty(reqVo.getMaterialList(),MdmProductException.PRODUCT_MATERIAL_NULL);
//        if (StringUtils.equals(ProductTypeEnum.STANDARD.getCode(), reqVo.getProductType())) {
//            if (reqVo.getMaterialList().size() > 1) {
//                throw new BusinessException(ProductTypeEnum.STANDARD.getDes() + "只能选择一个物料");
//            }
//        }
//        reqVo.getMaterialList().forEach(o -> {
//            AssertUtils.isNotEmpty(o.getMaterialCode(), "物料编码不能为空");
//            if (StringUtils.equals(ProductTypeEnum.STANDARD.getCode(), reqVo.getProductType())) {
//                //标准商品
//                o.setRatio(null);
//                AssertUtils.isNotNull(o.getCount(), ProductTypeEnum.STANDARD.getDes() + "物料数量不能为空");
//            } else if (StringUtils.equals(ProductTypeEnum.RANGE.getCode(), reqVo.getProductType())) {
//                //范围商品
//                o.setCount(null);
////                AssertUtils.isNotNull(o.getRatio(), ProductTypeEnum.RANGE.getDes() + "物料比例不能为空");
//            } else if (StringUtils.equals(ProductTypeEnum.GIFT_PACKAGE.getCode(), reqVo.getProductType())) {
//                //礼包商品
//                o.setRatio(null);
//                AssertUtils.isNotNull(o.getCount(), ProductTypeEnum.GIFT_PACKAGE.getDes() + "物料数量不能为空");
//            } else {
//
//            }
//        });
//        //物料数据重复校验
//        Set<String> set = reqVo.getMaterialList().stream().map(MdmProductMaterialReqVo::getMaterialCode).collect(Collectors.toSet());
//        AssertUtils.isTrue(reqVo.getMaterialList().size()==set.size(),MdmProductException.PRODUCT_MATERIAL_REPEAT);
//        //校验商品编码是否已经存在
//        List<MdmProductEntity> products = helper.findProductsByParam(reqVo);
//        if(CollectionUtils.isNotEmpty(products)){
//            if(StringUtils.isEmpty(reqVo.getId())||!StringUtils.equals(products.get(0)
//                    .getId(),reqVo.getId())){
//                new BusinessException(MdmProductException.PRODUCT_CODE_EXIST);
//            }
//        }
//        //如果商品类型是标准商品,物料只能唯一
//        if(StringUtils.equals(ProductTypeEnum.STANDARD.getCode(),reqVo.getProductType())){
//            if(reqVo.getMaterialList().size()>1){
//                new BusinessException(MdmProductException.PRODUCT_MATERIAL_ERROR);
//            }
//        }
//
//        //校验物料数据的有效性
//        Map<String, MdmMaterialEntity> materialMap = helper.findMaterialMapByCodes(set);
//        AssertUtils.isNotEmpty(materialMap,"您选择的物料数据已被删除");
//        set.forEach(o->{
//            MdmMaterialEntity entity = materialMap.get(o);
//            AssertUtils.isNotNull(entity,"您选择的物料:"+o+"已被删除");
//            AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(),entity.getEnableStatus()),"您选择的物料:"+o+"已被禁用");
//        });
        return this;
    }

    /**
     * 转换需要保存的商品数据
     *
     * @return
     */
    public MdmProductBuilder convert() {
        //转换商品主表
        MdmProductEntity mdmProductEntity = new MdmProductEntity();
        CrmBeanUtil.copyProperties(reqVo, mdmProductEntity);
        if (StringUtils.isEmpty(mdmProductEntity.getIsShelf())) {
            mdmProductEntity.setIsShelf(IsShelfEnum.DOWN.getCode());
        }
        //如果商品编码为空时,需要系统自动生成编码,项目如果需要,在这里调一下生成编码的方法
        this.productEntity = mdmProductEntity;
        //转换商品物料表
        if (!CollectionUtils.isEmpty(reqVo.getMaterialList())) {
            this.materialEntities = reqVo.getMaterialList().stream().map(o -> {
                MdmProductMaterialEntity materialEntity = new MdmProductMaterialEntity();
                CrmBeanUtil.copyProperties(o, materialEntity);
                materialEntity.setId(UUID.randomUUID().toString());
                materialEntity.setProductCode(mdmProductEntity.getProductCode());
                return materialEntity;
            }).collect(Collectors.toList());
        }
        //转换商品图片数据
        List<MdmProductMediaEntity> pictureList = null;
        if (!CollectionUtils.isEmpty(reqVo.getPictureList())) {
            pictureList = reqVo.getPictureList().stream().filter(Objects::nonNull).map(o -> {
                MdmProductMediaEntity mediaEntity = new MdmProductMediaEntity();
                CrmBeanUtil.copyProperties(o, mediaEntity);
                mediaEntity.setType(FileTypeEnum.PICTURE.getCode());
                mediaEntity.setProductCode(mdmProductEntity.getProductCode());
                return mediaEntity;
            }).collect(Collectors.toList());
            productEntity.setPrimaryPictureUrl(reqVo.getPictureList().get(0).getUrlAddress());
        }
        if (CollectionUtil.listNotEmpty(pictureList)) {
            this.mediaEntities.addAll(pictureList);
        }
        //转换商品视频数据
        List<MdmProductMediaEntity> videoList = null;
        if (!CollectionUtils.isEmpty(reqVo.getVideoList())) {
            videoList = reqVo.getVideoList().stream().filter(Objects::nonNull).map(o -> {
                MdmProductMediaEntity mediaEntity = new MdmProductMediaEntity();
                CrmBeanUtil.copyProperties(o, mediaEntity);
                mediaEntity.setProductCode(mdmProductEntity.getProductCode());
                mediaEntity.setType(FileTypeEnum.VIDEO.getCode());
                return mediaEntity;
            }).collect(Collectors.toList());
        }
        if (CollectionUtil.listNotEmpty(videoList)) {
            this.mediaEntities.addAll(videoList);
        }

        //转换商品富文本介绍数据
        if (Objects.nonNull(reqVo.getIntroductionVo())) {
            MdmProductIntroductionEntity introductionEntity = new MdmProductIntroductionEntity();
            CrmBeanUtil.copyProperties(reqVo.getIntroductionVo(), introductionEntity);
            introductionEntity.setProductCode(mdmProductEntity.getProductCode());
            this.introductionEntity = introductionEntity;
        }
        return this;
    }

    /**
     * 保存商品数据
     *
     * @return
     */
    public void save() {
        validDetails();
        helper.saveProductEntity(productEntity);
        helper.saveProductIntroduction(introductionEntity, Boolean.TRUE, productEntity.getProductCode());
        helper.saveProductMaterials(materialEntities, Boolean.TRUE, productEntity.getProductCode());
        helper.saveProductMedias(mediaEntities, Boolean.TRUE, productEntity.getProductCode());
    }

    /**
     * 校验图片维护和详情维护
     */
    protected void validDetails() {
        if (!CollectionUtils.isEmpty(mediaEntities)) {
            productEntity.setMaintenancePicture(YesNoEnum.yesNoEnum.ONE.getValue());
        } else {
            productEntity.setMaintenancePicture(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        if (introductionEntity != null && !StringUtils.isEmpty(introductionEntity.getIntroductionText())) {
            productEntity.setMaintenanceIntroduction(YesNoEnum.yesNoEnum.ONE.getValue());
        } else {
            productEntity.setMaintenanceIntroduction(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
    }

    /**
     * 更新商品数据
     *
     * @return
     */
    public void update() {
        validDetails();
        helper.saveProductEntity(productEntity);
        helper.saveProductIntroduction(introductionEntity, Boolean.FALSE, productEntity.getProductCode());
        helper.saveProductMaterials(materialEntities, Boolean.FALSE, productEntity.getProductCode());
        helper.saveProductMedias(mediaEntities, Boolean.FALSE, productEntity.getProductCode());
    }
}
