package com.biz.crm.product.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalDictConstants;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.ProductTypeEnum;
import com.biz.crm.exception.mdm.MdmProductException;
import com.biz.crm.material.mapper.MdmMaterialMapper;
import com.biz.crm.material.model.MdmMaterialEntity;
import com.biz.crm.nebular.mdm.product.req.MdmProductMaterialReqVo;
import com.biz.crm.nebular.mdm.product.req.MdmProductMediaReqVo;
import com.biz.crm.nebular.mdm.product.req.MdmProductReqVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductIntroductionRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductMaterialRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductMediaRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductRespVo;
import com.biz.crm.product.mapper.MdmProductIntroductionMapper;
import com.biz.crm.product.mapper.MdmProductMapper;
import com.biz.crm.product.mapper.MdmProductMaterialMapper;
import com.biz.crm.product.mapper.MdmProductMediaMapper;
import com.biz.crm.product.model.MdmProductEntity;
import com.biz.crm.product.model.MdmProductIntroductionEntity;
import com.biz.crm.product.model.MdmProductMaterialEntity;
import com.biz.crm.product.model.MdmProductMediaEntity;
import com.biz.crm.product.service.MdmProductMaterialService;
import com.biz.crm.product.service.MdmProductMediaService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.product.service.impl
 * @ClassName MdmProductServiceHelper
 * @Author HuangLong
 * @Date 2020/8/28 17:35
 * @Description 商品管理工具类
 */
@Component
@Slf4j
public class MdmProductServiceHelper {

    @Resource
    private MdmProductMapper mdmProductMapper;

    @Resource
    private MdmProductMediaMapper mdmProductMediaMapper;

    @Resource
    private MdmProductIntroductionMapper mdmProductIntroductionMapper;

    @Resource
    private MdmProductMaterialMapper mdmProductMaterialMapper;

    @Autowired
    private MdmProductMaterialService mdmProductMaterialService;

    @Autowired
    private MdmProductMediaService mdmProductMediaService;
    @Resource
    private MdmMaterialMapper mdmMaterialMapper;



    public Map<String, Map<String, String>> getProductDict() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(GlobalDictConstants.PRODUCT_SALE_UNIT);
        dictCodes.add(GlobalDictConstants.PRODUCT_BASE_UNIT);
        dictCodes.add(GlobalDictConstants.MATERIAL_BASE_UNIT);
        dictCodes.add(GlobalDictConstants.MATERIAL_SALE_UNIT);
        //查询字典数据
        Map<String, Map<String, String>> map = DictUtil.getDictValueMapsByCodes(dictCodes);
        return map;
    }

    /**
     * 转换商品列表数据
     *
     * @param list
     */
    public void convertListDate(List<MdmProductRespVo> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getProductDict();
            list.forEach(o -> {
                //销售单位(如果商品类型是标准商品,取物料的销售单位,否则取字典值)
                //基本单位(如果商品类型是标准商品,取物料的基本单位,否则取字典值)
                if (!StringUtils.equals(ProductTypeEnum.STANDARD.getCode(), o.getProductType())) {
                    if (StringUtils.isNotEmpty(o.getSaleUnit())) {
                        o.setSaleUnitName(Optional.ofNullable(map.get(GlobalDictConstants.PRODUCT_SALE_UNIT)).orElse(Maps.newHashMap()).get(o.getSaleUnit()));
                    }
                    if (StringUtils.isNotEmpty(o.getBaseUnit())) {
                        o.setBaseUnitName(Optional.ofNullable(map.get(GlobalDictConstants.PRODUCT_BASE_UNIT)).orElse(Maps.newHashMap()).get(o.getBaseUnit()));
                    }
                } else {
                    if (StringUtils.isNotEmpty(o.getSaleUnit())) {
                        o.setSaleUnitName(Optional.ofNullable(map.get(GlobalDictConstants.MATERIAL_SALE_UNIT)).orElse(Maps.newHashMap()).get(o.getSaleUnit()));
                    }
                    if (StringUtils.isNotEmpty(o.getBaseUnit())) {
                        o.setBaseUnitName(Optional.ofNullable(map.get(GlobalDictConstants.MATERIAL_BASE_UNIT)).orElse(Maps.newHashMap()).get(o.getBaseUnit()));
                    }
                }
            });
        }
    }

    /**
     * 保存商品主表数据
     *
     * @param mdmProductEntity
     */
    public void saveProductEntity(MdmProductEntity mdmProductEntity) {
        if (Objects.isNull(mdmProductEntity)) {
            throw new BusinessException(MdmProductException.PRODUCT_NULL);
        }
        if (StringUtils.isNotEmpty(mdmProductEntity.getId())) {
            mdmProductMapper.updateById(mdmProductEntity);
        } else {
            mdmProductMapper.insert(mdmProductEntity);
        }
    }

    /**
     * 保存商品物料数据
     *
     * @param materialEntities
     */
    public void saveProductMaterials(List<MdmProductMaterialEntity> materialEntities, Boolean isSave, String productCode) {
        if (isSave) {
            //新增
            materialEntities.forEach(o -> {
                mdmProductMaterialMapper.insert(o);
            });
        } else {
            //修改
            List<String> ids = materialEntities.stream().filter(O -> StringUtils.isNotEmpty(O.getId())).map(MdmProductMaterialEntity::getId).collect(Collectors.toList());
            //如果ids不为空,先将除ids之外的物料删除
            mdmProductMaterialService.lambdaUpdate()
                    .eq(MdmProductMaterialEntity::getProductCode, productCode)
                    .notIn(CollectionUtils.isNotEmpty(ids), MdmProductMaterialEntity::getId, ids)
                    .remove();
            mdmProductMaterialService.saveOrUpdateBatch(materialEntities);
        }
    }

    /**
     * 保存商品图片视频数据
     *
     * @param mediaEntities
     */
    public void saveProductMedias(List<MdmProductMediaEntity> mediaEntities, Boolean isSave, String productCode) {
        if (isSave) {
            //新增
            if (CollectionUtils.isNotEmpty(mediaEntities)) {
                mediaEntities.forEach(o -> {
                    mdmProductMediaMapper.insert(o);
                });
            }
        } else {
            //修改
            List<String> ids = mediaEntities.stream().filter(O -> StringUtils.isNotEmpty(O.getId())).map(MdmProductMediaEntity::getId).collect(Collectors.toList());
            //如果ids不为空,先将除ids之外的图片,视频删除
            mdmProductMediaService.lambdaUpdate()
                    .eq(MdmProductMediaEntity::getProductCode, productCode)
                    .notIn(CollectionUtils.isNotEmpty(ids), MdmProductMediaEntity::getId, ids)
                    .remove();
            if (CollectionUtil.listNotEmptyNotSizeZero(mediaEntities)) {
                mdmProductMediaService.saveOrUpdateBatch(mediaEntities);
            }
        }
    }

    /**
     * 保存商品介绍数据
     *
     * @param mdmProductIntroductionEntity
     */
    public void saveProductIntroduction(MdmProductIntroductionEntity mdmProductIntroductionEntity, Boolean isSave, String productCode) {
        if (!isSave){
            mdmProductIntroductionMapper.delete(new LambdaQueryWrapper<MdmProductIntroductionEntity>().eq(MdmProductIntroductionEntity::getProductCode, productCode));
        }
        if (mdmProductIntroductionEntity != null) {
            mdmProductIntroductionMapper.insert(mdmProductIntroductionEntity);
        }
    }

    /**
     * 通过参数查询商品主表数据
     *
     * @param reqVo
     * @return
     */
    public List<MdmProductEntity> findProductsByParam(MdmProductReqVo reqVo) {
        QueryWrapper<MdmProductEntity> wrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(reqVo.getProductCode())) {
            wrapper.eq("product_code", reqVo.getProductCode());
        }
        return mdmProductMapper.selectList(wrapper);
    }

    /**
     * 通过商品编码查询商品物料
     *
     * @param productCode
     * @return
     */
    @CrmDictMethod
    public List<MdmProductMaterialRespVo> findMaterialByProductCode(String productCode) {
        MdmProductMaterialReqVo materialReqVo = new MdmProductMaterialReqVo();
        materialReqVo.setProductCode(productCode);
        return mdmProductMaterialService.findList(materialReqVo).getData();
    }

    /**
     * 通过商品编码查询商品图片,视频
     *
     * @param productCode
     * @return
     */
    public List<MdmProductMediaRespVo> findMediaByProductCode(String productCode) {
        MdmProductMediaReqVo mediaReqVo = new MdmProductMediaReqVo();
        mediaReqVo.setProductCode(productCode);
        PageResult<MdmProductMediaRespVo> page = mdmProductMediaService.findList(mediaReqVo);
        return page.getData();
    }

    public MdmProductMediaRespVo getMediaByProductCode(String productCode) {
        MdmProductMediaReqVo mediaReqVo = new MdmProductMediaReqVo();
        mediaReqVo.setProductCode(productCode);
        return mdmProductMediaService.query(mediaReqVo);
    }

    /**
     * 通过商品编码查询商品介绍数据
     *
     * @param productCode
     * @return
     */
    public MdmProductIntroductionRespVo findIntroductionByProductCode(String productCode) {
        MdmProductIntroductionEntity introductionEntity = mdmProductIntroductionMapper.selectOne(new QueryWrapper<MdmProductIntroductionEntity>().eq("product_code", productCode));
        if (Objects.isNull(introductionEntity)) {
            return null;
        }
        MdmProductIntroductionRespVo respVo = new MdmProductIntroductionRespVo();
        CrmBeanUtil.copyProperties(introductionEntity, respVo);
        return respVo;
    }

    /**
     * 通过查询条件获取wrapper
     *
     * @return
     */
    public LambdaQueryWrapper<MdmProductEntity> getProductQueryWrapper(MdmProductReqVo reqVo) {
        LambdaQueryWrapper<MdmProductEntity> wrapper = new LambdaQueryWrapper<>();
        //商品编码
        wrapper.like(StringUtils.isNotEmpty(reqVo.getProductCode()), MdmProductEntity::getProductCode, reqVo.getProductCode());
        //商品名称
        wrapper.like(StringUtils.isNotEmpty(reqVo.getProductCode()), MdmProductEntity::getProductName, reqVo.getProductName());
        //产品层级
        wrapper.like(StringUtils.isNotEmpty(reqVo.getProductLevelCode()), MdmProductEntity::getProductLevelCode, reqVo.getProductLevelCode());
        return wrapper;
    }

    /**
     * 修改商品数据(启用,禁用,上架,下架)
     *
     * @param reqVo
     * @param entity
     */
    public void updateBySelectAll(MdmProductReqVo reqVo, MdmProductEntity entity) {
        LambdaQueryWrapper<MdmProductEntity> productQueryWrapper = this.getProductQueryWrapper(reqVo);
        //TODO  这里必须加上数据权限的限制!!!!
        mdmProductMapper.update(entity, productQueryWrapper);
    }

    public Map<String, MdmMaterialEntity> findMaterialMapByCodes(Set<String> materialCodes) {
        List<MdmMaterialEntity> mdmMaterialEntities = mdmMaterialMapper.selectList(new LambdaQueryWrapper<MdmMaterialEntity>().in(MdmMaterialEntity::getMaterialCode, materialCodes));
        Map<String, MdmMaterialEntity> map = Maps.newLinkedHashMap();
        mdmMaterialEntities.stream().filter(Objects::nonNull).forEach(o -> {
            map.put(o.getMaterialCode(), o);
        });
        return map;
    }
}
