package com.biz.crm.role.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.role.req.*;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleSelectRespVo;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 角色管理
 *
 * @author liaohua
 * @date 2020/10/23 2:35 下午
 */
@Slf4j
@RestController
@RequestMapping("/mdmRoleController")
@Api(tags = "MDM-角色管理")
@CrmGlobalLog
public class MdmRoleController {

    @Resource
    private MdmRoleService mdmRoleService;
    @Autowired
    private MdmPositionRoleService mdmPositionRoleService;

    @ApiOperation(value = "列表分页查询")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmRoleRespVo>> pageList(@RequestBody MdmRoleReqVo mdmRoleReqVo) {
        PageResult<MdmRoleRespVo> data = mdmRoleService.findList(mdmRoleReqVo);
        return Result.ok(data);
    }

    @ApiOperation("角色列表（不限制数量，一般用于穿梭框）")
    @PostMapping("/roleList")
    public Result<List<MdmRoleRespVo>> roleList(@RequestBody MdmRoleReqVo mdmRoleReqVo) {
        List<MdmRoleRespVo> data = mdmRoleService.roleList(mdmRoleReqVo);
        return Result.ok(data);
    }

    @ApiOperation("角色列表（限制数量，一般用于下拉框）")
    @PostMapping("/roleSelectList")
    public Result<List<MdmRoleSelectRespVo>> roleSelectList(@RequestBody MdmRoleSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        return Result.ok(mdmRoleService.roleSelectList(reqVo));
    }

    @ApiOperation(value = "详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "roleCode", value = "角色编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmDictMethod
    public Result<MdmRoleRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "roleCode", required = false) String roleCode) {
        return Result.ok(mdmRoleService.detail(id, roleCode));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result<Object> save(@RequestBody MdmRoleReqVo mdmRoleReqVo) {
        mdmRoleService.save(mdmRoleReqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result<Object> update(@RequestBody MdmRoleReqVo mdmRoleReqVo) {
        mdmRoleService.update(mdmRoleReqVo);
        return Result.ok();
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result<Object> delete(@RequestBody List<String> ids) {
        mdmRoleService.deleteBatch(new MdmRoleReqVo().setIds(ids));
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    public Result<Object> enable(@RequestBody List<String> ids) {
        mdmRoleService.enableBatch(new MdmRoleReqVo().setIds(ids));
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    public Result<Object> disable(@RequestBody List<String> ids) {
        mdmRoleService.disableBatch(new MdmRoleReqVo().setIds(ids));
        return Result.ok("禁用成功");
    }

    @ApiOperation("角色关联的菜单树")
    @PostMapping("/roleFunctionTree")
    public Result<MdmRoleFunctionRespVo> roleFunctionTree(@RequestBody MdmRoleFunctionReqVo mdmRoleFunctionReqVo) {
        MdmRoleFunctionRespVo data = mdmRoleService.roleFunctionTree(mdmRoleFunctionReqVo);
        return Result.ok(data);
    }

    @ApiOperation("角色关联的已显示按钮")
    @PostMapping("/roleButtonList")
    public Result<List<MdmRoleSubButtonVo>> roleButtonList(@RequestBody MdmRoleSubButtonReqVo mdmRoleSubButtonReqVo) {
        List<MdmRoleSubButtonVo> list = mdmRoleService.roleButtonList(mdmRoleSubButtonReqVo);
        return Result.ok(list);
    }

    @ApiOperation("保存菜单与按钮")
    @PostMapping("/saveFunctionButton")
    public Result<Object> saveFunctionButton(@RequestBody MdmRoleFunctionButtonStoreVo mdmRoleFunctionButtonStoreVo) {
        mdmRoleService.saveFunctionButton(mdmRoleFunctionButtonStoreVo);
        return Result.ok();
    }

    @ApiOperation("切换当前及下级菜单关联按钮状态")
    @PostMapping("/switchFunctionButton")
    public Result<Object> switchFunctionButton(@RequestBody MdmSwitchFunctionButtonReqVo mdmSwitchFunctionButtonReqVo) {
        mdmRoleService.switchFunctionButton(mdmSwitchFunctionButtonReqVo);
        return Result.ok();
    }

    @ApiOperation("通过职位编码查询关联的角色编码集合")
    @PostMapping("/getRoleByPositionCode")
    public Result<List<String>> getRoleByPositionCode(@RequestParam(value = "positionCode", required = true) String positionCode){
        if(StringUtils.isEmpty(positionCode)){
            throw new BusinessException("请输入职位编码");
        }
        Result<List<String>> result = new Result<>();
        List<String> positionCodes = new ArrayList<>();
        positionCodes.add(positionCode);
        List<String> roleCodes =  mdmPositionRoleService.findRoleCodeByPosition(positionCodes);
        result.setResult(roleCodes);
        return result;
    }

    @ApiOperation("通过用户账号查询关联的角色编码集合")
    @PostMapping("/getRoleByUserName")
    public Result<List<String>> getRoleByUserName(@RequestParam(value = "userName", required = true) String userName) {
        return Result.ok(mdmRoleService.getRoleByUserName(userName));
    }

    @ApiOperation("根据用户获取角色列表")
    @PostMapping("/obtainUserRoleList")
    public Result<List<MdmRoleRespVo>> obtainUserRoleList(@RequestBody MdmUserRoleReqVo mdmUserRoleReqVo) {
        return Result.ok(mdmRoleService.obtainUserRoleList(mdmUserRoleReqVo));
    }

}
